package org.nuiton.helper.plugin;

import org.nuiton.plugin.*;
import java.util.Properties;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.Settings;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcher;

/**
 * Obtain a server authentication and share it in the maven 
 * project properties.
 * <p/>
 * To select data to export from the server with the given {@code serverId},
 * fill the properties :
 * <pre>
 *   usernameOut
 *   passwordOut
 *   privateKeyOut
 *   passphraseOut
 * </pre>
 * 
 * @author chemit
 * @since 1.1.0
 *
 * @goal share-server-secret
 * @phase initialize
 * @requiresProject true
 */
public class ShareServerSecretPlugin extends AbstractPlugin {

    /**
     * Dependance du projet.
     *
     * @parameter default-value="${project}"
     * @required
     * @readonly
     * @since 1.1.0
     */
    protected MavenProject project;
    /**
     * Dependance du settings.
     *
     * @parameter default-value="${settings}"
     * @required
     * @readonly
     * @since 1.1.0
     */
    protected Settings settings;
    /**
     * Server id to use for authentication (must be defined in your setting
     * and use the maven >= 2.1.0 password encryption mecanism).
     * <p/>
     *
     * @parameter expression="${helper.serverId}"
     * @required
     * @since 1.1.0
     */
    protected String serverId;
    /**
     * The name of the property where to export the username of the server.
     * <p/>
     * <b>Note:</b> If not set - then no export of the username of the server.
     *
     * @parameter
     * @since 1.1.0
     */
    private String usernameOut;
    /**
     * The name of the property where to export the password of the server.
     * <p/>
     * <b>Note:</b> If not set - then no export of the password of the server.
     * <p/>
     * <b>Note:</b> If the password is crypted (since maven 2.1.0) then decrypt it.
     *
     * @parameter
     * @since 1.1.0
     */
    private String passwordOut;
    /**
     * The name of the property where to export the passphrase of the server.
     * <p/>
     * <b>Note:</b> If not set - then no export of the passphrase of the server.
     * <p/>
     * <b>Note:</b> If the passphrase is crypted (since maven 2.1.0) then decrypt it.
     *
     * @parameter
     * @since 1.1.0
     */
    private String passphraseOut;
    /**
     * The name of the property where to export the private key of the server.
     * <p/>
     * <b>Note:</b> If not set - then no export of the private key of the server.
     * 
     * @parameter
     * @since 1.1.0
     */
    private String privateKeyOut;
    /**
     * Un flag pour activer le mode verbeux.
     *
     * @parameter expression="${helper.verbose}"  default-value="${maven.verbose}"
     * @since 1.1.0
     */
    protected boolean verbose;
    /**
     * password decypher
     *
     * @component roleHint="maven-helper-plugin"
     * @since 1.1.0
     */
    protected SecDispatcher sec;
    private boolean exportUsername;
    private boolean exportPassword;
    private boolean exportPassphrase;
    private boolean exportPrivateKey;

    @Override
    protected boolean init() throws Exception {
        if (usernameOut != null && !usernameOut.trim().isEmpty()) {
            exportUsername = true;
        }
        if (passwordOut != null && !passwordOut.trim().isEmpty()) {
            exportPassword = true;
        }
        if (passphraseOut != null && !passphraseOut.trim().isEmpty()) {
            exportPassphrase = true;
        }
        if (privateKeyOut != null && !privateKeyOut.trim().isEmpty()) {
            exportPrivateKey = true;
        }
        if (!(exportUsername || exportPassword || exportPassphrase || exportPrivateKey)) {
            getLog().error("Nothing to export, you should specify what to export via 'usernameOut', 'passwordOut', 'passphraseOut', 'privateKeyOut' parameters.");
            return false;
        }
        return true;
    }

    @Override
    protected void doAction() throws Exception {
        Server server = null;

        if (serverId != null && !serverId.trim().isEmpty()) {

            server = settings.getServer(serverId);

            if (server == null) {
                throw new MojoExecutionException("Could not find server with id '" + serverId + "', check your settings.xml file.");
            }
        }

        Properties properties = project.getModel().getProperties();

        if (exportUsername) {

            String username = server.getUsername();
            username = sec.decrypt(username);
            getLog().info("export server [" + serverId + "] username in ${" + usernameOut + "}");
            properties.setProperty(usernameOut, username);
        }

        if (exportPassword) {
            String password = server.getPassword();
            password = sec.decrypt(password);

            getLog().info("export server [" + serverId + "] password in ${" + passwordOut + "}");
            properties.setProperty(passwordOut, password);
        }

        if (exportPassphrase) {
            String passphrase = server.getPassphrase();
            passphrase = sec.decrypt(passphrase);

            getLog().info("export server [" + serverId + "] passphrase in ${" + passphraseOut + "}");
            properties.setProperty(passphraseOut, passphrase);
        }
        if (exportPrivateKey) {
            String privateKey = server.getPrivateKey();

            getLog().info("export server [" + serverId + "] privateKey in ${" + privateKeyOut + "}");
            properties.setProperty(privateKeyOut, privateKey);
        }
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }
}
