package org.nuiton.i18n;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.logging.Logger;
import java.util.zip.ZipFile;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.LocaleConverter;

/**
 *
 * @author chemit
 */
public class I18nUtil {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log = LogFactory.getLog(I18nUtil.class);
    public static final String ISO_8859_1_ENCONDING = "ISO-8859-1";
    public static final String UTF_8_ENCONDING = "UTF-8";
    public static final String DEFAULT_ENCODING = ISO_8859_1_ENCONDING;
    public static final Locale DEFAULT_LOCALE = Locale.UK;

    /**
     * Parse a list of {@link Locale} seperated by comma.
     *
     * Example : fr_FR,en_GB
     *
     * @param str the string representation of locale separated by comma
     * @return list of available locales
     * @throws IllegalArgumentException ia a locale is not valid
     */
    public static Locale[] parseLocales(String str) throws IllegalArgumentException {
        List<Locale> result = new java.util.ArrayList<Locale>();
        String[] bundlesToUse = str.split(",");
        for (int i = 0, j = bundlesToUse.length; i < j; i++) {
            String s = bundlesToUse[i].trim();
            // on devrait verifier que le bundle existe
            try {
                Locale l = (Locale) new LocaleConverter().convert(Locale.class, s);
                result.add(l);
            } catch (Exception e) {
                throw new IllegalArgumentException("bundle " + s + " is not a valid locale,e");
            }
        }
        return result.toArray(new Locale[result.size()]);
    }

    public static Locale newLocale(String str) {
        if (str == null) {
            // get use locale
            return newLocale(null, null);
        }
        try {
            return (Locale) new LocaleConverter().convert(Locale.class, str);
        } catch (Exception e) {
            Logger.getLogger("org.nuiton.i18n.I18n").warning("could not load locale '" + str + " for reason : " + e.getMessage());
            // use default locale
            return DEFAULT_LOCALE;
        }
    }

    public static Locale newLocale(String language, String country) {
        if (language == null) {
            // get user locale
            language = System.getProperty("user.language", DEFAULT_LOCALE.getLanguage());
            country = System.getProperty("user.country", DEFAULT_LOCALE.getCountry());
        }
        return newLocale(language + (country == null ? "" : '_' + country));
    }

    /**
     * Test if an url contains the given directory with no recurse seeking.
     *
     * @param url       the url to seek
     * @param directory the directory to find
     * @return <code>true</code> if directory was found, <code>false</code> otherwise.
     * @throws java.io.IOException if any io pb
     */
    public static boolean containsDirectDirectory(URL url, String directory) throws IOException {
        String fileName = url.getFile();
        // TODO deal with encoding in windows, this is very durty, but it works...
        File file = new File(fileName.replaceAll("%20", " "));
        if (!file.exists()) {
            return false;
        }
        if (isJar(fileName) || isZip(fileName)) {
            // cas ou le fichier du classLoader est un fichier jar ou zip
            if (log.isTraceEnabled()) {
                log.trace("zip to search " + file);
            }
            return new ZipFile(file).getEntry(directory + "/") != null;
        }
        if (file.isDirectory()) {
            // cas ou le ichier du classLoader est un repertoire
            if (log.isTraceEnabled()) {
                log.trace("directory to search " + file);
            }
            return new File(file, directory).exists();
        }

        if (log.isWarnEnabled()) {
            log.warn("unknown resource type " + url);
        }
        return false;
    }

    /**
     * Verifie si le fichier est un fichier jar.
     *
     * @param name nom du fichier a tester
     * @return vrai si le fichier se termine par .jar faux sinon
     */
    static public boolean isJar(String name) {
        if (name != null && name.length() > 4) {
            String ext = name.substring(name.length() - 4, name.length());
            return ".jar".equalsIgnoreCase(ext);
        }
        return false;
    }

    /**
     * Verifie si le fichier est un fichier zip
     *
     * @param name nom du fichier a tester
     * @return vrai si le fichier se termine par .zip faux sinon
     */
    static public boolean isZip(String name) {
        if (name != null && name.length() > 4) {
            String ext = name.substring(name.length() - 4, name.length());
            return ".zip".equalsIgnoreCase(ext);
        }
        return false;
    }

    /**
     * Retourne la liste des fichiers correspondant au pattern donne, aucun
     * ordre ne doit être supposé sur les fichiers.
     *
     * @param repository repertoire dans lequel on recherche les fichiers
     * @param pattern    le nom du fichier a extraire du fichier du repertoire doit
     *                   correspondre au pattern (repertoire + nom compris). si le
     *                   pattern est null, tous les fichiers trouvé sont retourné.
     * @return la liste des urls correspondant au pattern
     */
    static public List<URL> getURLsFromDirectory(File repository, String pattern) {
        try {
            if (log.isTraceEnabled()) {
                log.trace("search '" + pattern + "' in " + repository);
            }

            List<URL> urlList = new ArrayList<URL>();
            File[] filesList = repository.listFiles();

            if (filesList != null) {

                for (File file : filesList) {

                    String name = file.getAbsolutePath();

                    if (log.isTraceEnabled()) {
                        log.trace("directory: " + repository + " name: " + name);
                    }

                    // cas de recursivite : repertoire dans un repertoire
                    if (file.exists() && file.isDirectory()) {
                        urlList.addAll(getURLsFromDirectory(file,
                                pattern));
                        // si le fichier du repertoire n'est pas un repertoire
                        // on verifie s'il correspond au pattern
                    } else if (pattern == null || name.matches(pattern)) {
                        URL url = file.toURI().toURL();
                        if (log.isTraceEnabled()) {
                            log.trace("directory: " + repository + " url: " + url);
                        }
                        urlList.add(url);
                    }
                }
            }
            if (log.isTraceEnabled()) {
                log.trace("found with pattern '" + pattern + "' : " + urlList);
            }
            return urlList;
        } catch (MalformedURLException eee) {
            throw new IllegalArgumentException("Erreur lors de la conversion de l'url " + repository + " (pattern " + pattern + ") " + eee.getMessage(), eee);
            //throw new ResourceException("Le fichier n'a pu être converti en URL", eee);
        }
    }
}
