package org.mortbay.cometd.continuation;
//========================================================================
//Copyright 2007 Mort Bay Consulting Pty. Ltd.
//------------------------------------------------------------------------
//Licensed under the Apache License, Version 2.0 (the "License");
//you may not use this file except in compliance with the License.
//You may obtain a copy of the License at 
//http://www.apache.org/licenses/LICENSE-2.0
//Unless required by applicable law or agreed to in writing, software
//distributed under the License is distributed on an "AS IS" BASIS,
//WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//See the License for the specific language governing permissions and
//limitations under the License.
//========================================================================

import java.io.IOException;
import java.lang.reflect.Array;
import java.util.Map;
import java.util.Queue;

import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.mortbay.cometd.AbstractBayeux;
import org.mortbay.cometd.AbstractCometdServlet;
import org.mortbay.cometd.ClientImpl;
import org.mortbay.cometd.Transport;
import org.mortbay.util.ajax.Continuation;
import org.mortbay.util.ajax.ContinuationSupport;
import org.mortbay.util.ajax.JSON;

import dojox.cometd.Bayeux;

public class ContinuationCometdServlet extends AbstractCometdServlet
{
    /* ------------------------------------------------------------ */
    protected AbstractBayeux newBayeux()
    {
        return new ContinuationBayeux();
    }

    /* ------------------------------------------------------------ */
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException
    {
        doPost(req,resp);
    }

    /* ------------------------------------------------------------ */
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException
    {
        // Look for an existing client and protect from context restarts
        Object clientObj=req.getAttribute(CLIENT_ATTR);
        ContinuationClient client=(clientObj instanceof ClientImpl)?(ContinuationClient)clientObj:null;
        Transport transport=null;
        boolean reconnect=false;

        // Have we seen this request before
        if (client!=null)
        {
            // This is a retry
            if (_bayeux.isLogDebug())
                _bayeux.logDebug("doPost: client seen before: "+client);
            
            // yes - extract saved properties
            transport=(Transport)req.getAttribute(TRANSPORT_ATTR);
            transport.setResponse(resp);
        }
        else
        {
            // variables to loop through batches and messages
            // May have multiple message parameters, each with an array of
            // messages - or just a message
            String[] batches=null;
            int batch_index=0;
            Object batch=null;
            int index=0;
            Map<String,Object> message=null;
            int message_count=0;

            batches=req.getParameterValues(MESSAGE_PARAM);

            /* aabeling: check jsonp parameter */
            String jsonpParam=req.getParameter("jsonp");
            if (jsonpParam!=null)
            {
                if (_bayeux.isLogDebug())
                    _bayeux.logDebug("doPost: jsonp="+jsonpParam);
            }

            // Loop to the first message - it may be handshake without a client
            while (batches!=null && batch_index<batches.length)
            {
                // Do we need to get another batch?
                if (batch==null)
                {
                    index=0;
                    batch=JSON.parse(batches[batch_index++],_bayeux.isJSONCommented());
                }

                if (batch==null)
                    continue;

                if (batch.getClass().isArray())
                {
                    message=(Map<String,Object>)Array.get(batch,index++);
                    if (index>=Array.getLength(batch))
                        batch=null;
                }
                else
                {
                    message=(Map)batch;
                    batch=null;
                }

                message_count++;

                /* aabeling: store jsonp parameter in message */
                if (jsonpParam!=null)
                {
                    message.put("jsonp",jsonpParam);
                }

                client=(ContinuationClient)_bayeux.getClient((String)message.get(AbstractBayeux.CLIENT_FIELD));
                

                // If no client, this is a handshake
                if (client==null)
                {
                    // Setup a browser ID
                    String browser_id=browserId(req);
                    if (browser_id==null)
                        browser_id=newBrowserId(req,resp);
                    
                    // handshake!
                    transport=_bayeux.newTransport(client,message);
                    transport.setResponse(resp);
                    _bayeux.handle(null,transport,message);
                    message=null;
                    
                }

                break;
            }

            // Handle all client messages
            if (client!=null)
            {
                client.access();
                client.setBrowserId(browserId(req));
                
                // resolve transport
                transport=_bayeux.newTransport(client,message);
                transport.setResponse(resp);
                
                // continue handling messages with a known client and transport!
                try
                {
                    // Tell client to hold messages as a response is likely to
                    // be sent.
                    client.responsePending();

                    // handle any message left over from client loop above
                    if (message!=null)
                    {
                        String channel=_bayeux.handle(client,transport,message);
                        reconnect|=AbstractBayeux.META_RECONNECT.equals(channel);
                    }
                    message=null;

                    // handle all other messages
                    while (batches!=null && batch_index<batches.length)
                    {
                        // Do we need to get another batch?
                        if (batch==null)
                        {
                            index=0;
                            batch=JSON.parse(batches[batch_index++],_bayeux.isJSONCommented());
                        }
                        if (batch==null)
                            continue;

                        // get the next message
                        if (batch.getClass().isArray())
                        {
                            message=(Map)Array.get(batch,index++);
                            if (index>=Array.getLength(batch))
                                batch=null;
                        }
                        else
                        {
                            message=(Map)batch;
                            batch=null;
                        }

                        // handle message
                        if (message!=null)
                        {
                            String channel = _bayeux.handle(client,transport,message);
                            reconnect|=AbstractBayeux.META_RECONNECT.equals(channel);
                        }
                        message=null;
                    }

                }
                finally
                {
                    client.responded();
                }
            }
        }

        // Do we need to wait for messages or are we streaming?
        while (transport.isPolling())
        {
            if (_bayeux.isLogDebug())
                _bayeux.logDebug("doPost: transport is polling");
            long timeout=_timeout;
            
            Continuation continuation=ContinuationSupport.getContinuation(req,client);
            if (!continuation.isPending())
                client.access();
            
            // Get messages or wait
            Queue<Map<String,Object>> messages=null;
            synchronized (client)
            {
                messages=client.takeMessages();

                if ((messages==null||messages.size()==0)&&!continuation.isPending())
                {
                    // save state and suspend
                    ((ContinuationClient)client).setContinuation(continuation);
                    req.setAttribute(CLIENT_ATTR,client);
                    req.setAttribute(TRANSPORT_ATTR,transport);
                    if (_bayeux.isLogDebug())
                        _bayeux.logDebug("doPost: setting transport for request: "+transport);
                    continuation.suspend(timeout);

                    messages=client.takeMessages();
                    
                }
                continuation.reset();
                ((ContinuationClient)client).setContinuation(null);

                if (messages==null) 
                    transport.setPolling(false);
            }

            // Send the messages
            if (messages!=null)
                transport.send(messages);

            // Only a simple poll if the transport does not flush
            if (!transport.keepAlive())
                transport.setPolling(false);
        }

        // Send any left over messages.
        if (client!=null) 
        { 
            if (reconnect)
                client.onBrowser(null);
            Queue<Map<String,Object>> messages = client.takeMessages(); 
            if (messages!=null) 
                transport.send(messages); 
        }
        
        transport.complete();

    }
}
