/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: ImportXmlStreamTasks.java 21 2011-05-09 16:43:58Z sletellier $
 * $HeadURL: http://svn.chorem.org/svn/vradi/tags/vradi-0.6/vradi-services/src/main/java/org/chorem/vradi/services/tasks/ImportXmlStreamTasks.java $
 * %%
 * Copyright (C) 2010 Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

package org.chorem.vradi.services.tasks;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.vradi.VradiServiceConfigurationHelper;
import org.chorem.vradi.beans.XmlStreamImportResult;
import org.chorem.vradi.entities.XmlStream;
import org.chorem.vradi.services.VradiException;
import org.chorem.vradi.services.managers.BindingManager;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.ExceptionUtil;
import org.nuiton.util.TimeLog;
import org.nuiton.wikitty.WikittyProxy;

import javax.mail.Authenticator;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.PasswordAuthentication;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.Properties;
import java.util.TimerTask;

import static org.nuiton.i18n.I18n._;

/**
 * This task run at specified stream import time to import new stream.
 *
 * @author chatellier
 * @version $Revision: 21 $
 *          <p/>
 *          Last update : $Date: 2011-05-09 18:43:58 +0200 (lun., 09 mai 2011) $
 *          By : $Author: sletellier $
 */
public class ImportXmlStreamTasks extends TimerTask {

    private static final Log log = LogFactory.getLog(ImportXmlStreamTasks.class);

    private final static TimeLog timeLog = new TimeLog(ImportXmlStreamTasks.class, 5000, 1000);

    protected ApplicationConfig config;

    protected WikittyProxy wikittyProxy;

    protected BindingManager bindingManager;

    protected XmlStreamImportResult importResult;

    /** Il faut l'id du flux et non le flux (wikitty obselete). */
    protected String xmlStreamId;

    public ImportXmlStreamTasks(ApplicationConfig config,
                                WikittyProxy wikittyProxy,
                                BindingManager bindingManager,
                                String xmlStreamId) {

        this.config = config;
        this.wikittyProxy = wikittyProxy;
        this.bindingManager = bindingManager;
        this.xmlStreamId = xmlStreamId;
    }

    /*
     * @see java.util.TimerTask#run()
     */
    @Override
    public void run() {

        long start = TimeLog.getTime();
        try {
            XmlStream xmlStream = wikittyProxy.restore(XmlStream.class, xmlStreamId);
            if (log.isInfoEnabled()) {
                log.info("Starting xml stream import on : " + xmlStream.getName() + "...");
            }
            importResult = bindingManager.importFormsFromXmlStream(xmlStream);
            if (log.isInfoEnabled()) {
                log.info("Xml stream " + xmlStream.getName() + " imported : " + importResult.getCreatedFormCount() + " new forms.");
            }
        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import forms from stream " + xmlStreamId, eee);
            }
            sendEmailError(xmlStreamId, eee);
        }

        // Log execution duration
        timeLog.log(start, "importXmlStreamTasks");
    }

    protected void sendEmailError(String xmlStreamId, VradiException error) {

        log.info("Error is occured, sending error email");

        // Restore XmlStream
        XmlStream xmlStream = wikittyProxy.restore(XmlStream.class, xmlStreamId);

        // Get admin emails
        List<String> mailAdmin = VradiServiceConfigurationHelper.getMailAdmin(config);
        for (String email : mailAdmin) {

            if (log.isDebugEnabled()) {
                log.debug("Sending error email to " + email);
            }
            if (StringUtils.isBlank(email)) {
                if (log.isWarnEnabled()) {
                    log.warn("Email is empty : " + email);
                }
                return;
            }

            // TODO EC-20100505 use Session.getDefaultInstance()
            // and common Properties form receiving/sending mails !
            Properties props = new Properties();
            props.put("mail.smtp.host", VradiServiceConfigurationHelper.getSmtpHost(config));
            props.put("mail.smtp.port", VradiServiceConfigurationHelper.getSmtpPort(config));
            //props.put("mail.smtp.auth", "true");
            //props.put("mail.smtp.socketFactory.port", config.getSmtpPort());
            //props.put("mail.smtp.socketFactory.class", SSL_FACTORY);
            //props.put("mail.smtp.socketFactory.fallback", "false");
            //props.put("mail.debug", config.isMailDebug());

            Session session = Session.getInstance(props,
                                                  new Authenticator() {
                                                      @Override
                                                      protected PasswordAuthentication getPasswordAuthentication() {
                                                          return new PasswordAuthentication(
                                                                  VradiServiceConfigurationHelper.getMailUser(config),
                                                                  VradiServiceConfigurationHelper.getMailPassword(config));
                                                      }
                                                  });

            // EC-20100427 : attention, ca affiche le contenu des pdf joint
            // c'est absolument illisible
            //session.setDebug(log.isDebugEnabled());

            // create a message
            MimeMessage msg = new MimeMessage(session);

            try {
                // set the from and to address
                InternetAddress addressFrom = new InternetAddress(VradiServiceConfigurationHelper.getMailFrom(config),
                                                                  VradiServiceConfigurationHelper.getMailFromName(config));

                msg.setFrom(addressFrom);

                InternetAddress addressTo = new InternetAddress(email);
                msg.addRecipient(Message.RecipientType.TO, addressTo);

                Multipart multiparts = new MimeMultipart();

                // create the message part
                MimeBodyPart msgBodyPart = new MimeBodyPart();
                msgBodyPart.setText(_("vradi.error.errorImportXmlStream", xmlStream.getName(),
                                      ExceptionUtil.stackTrace(error)));

                multiparts.addBodyPart(msgBodyPart);

                // Put parts in message
                msg.setContent(multiparts);

                // Setting the Subject and Content Type
                msg.setSubject(_("vradi.error.errorImportXmlStream.subject", xmlStream.getName()));

                try {
                    Transport.send(msg);
                } catch (MessagingException eee) {
                    if (log.isErrorEnabled()) {
                        log.error("Cant send mail", eee);
                    }
                }


            } catch (MessagingException eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't generate message", eee);
                }
            } catch (UnsupportedEncodingException eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't generate message", eee);
                }
            }
        }
    }
}
