/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: TemplateManager.java 21 2011-05-09 16:43:58Z sletellier $
 * $HeadURL: http://svn.chorem.org/svn/vradi/tags/vradi-0.6/vradi-services/src/main/java/org/chorem/vradi/services/managers/TemplateManager.java $
 * %%
 * Copyright (C) 2009 - 2010 Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.vradi.services.managers;

import com.sun.star.beans.PropertyValue;
import com.sun.star.beans.XPropertySet;
import com.sun.star.container.XEnumerationAccess;
import com.sun.star.container.XNameAccess;
import com.sun.star.frame.XComponentLoader;
import com.sun.star.frame.XStorable;
import com.sun.star.lang.XComponent;
import com.sun.star.lang.XMultiServiceFactory;
import com.sun.star.text.ControlCharacter;
import com.sun.star.text.TextContentAnchorType;
import com.sun.star.text.XText;
import com.sun.star.text.XTextContent;
import com.sun.star.text.XTextCursor;
import com.sun.star.text.XTextDocument;
import com.sun.star.text.XTextFieldsSupplier;
import com.sun.star.text.XTextRange;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.uno.XComponentContext;
import com.sun.star.uri.ExternalUriReferenceTranslator;
import com.sun.star.uri.XExternalUriReferenceTranslator;
import com.sun.star.util.XRefreshable;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.vradi.VradiServiceConfiguration;
import org.chorem.vradi.VradiServiceConfigurationHelper;
import org.chorem.vradi.services.VradiException;
import org.chorem.vradi.services.ooo.SingletonOOo;
import org.nuiton.util.ApplicationConfig;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * Template manager.
 * <p/>
 * Handle:
 * <ul>
 * <li>OpenOffice.org PDF creation</li>
 * </ul>
 *
 * @author chatellier
 */
public class TemplateManager {

    /** Log. */
    private static final Log log = LogFactory.getLog(TemplateManager.class);

    public static final String TEMPLATE_DATE_FORMAT = "EEEE, d MMMM yyyy HH:mm:ss";

    protected ApplicationConfig config;

    protected XComponentContext context = null;

    protected XComponentLoader loader = null;

    protected XComponent document = null;

    public static final int IMAGE_UNKNOWN = -1;

    public static final int IMAGE_JPEG = 0;

    public static final int IMAGE_PNG = 1;

    public static final int IMAGE_GIF = 2;

    /**
     * Prefix des champs texte utilisateurs dans les template ODT.
     * <p/>
     * TODO EC 20100528 according to documentation, it's "com.sun.star.text.textfield.User"
     * <p/>
     * En ooo2, la casse est differente : com.sun.star.text.FieldMaster.User
     */
    protected static final String USER_TEXT_FIELD_PREFIX = "com.sun.star.text.fieldmaster.User.";

    /**
     * Init template with managed template file.
     *
     * @param templateFile
     * @throws VradiException
     */
    public TemplateManager(ApplicationConfig config, File templateFile) throws VradiException {
        this.config = config;
        initOpenOfficeContext();
        document = createDoc(templateFile);
    }

    /** Init openoffice bootstrap?. */
    protected void initOpenOfficeContext() throws VradiException {

        try {
            String oOoExecFolder = VradiServiceConfigurationHelper.getOpenOfficeExecDir(config);
            context = SingletonOOo.GetInstance(oOoExecFolder).getXContext();
            loader = SingletonOOo.GetInstance(oOoExecFolder).getLoader();
        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't init openoffice bootstrap", ex);
            }
            throw new VradiException("Can't init openoffice bootstrap", ex);
        }

    }

    /**
     * Créer un document.
     *
     * @param templateFile Le template du document
     * @return le document construit
     * @throws VradiException if document can't be created
     */
    public XComponent createDoc(File templateFile) throws VradiException {
        try {
            // Propriété générale du document (voir com.sun.star.document.MediaDescriptor).
            List<PropertyValue> props = new ArrayList<PropertyValue>();

            // Autorise l'utilisation d'un Template.
            PropertyValue p1 = new PropertyValue();
            p1.Name = "AsTemplate";
            p1.Value = Boolean.TRUE;
            props.add(p1);

            // Rend le document invisible.
            PropertyValue p2 = new PropertyValue();
            p2.Name = "Hidden";
            p2.Value = Boolean.TRUE;
            props.add(p2);

            PropertyValue[] properties = new PropertyValue[props.size()];
            properties = props.toArray(properties);

            String templateFileURL = createUNOFileURL(templateFile);
            document = loader.loadComponentFromURL(templateFileURL, // URL du temlpate
                                                   "_blank", // Nom de la fenetre (_blank en crée une nouvelle).
                                                   0, // On ne cherche pas de flag.
                                                   properties); // Propriétées.

            return document;

        } catch (Exception e) {
            throw new VradiException("Can't create document. File " + templateFile
                                     + " can also not exist or not be read !", e);
        }
    }

    public void generateDoc(String targetFileName,
                            Map<String, Object> myValues, String... imagesUrls) throws VradiException {
        fillFields(myValues);
        for (String imageUrl : imagesUrls) {
            insertImageInEndOfDocument(imageUrl);
        }
        storeDocComponent(document, targetFileName);
    }

    /**
     * Sauver un doc donné a l'url donnée au format pdf.
     *
     * @param xDoc     Le document a sauvegarder (XComponent)
     * @param storeUrl Url complete du fichier (String)
     * @throws VradiException if pdf can't be exported
     */
    public void storeDocComponent(XComponent xDoc, String storeUrl)
            throws VradiException {
        try {
            XStorable xStorable = UnoRuntime.queryInterface(
                    XStorable.class, xDoc);
            // FIXME give a File here
            storeUrl = createUNOFileURL(new File(storeUrl));

            if (log.isDebugEnabled()) {
                log.debug("Storing pdf file to " + storeUrl);
            }

            PropertyValue[] storeProps = new PropertyValue[2];
            storeProps[0] = new PropertyValue();
            storeProps[0].Name = "FilterName";
            storeProps[0].Value = "writer_pdf_Export";
            storeProps[1] = new PropertyValue();
            storeProps[1].Name = "Overwrite";
            storeProps[1].Value = Boolean.TRUE;
            xStorable.storeToURL(storeUrl, storeProps);

            // TODO EC20100427 les lignes suivantes ont l'air
            // de fermer un truc qu'il ne faut pas, le les génération suivantes
            // echouent

            //xDoc.dispose();
            //XComponent xComp = (XComponent) UnoRuntime.queryInterface(
            //XComponent.class, xStorable);
            //xComp.dispose();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export template as PDF", eee);
            }
            throw new VradiException("Can't export template as PDF", eee);
        }
    }

    /**
     * Creating a correct File URL that OpenOffice can handle. This is
     * necessary to be platform independent.
     *
     * @param file
     * @return openoffice internal url
     * @throws VradiException in cas of MalformedURLException
     */
    protected String createUNOFileURL(File file) throws VradiException {
        String internalURL;
        try {
            URL before = file.toURI().toURL();

            XExternalUriReferenceTranslator translator = ExternalUriReferenceTranslator.create(context);
            internalURL = translator.translateToInternal(before.toExternalForm());
        } catch (MalformedURLException eee) {
            throw new VradiException("Can't convert url", eee);
        }

        return internalURL;
    }

    /**
     * Remplir les champs de mailing.
     * <p/>
     * Manage only com.sun.star.text.textfield.User.* fields.
     *
     * @param myValues value to replace into template fields
     * @throws VradiException if values can't be replaced
     */
    protected void fillFields(Map<String, Object> myValues) throws VradiException {

        try {
            XTextFieldsSupplier xTextFieldsSupplier = UnoRuntime
                    .queryInterface(XTextFieldsSupplier.class, document);

            XNameAccess xNamedFieldMasters = xTextFieldsSupplier.getTextFieldMasters();
            XEnumerationAccess xEnumeratedFields = xTextFieldsSupplier.getTextFields();
            String allFieldNames[] = xNamedFieldMasters.getElementNames();

            String[] fieldNames = new String[allFieldNames.length];
            //Object[] fieldMasters = new Object[allFieldNames.length];
            XPropertySet[] xPropertySets = new XPropertySet[allFieldNames.length];
            int fieldCount = 0;

            for (String allFieldName : allFieldNames) {

                // compare en toLowerCase parce que :
                // ooo2 : com.sun.star.text.FieldMaster.User
                // ooo3 : com.sun.star.text.fieldmaster.User
                if (allFieldName.toLowerCase().startsWith(USER_TEXT_FIELD_PREFIX.toLowerCase())) {
                    String shortFieldName = allFieldName.substring(USER_TEXT_FIELD_PREFIX.length());
                    fieldNames[fieldCount] = shortFieldName;
                    // ???
                    //fieldMasters[fieldCount] = xNamedFieldMasters.getByName(allFieldName);
                    // xPropertySets[fieldCount] = (XPropertySet) UnoRuntime.queryInterface(XPropertySet.class, fieldMasters[fieldCount]);
                    Object fieldMaster = xNamedFieldMasters.getByName(allFieldName);
                    xPropertySets[fieldCount] = UnoRuntime.queryInterface(XPropertySet.class, fieldMaster);

                    fieldCount++;
                }
            }

            XRefreshable xRefreshable = UnoRuntime
                    .queryInterface(XRefreshable.class, xEnumeratedFields);

            for (int i = 0; i < fieldCount; i++) {
                Object value = myValues.get(fieldNames[i]);

                // sletellier 20110503 : make date humain readable (http://chorem.org/issues/show/337)
                if (value instanceof Date) {
                    ApplicationConfig config = VradiServiceConfiguration.getConfig();
                    Locale locale = VradiServiceConfigurationHelper.getLocale(config);
                    DateFormat dateFormat = new SimpleDateFormat(TEMPLATE_DATE_FORMAT, locale);
                    value = dateFormat.format((Date) value);
                }

                if (log.isDebugEnabled()) {
                    log.debug("Replacing " + fieldNames[i] + " with "
                              + value);
                }

                xPropertySets[i].setPropertyValue("Content", value);


                if (log.isDebugEnabled() && value == null) {
                    log.debug("Field " + fieldNames[i] + " could not be found");
                }
            }

            xRefreshable.refresh();
        } catch (Exception eee) {
            throw new VradiException("Can't replace template fields", eee);
        }
    }

    /**
     * Inserts an image to the end of the document.
     * <p/>
     * Source : http://user.services.openoffice.org/fr/forum/viewtopic.php?f=15&t=6357&p=31958
     *
     * @param imageUrl the url containing the image to insert
     * @throws VradiException
     */
    protected void insertImageInEndOfDocument(String imageUrl) throws VradiException {

        if (log.isDebugEnabled()) {
            log.debug("Insert image at end of document : " + imageUrl);
        }

        try {
            BufferedImage image = ImageIO.read(new URL(imageUrl));
            int width = image.getWidth();
            int height = image.getHeight();
            if (log.isDebugEnabled()) {
                log.debug("Image size : " + width + "*" + height);
            }

            XTextDocument xTextDocument = UnoRuntime.queryInterface(
                    XTextDocument.class, document);

            // Créé un TextRange au début du document
            XTextRange xStart = xTextDocument.getText().getStart();
            XTextCursor xCursor = xStart.getText().createTextCursorByRange(xStart);

            // Querying for the interface XMultiServiceFactory on the XTextDocument
            XMultiServiceFactory xMSFDoc = UnoRuntime
                    .queryInterface(XMultiServiceFactory.class, xTextDocument);

            // Creating the service GraphicObject
            //XTextContent xImage = (XTextContent)xMSFDoc.createInstance("com.sun.star.text.TextGraphicObject");
            XTextContent xImage = UnoRuntime.queryInterface(XTextContent.class,
                                                            xMSFDoc.createInstance("com.sun.star.text.TextGraphicObject"));

            // Getting the text and append new paragraph
            XText xText = xCursor.getText();
            xText.insertControlCharacter(xText.getEnd(),
                                         ControlCharacter.APPEND_PARAGRAPH, false);

            // Querying for the interface XPropertySet on the graphic object
            XPropertySet xProps = UnoRuntime.queryInterface(
                    XPropertySet.class, xImage);

            // Détermine l'ancrage à la page
            xProps.setPropertyValue("AnchorType", TextContentAnchorType.AT_PARAGRAPH);

            // Détermine l'URL concernant l'image (Format: file:///C:/temp/image.gif)
            xProps.setPropertyValue("GraphicURL", imageUrl);

            //Détermine si l'objet sera imprimé aussi lors de l'impression du doc.
            xProps.setPropertyValue("Print", true);
            xProps.setPropertyValue("Width", width * 15);
            xProps.setPropertyValue("Height", height * 15);

            //Setting the top margin
            xProps.setPropertyValue("TopMargin", height * 15);

            // Insert image to the end
            xText.insertTextContent(xText.getEnd(), xImage, false);

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't insert image", eee);
            }
            throw new VradiException("Can't insert image", eee);
        }

    }

    /**
     * Gets the user fields or database fields of the ooo document.
     * <p/>
     * Some help : http://wiki.services.openoffice.org/wiki/Documentation/DevGuide/Text/Text_Fields
     * <p/>
     * Manage fields :
     * - com.sun.star.text.textfield.User
     * Variables - User Field. Creates a global document variable and displays it whenever this field occurs in the text.
     *
     * @return an array of String containing the field names
     */
    public List<String> getDocumentFields() {

        XTextFieldsSupplier xTextFieldsSupplier = UnoRuntime
                .queryInterface(XTextFieldsSupplier.class, document);
        XNameAccess xNamedFieldMasters = xTextFieldsSupplier.getTextFieldMasters();

        String allFieldNames[] = xNamedFieldMasters.getElementNames();
        List<String> fieldNames = new ArrayList<String>();

        for (String allFieldName : allFieldNames) {
            // compare en toLowerCase parce que :
            // ooo2 : com.sun.star.text.FieldMaster.User
            // ooo3 : com.sun.star.text.fieldmaster.User
            if (allFieldName.toLowerCase().startsWith(USER_TEXT_FIELD_PREFIX.toLowerCase())) {
                String shortFieldName = allFieldName.substring(USER_TEXT_FIELD_PREFIX.length());
                fieldNames.add(shortFieldName);
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Found field in document : " + fieldNames);
        }

        return fieldNames;
    }
}
