/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: ClientManager.java 21 2011-05-09 16:43:58Z sletellier $
 * $HeadURL: http://svn.chorem.org/svn/vradi/tags/vradi-0.6/vradi-services/src/main/java/org/chorem/vradi/services/managers/ClientManager.java $
 * %%
 * Copyright (C) 2009 - 2010 Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.vradi.services.managers;

import com.sun.syndication.feed.synd.SyndContent;
import com.sun.syndication.feed.synd.SyndContentImpl;
import com.sun.syndication.feed.synd.SyndEntry;
import com.sun.syndication.feed.synd.SyndEntryImpl;
import com.sun.syndication.feed.synd.SyndFeed;
import com.sun.syndication.feed.synd.SyndFeedImpl;
import com.sun.syndication.io.SyndFeedInput;
import com.sun.syndication.io.SyndFeedOutput;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.vradi.VradiServiceConfigurationHelper;
import org.chorem.vradi.beans.QueryBean;
import org.chorem.vradi.entities.Client;
import org.chorem.vradi.entities.Group;
import org.chorem.vradi.entities.ModificationTag;
import org.chorem.vradi.entities.QueryMaker;
import org.chorem.vradi.entities.Sending;
import org.chorem.vradi.entities.Session;
import org.chorem.vradi.entities.User;
import org.chorem.vradi.services.VradiException;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.operators.Element;

import java.io.File;
import java.io.FileWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * Class that contains the methods to retrieve the information
 * related to the clients
 *
 * @author schorlet
 * @version $Revision: 21 $ $Date: 2011-05-09 18:43:58 +0200 (lun., 09 mai 2011) $
 * @date 2010-01-22 20:18:29
 */
public class ClientManager {
    private static final Log log = LogFactory.getLog(ClientManager.class);

    protected ApplicationConfig config;

    protected WikittyProxy wikittyProxy;

    public ClientManager(ApplicationConfig config, WikittyProxy wikittyProxy) {
        this.config = config;
        this.wikittyProxy = wikittyProxy;
    }

    /**
     * Delete client, users associated and user groups association
     *
     * @param clientId to delete
     * @throws VradiException throw vradiException if exception is occured
     */
    public void deleteClient(String clientId) throws VradiException {

        List<User> users = getUsersByClient(clientId);

        for (User user : users) {
            deleteUser(user.getWikittyId());
        }
        wikittyProxy.delete(clientId);
    }

    /**
     * Delete users, and groups association
     *
     * @param userId to delete
     * @throws VradiException throw vradiException if exception is occured
     */
    public void deleteUser(String userId) throws VradiException {

        List<Group> groups = getGroupsForUser(userId);
        for (Group group : groups) {
            group.removeUser(userId);
        }

        // Delete sending associated
        Search search = Search.query();
        search = search.eq(Element.ELT_EXTENSION, Sending.EXT_SENDING);
        search = search.eq(Sending.FQ_FIELD_SENDING_USER, userId);
        Criteria criteria = search.criteria();

        PagedResult<Sending> sendings = wikittyProxy.findAllByCriteria(Sending.class, criteria);

        List<Sending> all = sendings.getAll();
        List<String> sendingIds = new ArrayList<String>();
        for (Sending sending : all) {
            String sendingId = sending.getWikittyId();
            sendingIds.add(sendingId);
        }

        wikittyProxy.delete(sendingIds);

        // Remove sending from session
        search = Search.query();
        search = search.eq(Element.ELT_EXTENSION, Session.EXT_SESSION);
        search = search.eq(Session.FQ_FIELD_SESSION_SENDING, sendingIds);
        criteria = search.criteria();

        PagedResult<Session> sessionsResult = wikittyProxy.findAllByCriteria(Session.class, criteria);

        List<Session> sessions = sessionsResult.getAll();
        for (Session session : sessions) {
            for (String sendingId : sendingIds) {
                session.removeSending(sendingId);
            }
        }

        wikittyProxy.store(sessions);
        wikittyProxy.delete(sendingIds);

        // Store groups
        wikittyProxy.store(groups);

        // Delete users
        wikittyProxy.delete(userId);
    }

    /**
     * Delete group
     *
     * @param groupId to delete
     * @throws VradiException throw vradiException if exception is occured
     */
    public void deleteGroup(String groupId) throws VradiException {
        wikittyProxy.delete(groupId);
    }

    public User getUser(String userId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getUser(" + userId + ")");
        }

        User user = wikittyProxy.restore(User.class, userId);
        return user;
    }

    public Client getClient(String clientId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getClient(" + clientId + ")");
        }

        Client client = wikittyProxy.restore(Client.class, clientId);
        return client;
    }

    public Group getGroup(String groupId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getGroup(" + groupId + ")");
        }

        Group group = wikittyProxy.restore(Group.class, groupId);
        return group;
    }

    public List<User> getGroupUsers(String groupId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getGroupUsers(" + groupId + ")");
        }

        List<User> users = new ArrayList<User>();
        Group group = getGroup(groupId);

        if (group != null && group.getUser() != null) {
            ArrayList<String> userIds = new ArrayList<String>(group.getUser());
            List<User> restore = wikittyProxy.restore(User.class, userIds);
            users.addAll(restore);
        }

        return users;
    }

    /**
     * Find all users for a client.
     *
     * @param clientId client wikitty id
     * @return all users for client
     */
    public List<User> getUsersByClient(String clientId) {

        // Get user of client
        Search search = Search.query().eq(User.FQ_FIELD_USER_CLIENT, clientId);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(User.FQ_FIELD_USER_NAME);

        PagedResult<User> usersResult = wikittyProxy.findAllByCriteria(User.class, criteria);
        List<User> users = new ArrayList<User>(usersResult.getAll());
        return users;
    }

    public List<Group> getGroupsForUser(String userId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("get Groups for user : " + userId);
        }


        Search search = Search.query();
        search = search.eq(Element.ELT_EXTENSION, Group.EXT_GROUP);
        search = search.contains(Group.FQ_FIELD_GROUP_USER, userId);
        Criteria criteria = search.criteria();

        PagedResult<Group> groups = wikittyProxy.findAllByCriteria(Group.class, criteria);

        List<Group> all = groups.getAll();

        if (log.isDebugEnabled()) {
            log.debug(all.size() + " groups found");
        }
        return all;
    }

    public Client getClientByUserId(String userId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getClientByUserId(" + userId + ")");
        }

        User user = getUser(userId);
        String clientId = user.getClient();

        return getClient(clientId);
    }

    public List<Client> getAllClients() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getAllClients()");
        }

        Search search = Search.query().eq(Element.ELT_EXTENSION, Client.EXT_CLIENT);
        Criteria criteria = search.criteria();

        PagedResult<Client> clients = wikittyProxy.findAllByCriteria(Client.class, criteria);
        List<Client> all = clients.getAll();

        return new ArrayList<Client>(all);
    }

    public List<User> getAllUsers() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getAllUsers()");
        }

        Search search = Search.query().eq(Element.ELT_EXTENSION, User.EXT_USER);
        Criteria criteria = search.criteria();

        PagedResult<User> users = wikittyProxy.findAllByCriteria(User.class, criteria);
        List<User> all = users.getAll();

        return new ArrayList<User>(all);
    }

    public List<Group> getAllGroups() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getAllGroups()");
        }

        Search search = Search.query().eq(Element.ELT_EXTENSION, Group.EXT_GROUP);
        Criteria criteria = search.criteria();

        PagedResult<Group> groups = wikittyProxy.findAllByCriteria(Group.class, criteria);
        List<Group> all = groups.getAll();

        return new ArrayList<Group>(all);
    }

    /**
     * Archives the queries of a user, client or group in a RSS file.
     *
     * @param queryMaker the query maker whose queries are archived
     * @throws VradiException throw vradiException if exception is occured
     */
    public void archiveQueries(QueryMaker queryMaker) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("archiveQueries(id:" + queryMaker.getWikittyId() +
                      "; version:" + queryMaker.getWikittyVersion() + ")");
        }

        try {
            String wikittyId = queryMaker.getWikittyId();
            Criteria criteria = Search.query().eq(Element.ELT_ID, wikittyId).criteria();
            QueryMaker restored = wikittyProxy.findByCriteria(QueryMaker.class, criteria);

            if (restored != null && restored.getQueries() != null) {
                Set<String> restoredQueries = restored.getQueries();
                Set<String> updatedQueries = queryMaker.getQueries();

                if (updatedQueries != null) {
                    Collection disjunction = CollectionUtils.disjunction(restoredQueries, updatedQueries);
                    if (disjunction.isEmpty()) {
                        if (log.isDebugEnabled()) {
                            log.debug("Disjunction between new and old queryMaker is empty");
                        }
                        return;
                    }
                }
            } else {
                boolean restoredQueriesEmpty = restored == null
                                               || restored.getQueries() == null
                                               || restored.getQueries().isEmpty();

                Set<String> updatedQueries = queryMaker.getQueries();
                boolean updatedQueriesEmpty = updatedQueries == null
                                              || updatedQueries.isEmpty();

                if (restoredQueriesEmpty && updatedQueriesEmpty) {
                    if (log.isDebugEnabled()) {
                        log.debug("Restored query and updated query are empty");
                    }
                    return;
                }
            }

            String extName;
            if (queryMaker instanceof Client) {
                extName = Client.EXT_CLIENT;

            } else if (queryMaker instanceof User) {
                extName = User.EXT_USER;

            } else if (queryMaker instanceof Group) {
                extName = Group.EXT_GROUP;

            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Extention not found, stopping archiving queries");
                }
                return;
            }

            File queryHistoryDir = VradiServiceConfigurationHelper.getQueryHistoryDir(config);
            File feedFile = new File(queryHistoryDir, wikittyId + ".rss");

            SyndFeed feed;
            if (feedFile.exists()) {
                // read existing feed
                SyndFeedInput input = new SyndFeedInput();
                feed = input.build(feedFile);

            } else {
                feedFile.getParentFile().mkdirs();
                feedFile.createNewFile();
                Object queryMakerName = queryMaker.getField(extName, "name");

                // creates new feed
                feed = new SyndFeedImpl();
                feed.setFeedType("rss_2.0");
                feed.setTitle("Requetes de " + queryMakerName);
                feed.setDescription("Historique des requetes");
                feed.setLink("");
                feed.setAuthor("Vradi");
                feed.setPublishedDate(new Date());
            }

            // creates new entry
            SyndEntry entry = new SyndEntryImpl();
            entry.setPublishedDate(new Date());

            if (queryMaker.getExtensionNames().contains(ModificationTag.EXT_MODIFICATIONTAG)) {
                entry.setAuthor((String) queryMaker.getField(
                        ModificationTag.EXT_MODIFICATIONTAG,
                        ModificationTag.FIELD_MODIFICATIONTAG_LASTMODIFIER));
            }

            // add queries
            StringBuffer sb = new StringBuffer("<ul>");
            Set<String> queries = queryMaker.getQueries();
            if (queries != null) {
                for (String query : queries) {
                    try {
                        QueryBean queryBean = new QueryBean(query);
                        sb.append("<li>")
                                .append("<name><![CDATA[").append(queryBean.getName()).append("]]</name>")
                                .append("<description><![CDATA[").append(queryBean.getDescription()).append("]]</description>")
                                .append("<query><![CDATA[").append(queryBean.getQuery()).append("]]</query>")
                                .append("</li>");
                    } catch (Exception e) {
                        log.warn(e.getMessage(), e);
                    }
                }
            }
            sb.append("</ul>");

            SyndContent syndQuery = new SyndContentImpl();
            syndQuery.setType("text/html");
            syndQuery.setValue(sb.toString());
            entry.setDescription(syndQuery);

            // add newly created entry to entries
            List entries = feed.getEntries();
            entries.add(0, entry);
            feed.setEntries(entries);

            Writer writer = new FileWriter(feedFile);
            SyndFeedOutput output = new SyndFeedOutput();
            output.output(feed, writer);
            writer.close();

            if (log.isDebugEnabled()) {
                log.debug("history recorded to: " + feedFile);
            }

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't archive queries", eee);
            }
            throw new VradiException("Can't archive queries", eee);
        }
    }

    public String getQueryHistoryFile(String id) {
        File queryHistoryDir = VradiServiceConfigurationHelper.getQueryHistoryDir(config);
        File queryHistoryFile = new File(queryHistoryDir, id + ".rss");

        if (!queryHistoryFile.exists()) {
            return null;
        }

        String dirName = queryHistoryDir.getName();

        String filePath = queryHistoryFile.getAbsolutePath();
        int indexOf = filePath.indexOf(dirName);

        filePath = filePath.substring(indexOf);
        return filePath;
    }

}
