/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.business;

import java.lang.reflect.Method;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.dto.PreventRuleDTO;
import org.chorem.pollen.business.utils.MailUtil;

/**
 * Classe de gestion des règles de notification. Cette classe exécute des
 * méthodes en fonction de la règle définie.
 *
 * @author rannou
 * @version $Id: PreventRuleManager.java 2904 2010-03-08 15:30:41Z echatellier $
 */
public class PreventRuleManager {

    /** log. */
    private static final Log log = LogFactory.getLog(PreventRuleManager.class);

    /** Représente l'action qui ne fait rien. */
    public static final String NULL_ACTION = "nullAction";
    /** Représente l'action qui écrit sur la sortie standard. */
    public static final String PRINT_ACTION = "printAction";
    /** Représente l'action qui envoi un email de notification. */
    public static final String EMAIL_ACTION = "emailAction";

    private PreventRuleDTO preventRule = null;

    /**
     * Constructeur.
     *
     * @param preventRule règle à utiliser
     */
    public PreventRuleManager(PreventRuleDTO preventRule) {
        this.preventRule = preventRule;
    }

    /**
     * Exécution de la méthode définie par la règle. La méthode s'exécute sans
     * prise en compte de la sensibilité de la règle.
     */
    public boolean execute() {
        return execute(null, null, null);
    }

    /**
     * Exécution de la méthode définie par la règle. La méthode s'exécute si la
     * valeur correspond à la sensibilité de la règle.
     *
     * @param value valeur conditionnant l'exécution de la règle.
     */
    public boolean execute(Integer value) {
        return execute(null, value, null);
    }

    /**
     * Exécution de la méthode définie par la règle. La méthode s'exécute si la
     * portée est égale à la portée de la règle.
     *
     * @param scope portée de la règle.
     */
    public boolean execute(String scope) {
        return execute(scope, null, null);
    }

    /**
     * Exécution de la méthode définie par la règle. La méthode s'exécute si la
     * valeur correspond à la sensibilité de la règle et si la portée est égale
     * à la portée de la règle.
     *
     * @param scope portée de la règle.
     * @param value valeur conditionnant l'exécution de la règle.
     */
    public boolean execute(String scope, Integer value) {
        return execute(scope, value, null);
    }

    /**
     * Exécution de la méthode définie par la règle. La méthode s'exécute si la
     * valeur correspond à la sensibilité de la règle et si la portée est égale
     * à la portée de la règle.
     *
     * @param scope portée de la règle.
     * @param value valeur conditionnant l'exécution de la règle.
     * @param data données utiles pour l'exécution de la règle.
     * @return true si la méthode à été exécutée, false sinon.
     */
    public boolean execute(String scope, Integer value, Object data) {

        // Si la règle n'est pas active, on ne fait rien
        if (!preventRule.isActive()) {
            log.debug("Règle non active");
            return false;
        }

        // Si la règle est à exécution unique, on la désactive
        if (preventRule.isOneTime()) {
            log.debug("Règle à exécution unique : désactivation");
            preventRule.setActive(false);
        }

        // Si la règle est répétitive (exécution tous les n), on met à jour la valeur
        if (value != null && preventRule.isRepeated()
                && preventRule.getSensibility() != 0) {
            value = value % preventRule.getSensibility()
                    + preventRule.getSensibility();
        }

        try {
            if ((value == null || value == preventRule.getSensibility())
                    && (scope == null || scope.equals(preventRule.getScope()))) {
                if (log.isDebugEnabled()) {
                    log.debug("méthod " + preventRule.getMethod() + "(scope : "
                            + preventRule.getScope() + ", sensibility : "
                            + preventRule.getSensibility() + ")");
                }

                Method method = this.getClass().getMethod(
                        preventRule.getMethod(), Object.class);
                method.invoke(this, data);
            } else {
                return false;
            }
        } catch (Exception e) {
            log.error("Erreur lors de l'appel de la méthode "
                    + preventRule.getMethod(), e);
            return false;
        }
        return true;
    }

    /**
     * Action qui ne fait rien.
     * 
     * TODO set this method to protected
     */
    public static void nullAction(Object data) {
        // ne fait rien
    }

    /**
     * Action qui écrit sur la sortie standard.
     * 
     * TODO set this method to protected
     */
    public static void printAction(Object data) {
        String message = "";
        if (data instanceof String) {
            message = (String) data;
        }
        System.out.println("printAction : " + message);
    }

    /**
     * Action qui envoi un email de notification.
     * 
     * @param data mail data
     * 
     * TODO set this method to protected
     */
    public static void emailAction(Object data) {
        Map<String, String> fields = null;
        if (data instanceof Map<?, ?>) {
            fields = (Map<String, String>) data;
            MailUtil.sendMail(fields.get("host"), Integer.parseInt(fields
                    .get("port")), fields.get("from"), fields.get("to"), fields
                    .get("title"), fields.get("msg"));
        }
        else {
            if (log.isWarnEnabled()) {
                log.warn("emailAction data parameter is not instance of Map");
            }
        }
        
    }
}
