package fr.ifremer.tutti.ui.swing.updater;

/*
 * #%L
 * Tutti :: UI Updater
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2015 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import javax.swing.JOptionPane;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.PosixFilePermission;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author Ludovic Pecquot (ludovic.pecquot@e-is.pro)
 * @author tony Chemit (chemit@codelutin.com)
 * @since 3.12
 */
public class Updater {

    public static final String APPLICATION_UPDATER_TITLE = "Allegro Campaign UI Updater";

    public static final String NEW_DIR = "NEW";

    public static final String OLD_DIR = "OLD";

    public static final String LAUNCHER_DIR = "launcher";

    public static final String UPDATE_RUNTIME_CMD = "update_runtime";

    public static final String BATCH_WINDOWS_EXT = ".bat";

    public static final String BATCH_UNIX_EXT = ".sh";

    public static final String VERSION_FILE = "version.appup";

    public static final int NORMAL_EXIT_CODE = 0;

    public static final int ERROR_EXIT_CODE = 1;

    public static final int RUNTIME_UPDATE_EXIT_CODE = 90;

    private final Path baseDir;

    private final String backupDate;

    private final boolean windowsOS;

    public static void main(String... args) {

        // Instantiate the launcher
        Updater updater = new Updater();
        int exitCode = updater.execute();

        // exit 
        System.exit(exitCode);

    }

    public Updater() {

        // Get the current directory where application has been launched
        baseDir = Paths.get(System.getProperty("user.dir"));

        // Compute the date to create backup directories
        Date now = new Date();
        DateFormat df = new SimpleDateFormat("yyyyMMddHHmmss");
        backupDate = df.format(now);

        windowsOS = System.getProperty("os.name").startsWith("Windows");

    }

    private Path getBackupDirectory() throws IOException {

        Path backupDirectory = baseDir.resolve(OLD_DIR);
        if (!Files.isDirectory(backupDirectory)) {
            Files.createDirectory(backupDirectory);
        }

        return backupDirectory;

    }


    public int execute() {

        System.out.println("updater started at " + new Date().toString());

        int exitCode;

        try {

            // Check runtime update (jre or launcher)
            boolean runtimeUpdate = checkRuntimeUpdate();

            if (runtimeUpdate) {

                exitCode = RUNTIME_UPDATE_EXIT_CODE;

            } else {

                // Launch update
                launchUpdate();

                exitCode = NORMAL_EXIT_CODE;

            }

        } catch (Exception ex) {

            ex.printStackTrace();
            exitCode = ERROR_EXIT_CODE;

        }

        System.out.println("updater ended   at " + new Date().toString());

        return exitCode;

    }

    protected boolean checkRuntimeUpdate() throws Exception {

        String scriptFilename = UPDATE_RUNTIME_CMD + (windowsOS ? BATCH_WINDOWS_EXT : BATCH_UNIX_EXT);

        Path runtimeUpdater = baseDir.resolve(scriptFilename);

        Files.deleteIfExists(runtimeUpdater);

        boolean mustUpdateRuntime = false;

        for (Module module : Module.values()) {

            if (module.isRuntimeModule()) {

                boolean updateFound = updateRuntimeModule(module);

                if (updateFound) {
                    mustUpdateRuntime = true;
                }

            }

        }

        if (mustUpdateRuntime) {

            // A runtime update is available, so generate the script
            URL resource = getClass().getResource("/" + scriptFilename);

            try (InputStream stream = resource.openStream()) {

                Files.copy(stream, runtimeUpdater, StandardCopyOption.REPLACE_EXISTING);

            }

            makeExecutable(runtimeUpdater);

            String message = String.format("Runtime updates available.\nYou must execute '%s' manually to apply new runtime.", runtimeUpdater.getFileName());
            System.out.println(message);
            JOptionPane.showMessageDialog(null, message);

        }

        return mustUpdateRuntime;

    }

    protected void launchUpdate() throws IOException {

        for (Module module : Module.values()) {

            if (!module.isRuntimeModule()) {

                updateModule(module);

            }
        }

        // Cleaning process
        cleanObsoleteFiles();
        DeleteHelper.deleteDirectory(baseDir.resolve(NEW_DIR));

    }

    protected boolean updateRuntimeModule(Module module) throws IOException {

        boolean updateFound;
        String moduleName = module.name();

        Path modulePath = baseDir.resolve(moduleName);
        String oldVersion = getVersion(modulePath);

        String moduleNameStr = module.getModuleLoggerName();

        System.out.println(String.format("%s Current version: %s", moduleNameStr, oldVersion));

        Path moduleNewPath = baseDir.resolve(NEW_DIR).resolve(moduleName);

        if (Files.isDirectory(moduleNewPath)) {

            String newVersion = getVersion(moduleNewPath);

            System.out.println(String.format("%s New version detected %s", moduleNameStr, newVersion));

            Path backupDirectory = getBackupDirectory();

            // Remove older backup
            System.out.println(String.format("%s Clean backup directory %s", moduleNameStr, backupDirectory + File.separator + moduleName + "-*"));
            DeleteHelper.deleteDirectories(backupDirectory, moduleName + "-*");

            updateFound = true;

        } else {

            System.out.println(String.format("%s No update found", moduleNameStr));
            updateFound = false;

        }

        return updateFound;

    }

    private void updateModule(Module module) throws IOException {

        String moduleName = module.name();

        Path modulePath = baseDir.resolve(moduleName);
        String oldVersion = getVersion(modulePath);

        String moduleNameStr = module.getModuleLoggerName();

        System.out.println(String.format("%s Current version: %s", moduleNameStr, oldVersion));

        // Update a single module. moduleName corresponds to the name of the directory inside the baseDir
        Path moduleNewPath = baseDir.resolve(NEW_DIR).resolve(moduleName);

        if (Files.isDirectory(moduleNewPath)) {

            String newVersion = getVersion(moduleNewPath);

            System.out.println(String.format("%s New version detected %s", moduleNameStr, newVersion));

            Path backupDirectory = getBackupDirectory();

            // Remove older backup
            System.out.println(String.format("%s Clean backup directory %s", moduleNameStr, backupDirectory + File.separator + moduleName + "-*"));
            DeleteHelper.deleteDirectories(backupDirectory, moduleName + "-*");

            // Backup existing module
            if (Files.isDirectory(modulePath)) {

                Path moduleOldPath = backupDirectory.resolve(String.format("%s-%s-%s", moduleName, oldVersion, backupDate));
                System.out.println(String.format("%s Backup old version %s to %s", moduleNameStr, oldVersion, moduleOldPath.toString()));
                Files.move(modulePath, moduleOldPath);

            }

            // Installing new module
            System.out.println(String.format("%s Install new version %s", moduleNameStr, newVersion));
            Files.move(moduleNewPath, modulePath, StandardCopyOption.REPLACE_EXISTING);

        } else {

            System.out.println(String.format("%s No update found", moduleNameStr));

        }

    }

    private String getVersion(Path path) throws IOException {

        // Return the version of a module from version.appup file
        Path versionFile = path.resolve(VERSION_FILE);
        List<String> lines = Files.readAllLines(versionFile, StandardCharsets.UTF_8);
        if (lines == null || lines.isEmpty()) {
            throw new IOException(versionFile.toString() + " is empty");
        }
        return lines.get(0);
    }

    protected void makeExecutable(Path path) throws IOException {

        if (!windowsOS) {

            Set<PosixFilePermission> perms = new HashSet<>();
            //add owners permission
            perms.add(PosixFilePermission.OWNER_READ);
            perms.add(PosixFilePermission.OWNER_WRITE);
            perms.add(PosixFilePermission.OWNER_EXECUTE);
            //add group permissions
            perms.add(PosixFilePermission.GROUP_READ);
            perms.add(PosixFilePermission.GROUP_WRITE);
            perms.add(PosixFilePermission.GROUP_EXECUTE);
            //add others permissions
            perms.add(PosixFilePermission.OTHERS_READ);
            perms.add(PosixFilePermission.OTHERS_EXECUTE);

            Files.setPosixFilePermissions(path, perms);

        }

    }

    private void cleanObsoleteFiles() throws IOException {

        Path applicationDirectoryPath = baseDir.resolve(Module.tutti.name());

        if (windowsOS) {

            // Delete obsolete batch files
            Files.deleteIfExists(baseDir.resolve("tutti.bat"));
            Files.deleteIfExists(applicationDirectoryPath.resolve("launch.bat"));

            // Delete non Windows files
            DeleteHelper.deleteFiles(baseDir, "*" + BATCH_UNIX_EXT);
        } else {

            // Delete obsolete script files
            Files.deleteIfExists(applicationDirectoryPath.resolve("launch.sh"));

            // Delete Windows files
            DeleteHelper.deleteFiles(baseDir, "*" + BATCH_WINDOWS_EXT);
            DeleteHelper.deleteFiles(baseDir, "*.exe");
        }

        // Delete embedded files
        Files.deleteIfExists(applicationDirectoryPath.resolve("tutti.exe"));
        Files.deleteIfExists(applicationDirectoryPath.resolve("tutti.sh"));
        DeleteHelper.deleteDirectory(applicationDirectoryPath.resolve("launcher"));

    }


}
