package fr.ifremer.echobase.ui.actions.exportCoser;

/*
 * #%L
 * EchoBase :: UI
 * %%
 * Copyright (C) 2011 - 2014 Ifremer, Codelutin, Chemit Tony
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.Maps;
import fr.ifremer.echobase.entities.references.DataMetadata;
import fr.ifremer.echobase.entities.references.Mission;
import fr.ifremer.echobase.persistence.JdbcConfiguration;
import fr.ifremer.echobase.services.service.CoserApiService;
import fr.ifremer.echobase.services.service.exportCoser.ExportCoserConfiguration;
import fr.ifremer.echobase.ui.EchoBaseSession;
import fr.ifremer.echobase.ui.actions.AbstractConfigureAction;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;

import javax.inject.Inject;
import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created on 3/1/14.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 2.6
 */
public class Configure extends AbstractConfigureAction<ExportCoserConfiguration> {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(Configure.class);

    @Inject
    private transient CoserApiService coserApiService;

    /**
     * Universe of missions.
     */
    protected Map<String, String> missions;

    /**
     * Universe of population indicators.
     */
    protected Map<String, String> populationIndicators;

    /**
     * Universe of community indicators.
     */
    protected Map<String, String> communityIndicators;

    /**
     * Universe of facades (coming from Coser).
     */
    protected Map<String, String> facades;

    /**
     * Universe of zones (coming from Coser)
     * for a given facade.
     */
    protected Map<String, String> zones;

    public Configure() {
        super(ExportCoserConfiguration.class);
    }

    public Map<String, String> getMissions() {
        return missions;
    }

    public Map<String, String> getFacades() {
        return facades;
    }

    public Map<String, String> getZones() {
        return zones;
    }

    public Map<String, String> getPopulationIndicators() {
        return populationIndicators;
    }

    public Map<String, String> getCommunityIndicators() {
        return communityIndicators;
    }

    @Override
    protected ExportCoserConfiguration createModel() {
        ExportCoserConfiguration result = new ExportCoserConfiguration();

        EchoBaseSession userSession = getEchoBaseSession();
        JdbcConfiguration workingDbConfiguration = userSession.getWorkingDbConfiguration();
        result.setDbConfiguration(workingDbConfiguration);
        result.setUserName(userSession.getUser().getEmail());
        return result;
    }

    @Override
    protected void prepareInputAction(ExportCoserConfiguration model) {
        loadInputs();
    }

    @Override
    protected void prepareExecuteAction(ExportCoserConfiguration model) throws IOException {

        File tempDirectory = FileUtils.getTempDirectory();
        File dataDirectory = new File(tempDirectory,
                                      "echobase-exportCoser-" + System.currentTimeMillis());
        FileUtil.createDirectoryIfNecessary(dataDirectory);
        model.setWorkingDirectory(dataDirectory);
        if (log.isInfoEnabled()) {
            log.info("Temporary directory to use : " + dataDirectory);
        }
        loadInputs();
    }

    protected void loadInputs() {

        // Get missions
        missions = userDbPersistenceService.loadSortAndDecorate(Mission.class);

        // Get facades from Coser
        facades = coserApiService.getFacades();

        // Get Zones from Coser
        if (StringUtils.isNotEmpty(getModel().getFacade())) {

            zones = coserApiService.getZonesForFacade(getModel().getFacade());
        } else {

            // No zone to display
            zones = Maps.newHashMap();
        }

        // Get indicators names from Coser
        Set<String> indicatorNames = coserApiService.getIndicators();

        List<DataMetadata> dataMetadatas = userDbPersistenceService.getDataMetadatasInName(indicatorNames);
        populationIndicators = decoratorService.sortAndDecorate(getLocale(), dataMetadatas, null);
        communityIndicators = decoratorService.sortAndDecorate(getLocale(), dataMetadatas, null);
    }
}
