/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: EchoBaseActionSupport.java 615 2012-09-05 09:26:02Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/echobase/tags/echobase-1.3/echobase-ui/src/main/java/fr/ifremer/echobase/ui/actions/EchoBaseActionSupport.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui.actions;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.opensymphony.xwork2.ActionContext;
import fr.ifremer.echobase.EchoBaseConfiguration;
import fr.ifremer.echobase.I18nAble;
import fr.ifremer.echobase.services.DecoratorService;
import fr.ifremer.echobase.services.DefaultEchoBaseServiceContext;
import fr.ifremer.echobase.services.EchoBaseService;
import fr.ifremer.echobase.services.EchoBaseServiceContext;
import fr.ifremer.echobase.services.EchoBaseServiceSupport;
import fr.ifremer.echobase.services.ServiceFactoryAware;
import fr.ifremer.echobase.ui.EchoBaseApplicationContext;
import fr.ifremer.echobase.ui.EchoBaseInternalDbTransactionFilter;
import fr.ifremer.echobase.ui.EchoBaseSession;
import fr.ifremer.echobase.ui.EchoBaseWorkingDbTransactionFilter;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.interceptor.ParameterAware;
import org.apache.struts2.interceptor.SessionAware;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.web.struts2.BaseAction;

import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * EchoBase action support.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EchoBaseActionSupport extends BaseAction implements ServiceFactoryAware {

    private static final long serialVersionUID = 1L;

    private static final SimpleDateFormat dateFormat =
            new SimpleDateFormat("dd/MM/yyyy");

    private static final SimpleDateFormat monthFormat =
            new SimpleDateFormat("mm-yyyy");

    private static EchoBaseApplicationContext applicationContext;

    public static EchoBaseApplicationContext getEchoBaseApplicationContext() {
        if (applicationContext == null) {
            applicationContext = EchoBaseApplicationContext.getApplicationContext(
                    getActionContext());
        }
        return applicationContext;
    }

    public static String getApplicationVersion() {
        return getEchoBaseApplicationContext().getConfiguration().getApplicationVersion().toString();
    }

    protected EchoBaseConfiguration getConfiguration() {
        return getEchoBaseApplicationContext().getConfiguration();
    }

    /** EchoBase User session. */
    private transient EchoBaseSession echoBaseSession;

    /**
     * Service context used to access and create services.
     *
     * @since 1.0
     */
    private transient EchoBaseServiceContext serviceContext;

    /**
     * Parameters injected if you put {@link ParameterAware} contract on your
     * action.
     *
     * @see ParameterAware
     * @since 1.1
     */
    private transient Map<String, String[]> parameters;

    /**
     * Session injected if you put {@link SessionAware} contract on your action.
     *
     * @see SessionAware
     * @since 1.1
     */
    private transient Map<String, Object> session;

    public EchoBaseSession getEchoBaseSession() {
        if (echoBaseSession == null) {

            // load it from session
            echoBaseSession = EchoBaseSession.getEchoBaseSession(getActionContext());
        }
        return echoBaseSession;
    }

    public EchoBaseServiceContext getServiceContext() {
        if (serviceContext == null) {
            serviceContext = DefaultEchoBaseServiceContext.newContext(
                    getLocale(),
                    EchoBaseInternalDbTransactionFilter.getTransaction(getActionContext()),
                    EchoBaseWorkingDbTransactionFilter.getTransaction(getActionContext()),
                    getConfiguration(),
                    getEchoBaseApplicationContext().getDbMeta()
            );
        }
        return serviceContext;
    }

    /**
     * Sub-classes should use this method to easily get a service instance.
     *
     * @param serviceClass The type of service to instantiate
     * @return A newly created service of the expected type with necessary data set
     */
    @Override
    public <E extends EchoBaseService> E getService(Class<E> serviceClass) {
        E service = getServiceContext().getService(serviceClass);
        return service;
    }

    public boolean isAdmin() {
        boolean result = getEchoBaseSession().isAdmin();
        return result;
    }

    public boolean isWorkingDbSelected() {
        boolean result = getEchoBaseSession().isWorkingDbSelected();
        return result;
    }

    public String formatDate(Date date) {
        String result = dateFormat.format(date);
        return result;
    }

    public String formatMonth(Date date) {
        String result = monthFormat.format(date);
        return result;
    }

    public void setParameters(Map<String, String[]> parameters) {
        this.parameters = parameters;
    }

    public void setSession(Map<String, Object> session) {
        this.session = session;
    }

    public void addFlashMessage(String message) {
        getEchoBaseSession().addMessage(EchoBaseSession.SESSION_TOKEN_MESSAGES, message);
    }

    public void addFlashError(String message) {
        getEchoBaseSession().addMessage(EchoBaseSession.SESSION_TOKEN_ERRORS, message);
    }

    public void addFlashWarning(String message) {
        getEchoBaseSession().addMessage(EchoBaseSession.SESSION_TOKEN_WARNINGS, message);
    }

    public Collection<String> getFlashMessages() {
        Collection<String> result = getEchoBaseSession().consumeDynamicSetData(EchoBaseSession.SESSION_TOKEN_MESSAGES);
        return result;
    }

    public Collection<String> getFlashErrors() {
        Collection<String> result = getEchoBaseSession().consumeDynamicSetData(EchoBaseSession.SESSION_TOKEN_ERRORS);
        return result;
    }

    public Collection<String> getFlashWarnings() {
        Collection<String> result = getEchoBaseSession().consumeDynamicSetData(EchoBaseSession.SESSION_TOKEN_WARNINGS);
        return result;
    }

    public boolean hasFlashMessages() {
        Collection<String> result = getEchoBaseSession().getDynamicSetData(EchoBaseSession.SESSION_TOKEN_MESSAGES);
        return CollectionUtils.isNotEmpty(result);
    }

    public boolean hasFlashErrors() {
        Collection<String> result = getEchoBaseSession().getDynamicSetData(EchoBaseSession.SESSION_TOKEN_ERRORS);
        return CollectionUtils.isNotEmpty(result);
    }

    public boolean hasFlashWarnings() {
        Collection<String> result = getEchoBaseSession().getDynamicSetData(EchoBaseSession.SESSION_TOKEN_WARNINGS);
        return CollectionUtils.isNotEmpty(result);
    }

    public void clearFlashMessages() {
        Collection<String> result = getEchoBaseSession().getDynamicSetData(EchoBaseSession.SESSION_TOKEN_MESSAGES);
        if (result != null) {
            result.clear();
        }
    }

    public void clearFlashErrors() {
        Collection<String> result = getEchoBaseSession().getDynamicSetData(EchoBaseSession.SESSION_TOKEN_ERRORS);
        if (result != null) {
            result.clear();
        }
    }

    public String getDocumentation(String page) {
        return getDocumentation(page, null);
    }

    public String getDocumentation(String page, String anchor) {
        Locale locale = getLocale();
        String result = getConfiguration().getDocumentationUrl().toString();
        if (!result.endsWith("/")) {
            result += "/";
        }
        if (Locale.ENGLISH.getLanguage().equals(locale.getLanguage())) {
            result += "en/";
        }
        result += page;
        if (StringUtils.isNotBlank(anchor)) {
            result += "#" + anchor;
        }
        return result;
    }

    public boolean hasAnyErrors() {
        boolean result = super.hasErrors();
        return result || hasFlashErrors();
    }

    protected static ActionContext getActionContext() {
        return ActionContext.getContext();
    }

    protected final <E extends Enum<E> & I18nAble> Map<String, String> decorateEnums(E... enumValues) {
        Map<String, String> result = Maps.newLinkedHashMap();
        for (E enumValue : enumValues) {
            result.put(enumValue.name(), _(enumValue.getI18nKey()));
        }
        return result;
    }

    protected final <E extends TopiaEntity> Map<String, String> loadSortAndDecorate(Class<E> beanType) {

        EchoBaseServiceSupport service = getService(EchoBaseServiceSupport.class);
        List<E> beans = service.getEntities(beanType);

        DecoratorService decoratorService = getService(DecoratorService.class);
        return decoratorService.sortAndDecorate(getLocale(), beans, null);
    }


    protected Map<String, String[]> getParameters() {
        Preconditions.checkNotNull(
                parameters,
                "You must put ParameterAware contract on your action " +
                getClass().getName());
        return parameters;
    }

    protected Map<String, Object> getSession() {
        Preconditions.checkNotNull(
                session,
                "You must put SessionAware contract on your action " +
                getClass().getName());
        return session;
    }
}
