/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: EchoBaseApplicationListener.java 343 2012-03-09 23:47:04Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/echobase/tags/echobase-0.3/echobase-ui/src/main/java/fr/ifremer/echobase/ui/EchoBaseApplicationListener.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui;

import fr.ifremer.echobase.EchoBaseConfiguration;
import fr.ifremer.echobase.EchoBaseTechnicalException;
import fr.ifremer.echobase.EchoBaseTopiaRootContextFactory;
import fr.ifremer.echobase.entities.EchoBaseUser;
import fr.ifremer.echobase.entities.EchoBaseUserImpl;
import fr.ifremer.echobase.entities.meta.DbMeta;
import fr.ifremer.echobase.services.DefaultEchoBaseServiceContext;
import fr.ifremer.echobase.services.EchoBaseServiceContext;
import fr.ifremer.echobase.services.EchoBaseServiceFactory;
import fr.ifremer.echobase.services.UserService;
import fr.ifremer.echobase.ui.actions.EchoBaseActionSupport;
import fr.ird.converter.FloatConverter;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaUtil;
import org.nuiton.util.converter.ConverterUtil;

import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import java.util.Date;
import java.util.List;
import java.util.Locale;

/**
 * To listen start or end of the application.
 * <p/>
 * On start we will load the configuration and check connection to internal
 * database, creates schema and create an admin user in none found in database.
 * <p/>
 * On stop, just release the application configuration.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EchoBaseApplicationListener implements ServletContextListener {

    /** Logger. */
    protected static final Log log =
            LogFactory.getLog(EchoBaseApplicationListener.class);

    private TopiaContext rootContext;

    @Override
    public void contextInitialized(ServletContextEvent sce) {

        if (log.isInfoEnabled()) {
            log.info("Application starting at " + new Date() + "...");
        }

        // init I18n
        DefaultI18nInitializer i18nInitializer =
                new DefaultI18nInitializer("echobase-i18n");
        i18nInitializer.setMissingKeyReturnNull(true);
        I18n.init(i18nInitializer, Locale.getDefault());

        EchoBaseApplicationContext applicationContext = new EchoBaseApplicationContext();
        sce.getServletContext().setAttribute(EchoBaseActionSupport.APPLICATION_CONTEXT_PARAMETER, applicationContext);

        // initialize configuration
        EchoBaseConfiguration configuration = new EchoBaseConfiguration();
        applicationContext.setConfiguration(configuration);

        if (log.isInfoEnabled()) {
            log.info("Initializing RootContext...");
        }
        EchoBaseTopiaRootContextFactory factory =
                new EchoBaseTopiaRootContextFactory();
        rootContext = factory.newDatabaseFromConfig(configuration);
        applicationContext.setRootContext(rootContext);
        DbMeta dbMeta = DbMeta.newDbMeta();

        applicationContext.setDbMeta(dbMeta);

        // register our not locale dependant converter
        Converter converter = ConverterUtil.getConverter(Float.class);
        if (converter != null) {
            ConvertUtils.deregister(Float.class);
        }
        ConvertUtils.register(new FloatConverter(), Float.class);

        // init database (and create minimal admin user if required)
        try {
            boolean schemaExist = TopiaUtil.isSchemaExist(
                    rootContext,
                    EchoBaseUserImpl.class.getName()
            );
            if (!schemaExist ||
                configuration.getOptionAsBoolean(EchoBaseConfiguration.OPTION_UPDATE_SCHEMA)) {

                if (schemaExist) {
                    if (log.isInfoEnabled()) {
                        log.info("Will update schema...");
                    }
                    rootContext.updateSchema();
                } else {
                    if (log.isInfoEnabled()) {
                        log.info("Will create schema...");
                    }
                    rootContext.createSchema();
                }
            }

            createAdminUser(applicationContext);
        } catch (TopiaException e) {
            throw new EchoBaseTechnicalException("Could not init db", e);
        }
    }

    @Override
    public void contextDestroyed(ServletContextEvent sce) {

        if (log.isInfoEnabled()) {
            log.info("Application is ending at " + new Date() + "...");
        }
        if (rootContext != null) {
            if (log.isInfoEnabled()) {
                log.info("Shuting down RootContext...");
            }
            if (!rootContext.isClosed()) {
                try {
                    rootContext.closeContext();
                } catch (TopiaException te) {
                    if (log.isErrorEnabled()) {
                        log.error("Could not close rootContext", te);
                    }
                }
            }
        }
    }

    /**
     * Creates the adminsitrator ({@code admin/admin}) on the database.
     *
     * @param applicationContext application context
     * @throws TopiaException if could not create the user.
     */
    protected void createAdminUser(EchoBaseApplicationContext applicationContext) throws TopiaException {

        EchoBaseConfiguration configuration =
                applicationContext.getConfiguration();

        EchoBaseServiceFactory serviceFactory =
                new EchoBaseServiceFactory();
        TopiaContext transaction = rootContext.beginTransaction();

        try {
            EchoBaseServiceContext serviceContext = DefaultEchoBaseServiceContext.newContext(
                    Locale.getDefault(),
                    transaction,
                    configuration,
                    applicationContext.getDbMeta(),
                    serviceFactory
            );

            UserService service =
                    serviceFactory.newService(UserService.class, serviceContext);

            List<EchoBaseUser> users = service.getUsers();

            if (CollectionUtils.isEmpty(users)) {

                // no users in database create the admin user
                if (log.isInfoEnabled()) {
                    log.info("No user in database, will create default " +
                             "admin user (password admin).");
                }

                service.createDefaultUsers();
            }
        } finally {
            transaction.closeContext();
        }
    }

    /**
     * Try to close the given transaction.
     *
     * @param tx the transaction to close
     * @throws TopiaException if could not close the transaction
     */
    protected void closeTransaction(TopiaContext tx) throws TopiaException {
        if (tx != null && !tx.isClosed()) {
            tx.closeContext();
        }
    }

}
