/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.color.awt;

import java.awt.color.ColorSpace;
import java.awt.image.ColorModel;

import de.intarsys.pdf.pd.PDCSDeviceCMYK;
import de.intarsys.pdf.pd.PDColorSpace;
import de.intarsys.pdf.pd.PDImage;

/**
 * 
 */
abstract public class AwtColorSpace {

	static class FallbackColorSpace extends ColorSpace {
		private ColorSpace cmykColorSpace;
		private ColorSpace grayColorSpace;
		private ColorSpace rgbColorSpace;

		FallbackColorSpace() {
			// as good as any
			super(ColorSpace.TYPE_GRAY, 1);
		}

		protected float[] average(float[] values) {
			float sum;

			if (values.length == 1) {
				return values;
			}
			sum = 0;
			for (float value : values) {
				sum += value;
			}
			return new float[] { sum / values.length };
		}

		@Override
		public float[] fromCIEXYZ(float[] colorvalue) {
			throw new UnsupportedOperationException();
		}

		@Override
		public float[] fromRGB(float[] rgbvalue) {
			throw new UnsupportedOperationException();
		}

		protected ColorSpace getCmykColorSpace() {
			if (cmykColorSpace == null) {
				cmykColorSpace = AwtColorSpaceFactory.get().createColorSpace(
						PDCSDeviceCMYK.SINGLETON).getColorSpace();
			}
			return cmykColorSpace;

		}

		protected ColorSpace getGrayColorSpace() {
			if (grayColorSpace == null) {
				grayColorSpace = ColorSpace.getInstance(ColorSpace.CS_GRAY);
			}
			return grayColorSpace;

		}

		protected ColorSpace getRgbColorSpace() {
			if (rgbColorSpace == null) {
				rgbColorSpace = ColorSpace.getInstance(ColorSpace.CS_GRAY);
			}
			return rgbColorSpace;

		}

		@Override
		public float[] toCIEXYZ(float[] colorvalue) {
			if (colorvalue.length == 3) {
				return getRgbColorSpace().toCIEXYZ(colorvalue);
			}
			if (colorvalue.length == 4) {
				return getCmykColorSpace().toCIEXYZ(colorvalue);
			}
			return getGrayColorSpace().toCIEXYZ(average(colorvalue));
		}

		@Override
		public float[] toRGB(float[] colorvalue) {
			if (colorvalue.length == 3) {
				return getRgbColorSpace().toRGB(colorvalue);
			}
			if (colorvalue.length == 4) {
				return getCmykColorSpace().toRGB(colorvalue);
			}
			return getGrayColorSpace().toRGB(average(colorvalue));
		}
	}

	public static final ColorSpace FALLBACK = new FallbackColorSpace();

	static protected AwtColorSpace getColorSpace(PDColorSpace colorSpace) {
		return AwtColorSpaceFactory.get().createColorSpace(colorSpace);
	}

	private ColorSpace colorSpace;
	private PDColorSpace pdColorSpace;

	/**
	 * 
	 */
	public AwtColorSpace(PDColorSpace pdColorSpace) {
		this.pdColorSpace = pdColorSpace;
	}

	abstract protected ColorSpace createColorSpace();

	/**
	 * return a corresponding java color model
	 * 
	 * @return a corresponding java color model
	 */
	public abstract ColorModel getColorModel(PDImage pdImage);

	/**
	 * return a java awt equivalent of this color space
	 * 
	 * @return return a java awt equivalent of this color space
	 */
	final public ColorSpace getColorSpace() {
		if (colorSpace == null) {
			colorSpace = createColorSpace();
		}
		return colorSpace;
	}

	public int getNumComponents() {
		return getColorSpace().getNumComponents();
	}

	public PDColorSpace getPDColorSpace() {
		return pdColorSpace;
	}

	/**
	 * The color values to be used in the RGB color space.
	 * <p>
	 * This MAY be different from using the AWT color space to convert the
	 * values, as an AWT color space may not exist (for example with Separation
	 * color space).
	 * 
	 * @param values
	 *            The color values in this color space.
	 * 
	 * @return The color values in RGB color space.
	 */
	public float[] getRGB(float[] values) {
		return getColorSpace().toRGB(values);
	}
}
