/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.DateUtils;
import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.FacetTopic;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.Wikitty;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.WikittyService;
import org.sharengo.wikitty.WikittyUtil;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;
import org.sharengo.wikitty.search.Like.SearchAs;

import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.QueryMakerImpl;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.dto.VradiCartographyDTO;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.dto.VradiQueryBean;
import com.jurismarches.vradi.services.search.CompareFilter;
import com.jurismarches.vradi.services.search.Filter;
import com.jurismarches.vradi.services.search.FilterList;
import com.jurismarches.vradi.services.search.RangeFilter;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;
import com.jurismarches.vradi.services.search.VradiQueryParser;

/**
 * Class containing the methods to manage the form research
 *
 * @author schorlet
 * @date 2010-01-29 12:40:26
 * @version $Revision: 852 $ $Date: 2010-05-06 10:29:21 +0200 (jeu., 06 mai 2010) $
 */
public class SearchManager {
    private static final Log log = LogFactory.getLog(SearchManager.class);

    private final WikittyProxy proxy;
    private final ThesaurusManager thesaurusManager;
    
    private static final DateFormat frenchDateFormat = DateFormat.getDateInstance(
            DateFormat.SHORT, Locale.FRANCE);
    
    private static final DateFormat numericDateFormat = new SimpleDateFormat("yyyyMMdd", Locale.FRANCE);
    
    /**
     * Match dd/mm/yyyy, requiring leading zeros.
     * 
     * Match the pattern but not the validity of the date, so that <code>01/01/0000</code> will match.
     */
    private static final Pattern frenchDateFormatPattern =
            Pattern.compile("^(3[01]|[12][0-9]|0[1-9])/(1[0-2]|0[1-9])/[0-9]{4}$");
    
    /**
     * Match yyyymmdd, requiring leading zeros.
     */
    private static final Pattern numericDateFormatPattern =
            Pattern.compile("^[0-9]{4}(1[0-2]|0[1-9])(3[01]|[12][0-9]|0[1-9])$");
    
    /**
     * Match UUID pattern, eg. 32 hexadecimal digits, displayed in 5 groups separated by hyphens,
     * in the form 8-4-4-4-12 for a total of 36 characters.
     */
    private static final Pattern uuidPattern =
            Pattern.compile("^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$");
    
    /**
     * Match yyyy-mm-ddUUID pattern.
     */
    private static final Pattern formIdPattern =
            Pattern.compile("^[0-9]{4}-(1[0-2]|0[1-9])-(3[01]|[12][0-9]|0[1-9])([a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12})$");
    
    
    public static final String ALIAS_THESAURUS_1 = "thesaurus";
    public static final String ALIAS_THESAURUS_2 = "descripteurs";
    public static final String ALIAS_LAST_MODIFIER = "modificateur";
    public static final String ALIAS_LAST_STATUS_MODIFIER = "modificateur.status";
    
    public SearchManager(WikittyProxy proxy, ThesaurusManager thesaurusManager) {
        this.proxy = proxy;
        this.thesaurusManager = thesaurusManager;
    }

    public VradiFormPageDTO findForms(String query, final VradiFormPageDTO formPageDTO)
            throws UnsupportedQueryException {
        return findForms(query, null, null, null, null, null, null, formPageDTO);
    }

    public VradiFormPageDTO findForms(String query, final VradiFormPageDTO formPageDTO, Date fromDate, String statusId)
            throws UnsupportedQueryException {
        String[] statusIds = new String[1];
        statusIds[0] = statusId;
        return findForms(query, null, null, fromDate, new Date(), null, statusIds, formPageDTO);
    }
    
    public VradiFormPageDTO findForms(String query, WikittyExtension extension,
            String dateType, Date beginDate, Date endDate,
            List<String>[] thesaurus, String[] statusIds,
            final VradiFormPageDTO formPageDTO) throws UnsupportedQueryException {

        if (log.isDebugEnabled()) {
            log.debug(String.format(
                    "findForms(query:%s, extension:%s, dateType:%s, beginDate:%tc, endDate:%tc)",
                    query, extension, dateType, beginDate, endDate));
        }

        Search search = createSearch(query, extension, dateType, beginDate, endDate,
                thesaurus, statusIds);
        
        Criteria criteria = search.criteria();
        //sets the pagination parameters
        int firstIndex = (formPageDTO.getPageToShow() - 1) * formPageDTO.getNbFormsToShow();
        int lastIndex = formPageDTO.getPageToShow() * formPageDTO.getNbFormsToShow() - 1;

        if (firstIndex >= 0 && lastIndex > 0 && lastIndex > firstIndex) {
            criteria.setFirstIndex(firstIndex).setEndIndex(lastIndex);
        }
        
        if (formPageDTO.getFieldToSort() != null) {
            if (!formPageDTO.isAscending()) {
                criteria.addSortDescending(formPageDTO.getFieldToSort());

            } else {
                criteria.addSortAscending(formPageDTO.getFieldToSort());
            }
        }

        criteria.addSortAscending(Form.FQ_FIELD_ID);

        //finds the forms
        PagedResult<Form> queryResult = proxy.findAllByCriteria(Form.class, criteria);
        List<Form> result = new ArrayList<Form>(queryResult.getAll());

        if (log.isTraceEnabled()) {
            for (Form form : result) {
                log.trace("found: " + form.toString());
            }
        }

        VradiFormPageDTO formPageResult = new VradiFormPageDTO(result,
                queryResult.getNumFound(), formPageDTO.getPageToShow(),
                formPageDTO.getNbFormsToShow());
        
        return formPageResult;
    }

    public VradiCartographyDTO getThesaurusCartography(String query,
            WikittyExtension extension, String dateType, Date beginDate,
            Date endDate, List<String>[] thesaurus, String[] statusIds)
            throws VradiException, UnsupportedQueryException {
        
        if (log.isDebugEnabled()) {
            log.debug("getThesaurusCartography()");
        }

        // count forms indexed by thesaurus nodes
        Map<TreeNodeImpl, Integer> cartography = new HashMap<TreeNodeImpl, Integer>();
        // thesaurus nodes indexed by id
        Map<String, TreeNodeImpl> nodes = new HashMap<String, TreeNodeImpl>();
        
        // getting all thesaurus nodes
        List<TreeNodeImpl> treeNodes = thesaurusManager.getAllThesaurus();
        TreeNodeImpl rootThesaurus = thesaurusManager.getRootThesaurus();
        
        // create search
        Search search = createSearch(query, extension, dateType, beginDate, endDate, thesaurus, statusIds);
        Criteria criteria = search.criteria();
        
        // add facet fields on search
        criteria.addFacetField(TreeNode.EXT_TREENODE + "." + rootThesaurus.getWikittyId());
        for (TreeNodeImpl treeNode : treeNodes) {
            criteria.addFacetField(TreeNode.EXT_TREENODE + "." + treeNode.getWikittyId());
            
            cartography.put(treeNode, 0);
            nodes.put(treeNode.getWikittyId(), treeNode);
        }
        
        // execute search
        PagedResult<Form> results = proxy.findAllByCriteria(Form.class, criteria);
        
        // compute facets
        Collection<String> facetNames = results.getFacetNames();
        for (String facetName : facetNames) {
            List<FacetTopic> topics = results.getTopic(facetName);
            
            for (FacetTopic topic : topics) {
                TreeNodeImpl treeNode = nodes.get(topic.getTopicName());
                if (treeNode == null) {
                    continue;
                }
                cartography.put(treeNode, topic.getCount());
            }
        }
        
        VradiCartographyDTO cartographyDTO = new VradiCartographyDTO(cartography, results.getAll());
        return cartographyDTO;
    }
    
    public Map<QueryMaker, List<VradiQueryBean>> findQueriesReturningForm(Form form)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("findQueriesReturningForm(form)");
        }
        
        Map<QueryMaker, List<VradiQueryBean>> results = new HashMap<QueryMaker, List<VradiQueryBean>>();
        
        if (form == null) {
            return results;
        }
        
        // find QueryMaker which do have queries defined
        List<QueryMaker> queryMakers = findQueryMakersWithQueries();

        for (QueryMaker queryMaker : queryMakers) {
            Set<String> queries = queryMaker.getQueries();
            // just to make sure ...
            if (queries == null || queries.isEmpty()) {
                continue;
            }
            
            QueryMaker realQueryMaker = ClientManager.castAsRealQueryMaker((QueryMakerImpl) queryMaker);
            if (realQueryMaker == null) {
                continue;
            }
            
            for (String query : queries) {
                try {
                    VradiQueryBean bean = new VradiQueryBean(query, queryMaker.getWikittyId());
                    String realQuery = bean.getQuery();
                    FilterList filter = VradiQueryParser.parse(realQuery);
                    
                    Search search = Search.query();
                    search.eq(Form.FQ_FIELD_ID, form.getId());
                    buildSearch(filter, search);
    
                    Criteria criteria = search.criteria();
                    PagedResult<Form> forms = proxy.findAllByCriteria(Form.class, criteria);
                    
                    if (forms.getNumFound() > 0) {
                        if (results.containsKey(realQueryMaker)) {
                            results.get(realQueryMaker).add(bean);
                            
                        } else {
                            List<VradiQueryBean> list = new ArrayList<VradiQueryBean>();
                            list.add(bean);
                            results.put(realQueryMaker, list);
                        }
                    }
                    
                } catch (Exception e) {
                    log.warn(e.getMessage(), e);
                }
            }
        }

        return results;
    }
    
    public List<QueryMaker> findQueryMakersWithQueries() {
        // find QueryMaker which do have queries defined
        Criteria criteria = Search.query().bw(QueryMaker.FQ_FIELD_QUERIES, "*", "*").criteria();
        PagedResult<QueryMaker> pagedResult = proxy.findAllByCriteria(QueryMaker.class, criteria);
        List<QueryMaker> all = pagedResult.getAll();
        return all;
    }
    
    Search createSearch(String query, WikittyExtension extension,
            String dateType, Date beginDate, Date endDate,
            List<String>[] thesaurus, String[] statusIds)
            throws UnsupportedQueryException {

        FilterList filter = VradiQueryParser.parse(query);
        Search search = Search.query();
        buildSearch(filter, search);

        //add the extension in the criteria
        if (extension != null) {
            search.eq(Element.ELT_EXTENSION, extension.getName());
        }

        //add the date in the criteria
        if (dateType != null) {
            if (beginDate != null && endDate != null) {
                String beginString = WikittyUtil.solrDateFormat.format(beginDate);
                String endString = WikittyUtil.solrDateFormat.format(endDate);
                search.bw(dateType, beginString, endString);
            }
        }

        //add the thesaurus in the criteria
        if (thesaurus != null) {
            for (int i = 0; i < thesaurus.length; i++) {
                
                if (thesaurus[i] != null) {
                    Search subSearch = search.or();
                    for (String th : thesaurus[i]) {
                        subSearch.eq(Form.FQ_FIELD_THESAURUS, th);
                    }
                }
            }
        }

        //add the status in the criteria
        if(statusIds != null && statusIds.length > 0) {
            Search subSearch = search.or();
            for (String statusId : statusIds) {
                subSearch.eq(Form.FQ_FIELD_STATUS, statusId);
            }
        }

        return search;
    }
    
    void buildSearch(FilterList list, Search search) {
        Search subSearch = null;
        FilterList.Operator operator = list.getOperator();

        if (operator == FilterList.Operator.MUST_PASS_ONE) {
            subSearch = search.or();
            
        } else if (operator == FilterList.Operator.MUST_PASS_ALL) {
            subSearch = search.and();
            
        } else if (operator == FilterList.Operator.MUST_NOT_PASS) {
            subSearch = search.not();
        }
        
        List<Filter> filters = list.getFilters();
        for (Filter filter : filters) {
            
            if (filter instanceof FilterList) {
                buildSearch((FilterList) filter, subSearch);
        
            } else if (filter instanceof RangeFilter) {
                buildRangeSearch(operator, (RangeFilter) filter, subSearch);
                
            } else if (filter instanceof CompareFilter) {
                buildCompareSearch(operator, (CompareFilter) filter, subSearch);
            }
            
        }
    }
    
    void buildDescripteurSearch(Search search, String value) {
        Criteria criteria = Search.query()
            .eq(TreeNode.FQ_FIELD_NAME, value)
            .eq(Element.ELT_EXTENSION, TreeNodeImpl.EXT_TREENODE)
            .criteria();

        WikittyService wikittyService = proxy.getWikittyService();
        PagedResult<Wikitty> queryResult = wikittyService.findAllByCriteria(criteria);
        
        if (queryResult.getNumFound() > 0) {
            Search subSearch = search.or();

            List<Wikitty> result = queryResult.getAll();
            for (Wikitty wikitty : result) {
                subSearch.eq(Form.FQ_FIELD_THESAURUS, wikitty.getId());
            }
        }
        
    }
    
    void buildRangeSearch(FilterList.Operator operator, RangeFilter rangeFilter,
            Search search) {
        
        String name = rangeFilter.getName();
        String lowerValue = rangeFilter.getLowerValue();
        String upperValue = rangeFilter.getUpperValue();
        
        Search subSearch = search.or();
        
        // date search
        if (rangeFilter.match(numericDateFormatPattern)) {
            try {
                Date lowerTime = numericDateFormat.parse(lowerValue);
                lowerTime = DateUtils.setMinTimeOfDay(lowerTime);
                String lowerTimeString = WikittyUtil.solrDateFormat.format(lowerTime);
                
                Date upperTime = numericDateFormat.parse(upperValue);
                upperTime = DateUtils.setMaxTimeOfDay(upperTime);
                String upperTimeString = WikittyUtil.solrDateFormat.format(upperTime);
                
                subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                        + name + Criteria.SEPARATOR
                        + Element.ElementType.DATE, lowerTimeString, upperTimeString);
                
            } catch (ParseException e) {
                log.warn(lowerValue + " OR " + upperValue + " cannot be a date.");
            }
        
        // number search
        } else if (rangeFilter.isNumber()) {
            subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR
                    + Element.ElementType.NUMERIC, lowerValue, upperValue);
        }
        
        // default to string search
        subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, lowerValue, upperValue);
    }
    
    void buildCompareSearch(FilterList.Operator operator, CompareFilter compareFilter,
            Search search) {
        String name = compareFilter.getName();
        String value = compareFilter.getValue();
     
        if (VradiQueryParser.DEFAULT_FIELD.equals(name)) {
            search.keyword(value);
            return;
            
        } else if (ALIAS_THESAURUS_1.equals(name) || ALIAS_THESAURUS_2.equals(name)) {
            buildDescripteurSearch(search, value);
            return;
            
        } else if (ALIAS_LAST_MODIFIER.equals(name)) {
            name = ModificationTag.FIELD_LAST_MODIFIER;
            
        } else if (ALIAS_LAST_STATUS_MODIFIER.equals(name)) {
            name = ModificationTag.FIELD_LAST_STATUS_MODIFIER;
        }
        
        Search subSearch = search.or();
        
        if (operator == FilterList.Operator.MUST_NOT_PASS) {
            // date search
            if (compareFilter.match(frenchDateFormatPattern)) {
                try {
                    Date time = frenchDateFormat.parse(value);
                    String timeString = WikittyUtil.solrDateFormat.format(time.getTime());
                    
                    subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + name + Criteria.SEPARATOR
                            + Element.ElementType.DATE, timeString);
                    
                } catch (ParseException e) {
                    log.warn(value + " cannot be a date.");
                }
            }
            
        } else {
            // date search
            if (compareFilter.match(frenchDateFormatPattern)) {
                try {
                    Date time = frenchDateFormat.parse(value);
        
                    Date beginDate = DateUtils.setMinTimeOfDay(time);
                    Date endDate = DateUtils.setMaxTimeOfDay(time);
                    
                    String beginString = WikittyUtil.solrDateFormat.format(beginDate);
                    String endString = WikittyUtil.solrDateFormat.format(endDate);
                    
                    subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + name + Criteria.SEPARATOR
                            + Element.ElementType.DATE, beginString, endString);
                    
                } catch (ParseException e) {
                    log.warn(value + " cannot be a date.");
                }
            }
        }
        
        // string search
        subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, value);
        
        // text search
        if (!compareFilter.isPhrase()) {
            subSearch.like(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR + Element.ElementType.STRING,
                    value, SearchAs.AsText);
        }

        // boolean search
        if (compareFilter.isBoolean()) {
            subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR
                    + Element.ElementType.BOOLEAN, value);

        // number search
        } else if (compareFilter.isNumber()) {
            subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR
                    + Element.ElementType.NUMERIC, value);
            
        // extension id search (eq)
        } else if (compareFilter.match(formIdPattern)) {
            subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, value);
            
        // extension id search (ew)
        } else if (compareFilter.match(uuidPattern)) {
            subSearch.ew(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, value);
        }
    }
}
