/*
 * *##%
z * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.dto;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.Wikitty;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.entities.Thesaurus;

/**
 * User: sletellier
 * Date: 30 dec. 2009
 * Time: 12:23:27
 */
public class VradiThesaurusDTO implements VradiDTO<TreeNodeImpl> {

    private static final long serialVersionUID = 1L;

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(VradiThesaurusDTO.class);

    protected PropertyChangeSupport propertyChange = new PropertyChangeSupport(this);

    protected boolean toSave = false;
    protected boolean toCreate = false;
    protected boolean toDelete = false;
    protected String name;
    protected String wikittyId;
    protected int formsForThesaurus = 0;
    protected String comment;
    protected int order = 0;

    protected VradiThesaurusDTO parentThesaurus;

    protected String tagsAsString;

    protected List<String> tags = new ArrayList<String>();

    protected Map<String, VradiThesaurusDTO> children = new HashMap<String, VradiThesaurusDTO>();

    public VradiThesaurusDTO() {
    }

    public boolean isRoot(){
        return parentThesaurus == null;
    }

    public int getOrder() {
        return order;
    }

    public void setOrder(int order) {
        int orderOld = this.order;
        this.order = order;
        propertyChange.firePropertyChange("order", orderOld, order);
    }

    public boolean isToDelete() {
        return toDelete;
    }

    public void setToDelete(boolean toDelete) {
        boolean toDeleteOld = this.toDelete;
        this.toDelete = toDelete;
        for (VradiThesaurusDTO child : getChildren()) {
            child.setToDelete(toDelete);
        }
        propertyChange.firePropertyChange("toDelete", toDeleteOld, toDelete);
    }

    public boolean isToCreate() {
        return toCreate;
    }

    public void setToCreate(boolean toCreate) {
        boolean old = this.toCreate;
        this.toCreate = toCreate;
        propertyChange.firePropertyChange("toCreate", old, toCreate);
    }

    public boolean isToSave() {
        return toSave;
    }

    public void setToSave(boolean toSave) {
        boolean old = this.toSave;
        this.toSave = toSave;
        propertyChange.firePropertyChange("toSave", old, toSave);
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        String old = this.name;
        this.name = name;
        propertyChange.firePropertyChange("name", old, name);
    }

    // To decorate this
    public String getOrderName() {
        String orderName = name;
        if (order != 0){
            orderName = order + " - " + name;
        }
        return orderName;
    }

    @Override
    public String getWikittyId() {
        return wikittyId;
    }

    @Override
    public void setWikittyId(String wikittyId) {
        String old = this.wikittyId;
        this.wikittyId = wikittyId;
        propertyChange.firePropertyChange("wikittyId", old, wikittyId);
    }

    public VradiThesaurusDTO getParentThesaurus() {
        return parentThesaurus;
    }

    public void setParentThesaurus(VradiThesaurusDTO parentThesaurus) {
        VradiThesaurusDTO old = this.parentThesaurus;
        this.parentThesaurus = parentThesaurus;
        propertyChange.firePropertyChange("parentThesaurus", old, parentThesaurus);
    }

    public String getTagsAsString() {
        if (tagsAsString == null || tagsAsString.isEmpty()){
            initTagsAsString();
        }
        return tagsAsString;
    }

    public void initTagsAsString() {
        // Tags as a string
        String tagAsString = "";
        if (tags != null) {
            for (String tag : tags) {
                tagAsString += tag + ", ";
            }
            if (!tagAsString.isEmpty()) {
                tagAsString = tagAsString.substring(0, tagAsString.length() - 2);
            }
        }
        setTagsAsString(tagAsString);
    }

    public void setTagsAsString(String values) {
        String old = this.tagsAsString;
        tagsAsString = values;
        propertyChange.firePropertyChange("tagsAsString", old, tagsAsString);
    }

    protected List<String> getTags() {
        return tags;
    }

    protected void setTags(List<String> tags) {
        this.tags = tags;
    }

    protected void addTag(String tag) {
        this.tags.add(tag);
    }

    public int getFormsForThesaurus() {
        return formsForThesaurus;
    }

    public int decFormsForThesaurus() {
        if (parentThesaurus != null){
            parentThesaurus.decFormsForThesaurus();
        }
        return --formsForThesaurus;
    }

    public int incFormsForThesaurus() {
        if (parentThesaurus != null){
            parentThesaurus.incFormsForThesaurus();
        }
        return ++formsForThesaurus;
    }

    public void setFormsForThesaurus(int formsForThesaurus) {
        this.formsForThesaurus = formsForThesaurus;
    }

    public String getComment() {
        return comment;
    }

    public void setComment(String comment) {
        String old = this.comment;
        this.comment = comment;
        propertyChange.firePropertyChange("comment", old, comment);
    }

    public List<VradiThesaurusDTO> getChildren() {
        // Get children
        Collection<VradiThesaurusDTO> children = this.children.values();
        List<VradiThesaurusDTO> result = new ArrayList<VradiThesaurusDTO>(children);

        // Sort children by name
//        Collections.sort(result, THESAURUS_COMPARATOR);

        return result;
    }

    public void addChild(VradiThesaurusDTO element) {
        Map<String, VradiThesaurusDTO> old = this.children;
        children.put(element.getWikittyId(), element);
        propertyChange.firePropertyChange("children", old, children);
    }

    public void clearChildren() {
        Map<String, VradiThesaurusDTO> old = this.children;
        children.clear();
        propertyChange.firePropertyChange("children", old, children);
    }

    public void removeChild(VradiThesaurusDTO child) {
        Map<String, VradiThesaurusDTO> old = this.children;
        children.remove(child.getWikittyId());
        propertyChange.firePropertyChange("children", old, children);
    }

    public List<String> getChildrenRecursif() {
        List<String> result = new ArrayList<String>();
        for (VradiThesaurusDTO child : getChildren()) {
            result.add(child.getWikittyId());
            result.addAll(child.getChildrenRecursif());
        }
        return result;
    }

    public boolean addChildRecursif(VradiThesaurusDTO child) {
        VradiThesaurusDTO p = child.getParentThesaurus();
        if (getWikittyId().equals(p.getWikittyId())) {
            addChild(child);
            return true;
        } else {
            for (VradiThesaurusDTO c : getChildren()) {
                return c.addChildRecursif(child);
            }
        }
        return false;
    }

    public boolean removeThesaurusRecursivly(VradiThesaurusDTO toRemove) {
        if (children.get(toRemove.getWikittyId()) != null) {
            removeChild(toRemove);
            return true;
        } else {
            for (VradiThesaurusDTO c : getChildren()) {
                return c.removeThesaurusRecursivly(toRemove);
            }
        }
        return false;
    }

    public VradiThesaurusDTO isNameExisting(String name) {
        if (this.name.equalsIgnoreCase(name)){
            return this;
        }
        for (VradiThesaurusDTO child : getChildren()){
            VradiThesaurusDTO result = child.isNameExisting(name);
            if (result != null){
                return result;
            }
        }
        return null;
    }

    public VradiThesaurusDTO findThesaurus(String id) {
        if (wikittyId.equals(id)) {
            return this;
        }
        VradiThesaurusDTO child = children.get(id);
        if (child != null) {
            return child;
        }
        for (VradiThesaurusDTO c : children.values()) {
            VradiThesaurusDTO foundThesaurus = c.findThesaurus(id);
            if(foundThesaurus != null) {
                return foundThesaurus;
            }
        }
        return null;
    }

    // Verify if is child of parent in param
    public boolean isChildOf(String parentId) {
        if (parentThesaurus == null){
            return false;
        }
        if (parentThesaurus.getWikittyId().equals(parentId)){
            return true;
        }
        return parentThesaurus.isChildOf(parentId);
    }

    public String getRecursifName() {
        return getRecursifName(getOrderName());
    }

    protected String getRecursifName(String name) {
        Collection<VradiThesaurusDTO> children = getChildren();
        for (VradiThesaurusDTO child : children) {
            name += ", " + child.getRecursifName();
        }
        return name;
    }

    // Get jxpath
    public String getParentPath(String pathSeparator) {
        if (getParentThesaurus() != null && !getParentThesaurus().getName().equals(
                VradiConstants.ROOT_THESAURUS_NAME)) {
            return getParentThesaurus().getParentPath(pathSeparator) + pathSeparator + getParentThesaurus().getOrderName();
        }
        return "";
    }

    public String getNamePath(String pathSeparator) {
        if (parentThesaurus != null){
            return parentThesaurus.getNamePath(pathSeparator) + pathSeparator + getOrderName();
        } else {
            return "";
        }
    }

    public String getPath(String pathSeparator){
        return getPath(null, pathSeparator);
    }

    public String getPath(String defName, String pathSeparator){
        if (parentThesaurus != null){
            return parentThesaurus.getPath(defName, pathSeparator) + pathSeparator + wikittyId;
        } else {
            if (defName != null){
                return defName;
            }
            return name;
        }
    }

    public String getBuildPath(String pathSeparator){
        if (parentThesaurus != null){
            return parentThesaurus.getBuildPath(pathSeparator) + pathSeparator + "children[@wikittyId=\"" + wikittyId + "\"]";
        } else {
            return "..";
        }
    }

    public int getDepth(){
        if (getParentThesaurus() == null){
            return 0;
        }
        return 1 + getParentThesaurus().getDepth();
    }

    @Override
    public String toString() {
        return getNamePath("/");
    }

// Clone recursively
    @Override
    public VradiThesaurusDTO clone() {
        return clone(false);
    }

    protected VradiThesaurusDTO clone(boolean parentAlreadyCloned) {
        VradiThesaurusDTO clone = new VradiThesaurusDTO();
        clone.setWikittyId(wikittyId);
        clone.setToCreate(toCreate);
        clone.setToSave(toSave);
        clone.setComment(comment);
        clone.setName(name);
        clone.setFormsForThesaurus(formsForThesaurus);
        for (VradiThesaurusDTO child : getChildren()){
            clone.addChild(child.cloneChild(clone));
        }
        if (!parentAlreadyCloned && parentThesaurus != null){
            clone.setParentThesaurus(parentThesaurus.cloneParents(clone));
        }
        clone.setTags(tags);
        clone.setTagsAsString(tagsAsString);
        clone.setOrder(order);
        return clone;
    }

    protected VradiThesaurusDTO cloneParents(VradiThesaurusDTO childClone) {
        VradiThesaurusDTO clone = clone(false);
        clone.addChild(childClone);
        return clone;
    }

    // Clone recursively and set parent
    protected VradiThesaurusDTO cloneChild(VradiThesaurusDTO parent){
        VradiThesaurusDTO clone = clone(true);
        clone.setParentThesaurus(parent);
        return clone;
    }

    @Override
    public void fromWikitty(TreeNodeImpl treeNode) {
        // Attach bean
        setName(treeNode.getName());
        this.setWikittyId(treeNode.getWikittyId());

        // Extentions
        if (treeNode.getExtensionNames().contains(Thesaurus.EXT_THESAURUS)) {
            // Tags
            setTags((List<String>) treeNode.getField(Thesaurus.EXT_THESAURUS, Thesaurus.TAGS_FIELD));

            // Order
            Object orderField = treeNode.getField(Thesaurus.EXT_THESAURUS, Thesaurus.ORDER_FIELD);
            if (orderField != null){
                setOrder(((BigDecimal)orderField).intValue());
            } else {
                setOrder(0);
            }

            // Comment
            setComment((String) treeNode.getField(Thesaurus.EXT_THESAURUS, Thesaurus.COMMENT_FIELD));

            initTagsAsString();
        }
    }

    @Override
    public void toWikitty(TreeNodeImpl treeNodeImpl) {
        // Resore name
        treeNodeImpl.setName(getName());
        // Resore parent
        VradiThesaurusDTO parentThesaurus = getParentThesaurus();
        // If it's root
        if (parentThesaurus != null){
            treeNodeImpl.setParent(parentThesaurus.getWikittyId());
        }
        // Resore extentions
        if (!treeNodeImpl.getExtensionNames().contains(Thesaurus.EXT_THESAURUS)) {
            Wikitty wikitty = treeNodeImpl.getWikitty();
            wikitty.addExtension(Thesaurus.EXTENSION_THESAURUS);
        }

        // Restore order
        treeNodeImpl.setField(Thesaurus.EXT_THESAURUS, Thesaurus.ORDER_FIELD, order);

        // Resore tags from tags as string
        if (tags == null){
            tags = new ArrayList<String>();
        } else {
            tags.clear();
        }
        if (tagsAsString != null) {
            for (String tag : tagsAsString.split(",")) {
                addTag(tag.trim());
            }
        }
        treeNodeImpl.setField(Thesaurus.EXT_THESAURUS, Thesaurus.TAGS_FIELD, tags);
        // Resore comment
        treeNodeImpl.setField(Thesaurus.EXT_THESAURUS, Thesaurus.COMMENT_FIELD, comment);
    }

    @Override
    public void reset() {
        setWikittyId(null);
        setName(null);
        setTags(null);
        setComment(null);
        setToDelete(false);
        setToSave(false);
        setToCreate(false);
        setFormsForThesaurus(0);
        setOrder(0);
        children =  Collections.emptyMap();
    }

    @Override
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        propertyChange.addPropertyChangeListener(listener);
    }

    @Override
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        propertyChange.removePropertyChangeListener(listener);
    }

    @Override
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        propertyChange.addPropertyChangeListener(propertyName, listener);
    }

    @Override
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        propertyChange.removePropertyChangeListener(propertyName, listener);
    }

    /**
     * Compares VradiThesaurusDTO by getOrder() + ":" + getName().
     */
    public static final Comparator<VradiThesaurusDTO> THESAURUS_COMPARATOR = new Comparator<VradiThesaurusDTO>() {

        @Override
        public int compare(VradiThesaurusDTO o1, VradiThesaurusDTO o2) {
            int result = 0;

            if (o1 != null || o2 != null) {
                if ((o1.getOrder() == 0 && o2.getOrder() == 0) || o1.getOrder() == o2.getOrder()){
                    result = o1.getName().compareToIgnoreCase(o2.getName());
                } else if (o1.getOrder() != 0 && o2.getOrder() != 0) {
                    result = (new Integer(o1.getOrder()).compareTo(o2.getOrder()));
                } else if (o1.getOrder() != 0 && o2.getOrder() == 0) {
                    result =  new Integer(o1.getOrder()).compareTo(Integer.MAX_VALUE);
                } else if (o1.getOrder() == 0 && o2.getOrder() != 0) {
                    result =  new Integer(o2.getOrder()).compareTo(Integer.MIN_VALUE);
                }
                if (log.isTraceEnabled()){
                    log.trace("Comparing " + o1.getOrder()
                            + "-" + o1.getName() +
                            " with " + o2.getOrder() +
                            "-" + o2.getName() +
                            " result : " + result);
                }
            }
            return result;
        }
    };
}
