package com.franciaflex.faxtomail.ui.swing.content.reply;

/*
 * #%L
 * FaxToMail :: UI
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2014 Mac-Groupe, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFile;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFileImpl;
import com.franciaflex.faxtomail.persistence.entities.AttachmentImpl;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.services.service.ldap.Contact;
import com.franciaflex.faxtomail.ui.swing.actions.AddAttachmentToReplyAction;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeUIModel;
import com.franciaflex.faxtomail.ui.swing.util.AbstractFaxToMailUIHandler;
import com.franciaflex.faxtomail.ui.swing.util.FaxToMailUIUtil;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.jaxx.application.swing.util.Cancelable;
import org.nuiton.validator.bean.simple.SimpleBeanValidator;

import javax.swing.ComboBoxEditor;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.JTextPane;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import static org.nuiton.i18n.I18n.t;

/**
 * @author Kevin Morin (Code Lutin)
 *
 */
public class ReplyFormUIHandler extends AbstractFaxToMailUIHandler<ReplyFormUIModel, ReplyFormUI> implements Cancelable {

    private static final Log log = LogFactory.getLog(ReplyFormUIHandler.class);

    protected AddAttachmentToReplyAction addAttachmentToReplyAction;

    @Override
    public void beforeInit(ReplyFormUI ui) {
        super.beforeInit(ui);

        addAttachmentToReplyAction =
                getContext().getActionFactory().createLogicAction(this, AddAttachmentToReplyAction.class);

        ReplyFormUIModel model = new ReplyFormUIModel();
        //TODO kmorin 20140813 action ?
        long maxSize = getContext().newServiceContext().getConfigurationService().getEmailMaxSize();
        model.setMaxAttachmentLength(maxSize);

        this.ui.setContextValue(model);
    }

    @Override
    public void afterInit(ReplyFormUI replyFormUI) {
        initUI(ui);

        JTextPane editor = ui.getMessage();
        editor.setCaretPosition(0);

        ReplyFormUIModel model = getModel();
        model.addPropertyChangeListener(ReplyFormUIModel.PROPERTY_ORIGINAL_DEMAND, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                DemandeUIModel demand = (DemandeUIModel) evt.getNewValue();
                ReplyFormUIModel model = (ReplyFormUIModel) evt.getSource();

                List<DemandeUIModel> models = new ArrayList<DemandeUIModel>();
                models.addAll(demand.getGroupedDemandes());

                for (DemandeUIModel demandeUIModel : models) {
                    // remove newline character (cf #6960)
                    String prefix = StringUtils.remove(demandeUIModel.getTitle(), '\n');

                    for (Attachment attachment : demandeUIModel.getAttachment()) {

                        ReplyAttachmentModel replyOriginalAttachmentModel =
                                 new ReplyAttachmentModel(attachment, true, prefix + attachment.getOriginalFileName());
                        model.addAvailableAttachment(replyOriginalAttachmentModel);

                        if (!Strings.isNullOrEmpty(attachment.getEditedFileName())) {
                            ReplyAttachmentModel replyEditedAttachmentModel =
                                    new ReplyAttachmentModel(attachment, false, prefix + attachment.getEditedFileName());
                            model.addAvailableAttachment(replyEditedAttachmentModel);
                        }
                    }
                }

                model.setSubject(t("faxtomail.reply.subject", demand.getSubject()));

                String plainContent = StringUtils.join(demand.getPlainContent(), "\n\n----------------\n\n");
                String quotedReply = t("faxtomail.reply.message",
                                       decorate(demand.getReceptionDate()),
                                       demand.getSender(),
                                       plainContent.replaceAll("\n", "\n> "));
                model.setMessage(quotedReply);

                String recipient = demand.getRecipient();
                // this have to be disable for read only mode (useless)
                if (!model.isReadonly()) {
                    
                    // attachment combo
                    JComboBox<AttachmentFile> addAttachmentFile = ui.getAddAttachmentFile();
                    addAttachmentFile.setModel(SwingUtil.newComboModel(model.getAvailableAttachments().toArray()));
                    addAttachmentFile.setSelectedItem(null);

                    // sender combo
                    MailFolder folder = demand.getMailFolder();
                    List<String> folderReplyAdresses = new ArrayList<String>();
                    List<String> folderReplyDomains = new ArrayList<String>();
                    while (folder != null) {
                        folderReplyAdresses.addAll(folder.getReplyAddresses());
                        folderReplyDomains.addAll(folder.getReplyDomains());
                        folder = folder.getParent();
                    }
                    model.setSenderAllowedDomains(folderReplyDomains);
    
                    List<String> replyToAddresses = Lists.newArrayList();
                    String selectedItem = null;
                    if (!demand.isFax() && recipient != null) {
                        replyToAddresses.add(recipient);
                        model.setFrom(recipient);
                        selectedItem = recipient;
                    }
                    if (folderReplyAdresses != null) {
                        replyToAddresses.addAll(folderReplyAdresses);
                    }
                    Collections.sort(replyToAddresses);

                    DefaultComboBoxModel comboBoxModel = SwingUtil.newComboModel(replyToAddresses.toArray());
                    comboBoxModel.setSelectedItem(selectedItem);
                    ui.getFromComboBox().setModel(comboBoxModel);
                    model.setSenderAllowedAddresses(replyToAddresses);

                    initBeanFilterableComboBox(ui.getContactField(), getContext().getContactCache(), null);

                } else {
                    model.setFrom(recipient);
                }
            }
        });

        model.addPropertyChangeListener(ReplyFormUIModel.PROPERTY_ATTACHMENT, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                updateAttachmentPanel((Collection<ReplyAttachmentModel>) evt.getNewValue());
            }
        });

        model.addPropertyChangeListener(ReplyFormUIModel.PROPERTY_READONLY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                updateAttachmentPanel(getModel().getAttachments());
            }
        });

        JComboBox addAttachmentFile = ui.getAddAttachmentFile();
        addAttachmentFile.setEditor(new FileComboBoxEditor());

        getValidator().addPropertyChangeListener(SimpleBeanValidator.VALID_PROPERTY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                ReplyFormUIModel model = getModel();
                if (log.isDebugEnabled()) {
                    log.debug("Model [" + model +
                                      "] pass to valid state [" +
                                      evt.getNewValue() + "]");
                }
                model.setValid((Boolean) evt.getNewValue());
            }
        });
    }

    @Override
    public void cancel() {
        closeFrame();
    }

    @Override
    public void onCloseUI() {
    }

    @Override
    public SwingValidator<ReplyFormUIModel> getValidator() {
        return getUI().getValidator();
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getMessage();
    }

    @Override
    public Component getTopestUI() {
        return getUI();
    }

    public void removeAttachment(ReplyAttachmentModel attachment) {
        ReplyFormUIModel model = getModel();

        model.removeAttachment(attachment);

        model.addAvailableAttachment(attachment);
        ui.getAddAttachmentFile().addItem(attachment);
    }

    public void addAttachment() {
        ReplyFormUIModel model = getModel();

        JComboBox addAttachmentFile = ui.getAddAttachmentFile();
        ReplyAttachmentModel replyAttachmentModel = null;
        Object selectedItem = addAttachmentFile.getSelectedItem();

        if (ReplyAttachmentModel.class.isAssignableFrom(selectedItem.getClass())) {
            replyAttachmentModel = (ReplyAttachmentModel) selectedItem;
            addAttachmentToReplyAction.setReplyAttachmentModel(replyAttachmentModel);
            getContext().getActionEngine().runActionAndWait(addAttachmentToReplyAction);

        } else if (File.class.isAssignableFrom(selectedItem.getClass())) {
            File file = (File) selectedItem;

            AttachmentFile attachmentFile = new AttachmentFileImpl();
            String fileName = file.getName();
            attachmentFile.setFilename(fileName);

            try {
                attachmentFile.setContent(FileUtils.readFileToByteArray(file));

            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("Error while converting the file " + fileName + " into a byte[]", e);
                }
            }

            Attachment attachment = new AttachmentImpl();
            attachment.setOriginalFile(attachmentFile);
            replyAttachmentModel = new ReplyAttachmentModel(attachment, true, fileName);

        }
        if (replyAttachmentModel != null) {
            model.addAttachment(replyAttachmentModel);

            model.removeAvailableAttachment(replyAttachmentModel);
            ui.getAddAttachmentFile().removeItem(replyAttachmentModel);

            addAttachmentFile.setSelectedItem(null);
        }
    }

    protected void updateAttachmentPanel(Collection<ReplyAttachmentModel> attachments) {
        JPanel attachmentsPanel = ui.getAttachmentsPanel();
        attachmentsPanel.removeAll();
        for (ReplyAttachmentModel attachment : attachments) {
            AttachmentItem item = new AttachmentItem();
            item.setHandler(this);
            item.setReplyAttachmentModel(attachment);
            attachmentsPanel.add(item);
        }
        attachmentsPanel.updateUI();
    }

    public void openLocation() {
        // use last selected file
        ReplyFormUIModel model = getModel();

        File startFile = model.getLastVisitedDirectory();
        JFileChooser fc = new JFileChooser(startFile);

//        fc.setDialogTitle(view.getTitle());

        // used to enable file selection
        fc.setFileSelectionMode(JFileChooser.FILES_ONLY);

        int returnVal = fc.showOpenDialog(ui);
        model.setLastVisitedDirectory(fc.getCurrentDirectory());

        if (returnVal == JFileChooser.APPROVE_OPTION) {

            // get selected to display in ui
            File file = fc.getSelectedFile();

            ui.getAddAttachmentFile().setSelectedItem(file);
        }
    }

    public void openAttachment(ReplyAttachmentModel attachment) {
//TODO kmorin 20140813 action ?
        FaxToMailUIUtil.openFile(getContext(), attachment.getAttachmentFile());

    }

    protected class FileComboBoxEditor implements ComboBoxEditor {

        protected ReplyAttachmentModel oldValue;

        protected JTextField editorComponent = new JTextField();

        @Override
        public Component getEditorComponent() {
            return editorComponent;
        }

        @Override
        public void setItem(Object anObject) {
            String text;
            ReplyAttachmentModel replyAttachmentModel = null;
            if (anObject != null) {
                if (ReplyAttachmentModel.class.isAssignableFrom(anObject.getClass())) {
                    replyAttachmentModel = (ReplyAttachmentModel) anObject;

                } else if (File.class.isAssignableFrom(anObject.getClass())) {
                    File file = (File) anObject;

                    AttachmentFile attachmentFile = new AttachmentFileImpl();
                    String fileName = file.getName();
                    attachmentFile.setFilename(fileName);

                    replyAttachmentModel = new ReplyAttachmentModel(new AttachmentImpl(), true, fileName);
                }
            }

            if (replyAttachmentModel != null)  {
                text = replyAttachmentModel.getLabel();
                oldValue = replyAttachmentModel;

            } else {
                text = "";
            }
            if (! text.equals(editorComponent.getText())) {
                editorComponent.setText(text);
            }
        }

        @Override
        public Object getItem() {
            Object newValue = editorComponent.getText();

            if (oldValue != null)  {
                // The original value is not a string. Should return the value in it's
                // original type.
                if (newValue.equals(oldValue.getLabel())) {
                    return oldValue;

                } else {
                    // Must take the value from the editor and get the value and cast it to the new type.
                    Class<?> cls = oldValue.getClass();
                    try {
                        Method method = cls.getMethod("valueOf", new Class[]{String.class});
                        newValue = method.invoke(oldValue, new Object[] { editorComponent.getText()});
                    } catch (Exception ex) {
                        // Fail silently and return the newValue (a String object)
                    }
                }
            }
            return newValue;
        }

        @Override
        public void selectAll() {
            editorComponent.selectAll();
            editorComponent.requestFocus();
        }

        @Override
        public void addActionListener(ActionListener l) {
            editorComponent.addActionListener(l);
        }

        @Override
        public void removeActionListener(ActionListener l) {
            editorComponent.removeActionListener(l);
        }
    }

    /**
     * Close current dialog ui, and reopen a new one for transfering repons to new recipient.
     */
    public void forward() {
        closeFrame();

        try {
            // display a new ui with a copy of original ui model
            ReplyFormUI dialogContent = new ReplyFormUI(ui);
            ReplyFormUIModel replyModel = dialogContent.getModel();
            replyModel.fromModel(ui.getModel());

            // XXX: à verifier pour le format, et si on quote ou pas encore le forward
            String quotedReply = t("faxtomail.reply.message",
                    decorate(replyModel.getReadSentDate()),
                    replyModel.getFrom(),
                    replyModel.getMessage().replaceAll("\n", "\n> "));
            replyModel.setMessage(quotedReply);
            replyModel.setSubject(t("faxtomail.reply.forwardsubject", replyModel.getSubject()));

            // clear fields that need to be filled by hand
            replyModel.setReadonly(false);
            replyModel.setOriginalDemand(getModel().getOriginalDemand());
            replyModel.setReadSentDate(null);
            replyModel.setTo(null);
            replyModel.setCc(null);
            replyModel.setCci(null);

            openModalFrame(dialogContent, t("faxtomail.reply.forward", getModel().getSubject()), new Dimension(800, 600));

        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't display forward frame", ex);
            }
            getContext().getErrorHelper().showErrorDialog(t("faxtomail.demandReplies.error"), ex);
        }

    }

    public void addTo() {
        ReplyFormUIModel model = getModel();
        Contact contact = model.getContact();
        if (contact != null) {
            String to = model.getTo();
            String email = contact.getEmail();
            if (StringUtils.isNotBlank(to)) {
                to += ";" + email;
            } else {
                to = email;
            }
            model.setTo(to);
        }
    }

    public void addCc() {
        ReplyFormUIModel model = getModel();
        Contact contact = model.getContact();
        if (contact != null) {
            String cc = model.getCc();
            String email = contact.getEmail();
            if (StringUtils.isNotBlank(cc)) {
                cc += ";" + email;
            } else {
                cc = email;
            }
            model.setCc(cc);
        }
    }

    public void addCci() {
        ReplyFormUIModel model = getModel();
        Contact contact = model.getContact();
        if (contact != null) {
            String cci = model.getCci();
            String email = contact.getEmail();
            if (StringUtils.isNotBlank(cci)) {
                cci += ";" + email;
            } else {
                cci = email;
            }
            model.setCci(cci);
        }
    }
}
