package org.nuiton.topia.persistence;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: HqlAndParametersBuilder.java 3163 2014-06-11 16:38:51Z bleny $
 * $HeadURL: https://svn.nuiton.org/topia/tags/topia-3.0-beta-6/topia-persistence/src/main/java/org/nuiton/topia/persistence/HqlAndParametersBuilder.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.text.WordUtils;

import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.Set;

/**
 * A builder to create syntactically correct HQL and associated parameters given properties or after various constraint
 * adds.
 * <p/>
 * It may be used in a Dao to ease dynamic construction of queries.
 *
 * @since 3.0
 */
public class HqlAndParametersBuilder<E extends TopiaEntity> {

    protected Joiner hqlClausesJoiner = Joiner.on(' ').skipNulls();

    protected Class<E> entityClass;

    protected String selectClause;

    protected Set<String> whereClauses = Sets.newLinkedHashSet();

    protected String alias = "topiaEntity_";

    protected Set<String> orderByArguments;

    protected Map<String, Object> parameters = Maps.newLinkedHashMap();

    public HqlAndParametersBuilder(Class<E> entityClass) {
        this.entityClass = entityClass;
    }

    public HqlAndParametersBuilder(Class<E> entityClass, String alias) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(alias));
        this.entityClass = entityClass;
        this.alias = alias;
    }

    /**
     * @deprecated use constructor, alias should not be changed
     */
    @Deprecated
    public void setAlias(String alias) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(alias));
        this.alias = alias;
    }

    public String getAlias() {
        return alias;
    }

    public String getHqlSelectClause() {
        return selectClause;
    }

    public void setSelectClause(String selectClause) {
        this.selectClause = selectClause;
    }

    public void addNull(String property) {
        Preconditions.checkArgument(StringUtils.isNotBlank(property));
        whereClauses.add(alias + "." + property + " is null");
    }

    public void addNotNull(String property) {
        Preconditions.checkArgument(StringUtils.isNotBlank(property));
        whereClauses.add(alias + "." + property + " is not null");
    }

    public void addEquals(String property, Object value) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(property));
        // TODO brendan 02/10/13 do not use HQL parameters of Object are primitive types
        // TODO brendan 02/10/13 if value is instanceof TopiaEntity, we can check type
        if (value == null) {
            addNull(property);
        } else {
            String hqlParameterName = putHqlParameterWithAvailableName(property, value);
            whereClauses.add(alias + "." + property + " = :" + hqlParameterName);
        }
    }

    public void addNotEquals(String property, Object value) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(property));
        // TODO brendan 02/10/13 do not use HQL parameters of Object are primitive types
        // TODO brendan 02/10/13 if value is instanceof TopiaEntity, we can check type
        if (value == null) {
            addNotNull(property);
        } else {
            String hqlParameterName = putHqlParameterWithAvailableName(property, value);
            whereClauses.add(alias + "." + property + " != :" + hqlParameterName);
        }
    }

    public void addIn(String property, Collection<?> values) {
        addInOrNotIn(property, values, true);
    }

    public void addNotIn(String property, Collection<?> values) {
        addInOrNotIn(property, values, false);
    }

    /**
     * @param in true if property value must be in given collection, false if value must not be in given collection
     */
    protected void addInOrNotIn(String property, Collection<?> values, boolean in) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(property));
        Preconditions.checkNotNull(values);
        // TODO brendan 02/10/13 if value is instanceof TopiaEntity, we can check type
        String aliasedProperty = alias + "." + property;
        int valuesSize = values.size();
        if (valuesSize == 0) {
            // XXX brendan 27/02/14 workaround to prevent generating "in ()" which in not supported by PostegreSQL (syntax error)
            if (in) {
                whereClauses.add(" 0 = 1 ");
            } else {
                whereClauses.add(" 1 = 1 ");
            }
        } else if (valuesSize == 1) {
            // if there is only one possible value, replace "in" clause by a "=" clause
            Object onlyElement = Iterables.getOnlyElement(values);
            if (in) {
                addEquals(property, onlyElement);
            } else {
                addNotEquals(property, onlyElement);
            }
        } else {
            boolean propertyMayBeNull = values.contains(null);
            Collection<?> hqlParameterValue = values;
            if (propertyMayBeNull /* && ! in */) {
                // duplicate given collection because we don't want 'null'
                // in hqlParameterValue and we don't want side effect on parameters
                hqlParameterValue = Sets.newLinkedHashSet(values);
                hqlParameterValue.remove(null);
            }
            String hqlParameterName = putHqlParameterWithAvailableName(property, hqlParameterValue);
            String whereClause;
            if (in) {
                whereClause = String.format(" %s in ( :%s ) ", aliasedProperty, hqlParameterName);
                if (propertyMayBeNull) {
                    whereClause += " or " + aliasedProperty + " is null";
                }
            } else {
                whereClause = String.format(" %s not in ( :%s ) ", aliasedProperty, hqlParameterName);
                if (propertyMayBeNull) {
                    whereClause += " and " + aliasedProperty + " is not null";
                }
            }
            whereClauses.add(whereClause);
        }
    }

    public void addTopiaIdEquals(String property, String topiaId) {
        Preconditions.checkNotNull(topiaId);
        addEquals(property + "." + TopiaEntity.PROPERTY_TOPIA_ID, topiaId);
    }

    public void addTopiaIdIn(String property, Collection<String> topiaIds) {
        addIn(property + "." + TopiaEntity.PROPERTY_TOPIA_ID, topiaIds);
    }

    public void addTopiaIdNotEquals(String property, String topiaId) {
        Preconditions.checkNotNull(topiaId);
        addNotEquals(property + "." + TopiaEntity.PROPERTY_TOPIA_ID, topiaId);
    }

    public void addTopiaIdNotIn(String property, Collection<String> topiaIds) {
        addNotIn(property + "." + TopiaEntity.PROPERTY_TOPIA_ID, (Collection) topiaIds);
    }

    public void addContains(String property, Object value) {
        String hqlParameterName = putHqlParameterWithAvailableName(property, value);
        whereClauses.add(":" + hqlParameterName + " in elements(" + alias + "." + property + ")");
    }

    public void addNotContains(String property, Object value) {
        String hqlParameterName = putHqlParameterWithAvailableName(property, value);
        whereClauses.add(":" + hqlParameterName + " not in elements(" + alias + "." + property + ")");
    }

    public void addLike(String property, String pattern) {
        Preconditions.checkNotNull(pattern);
        Preconditions.checkArgument(StringUtils.isNotEmpty(property));
        String hqlParameterName = putHqlParameterWithAvailableName(property, pattern);
        whereClauses.add(alias + "." + property + " like :" + hqlParameterName);
    }

    public void addNotLike(String property, String pattern) {
        Preconditions.checkNotNull(pattern);
        Preconditions.checkArgument(StringUtils.isNotEmpty(property));
        String hqlParameterName = putHqlParameterWithAvailableName(property, pattern);
        whereClauses.add(alias + "." + property + " not like :" + hqlParameterName);
    }

    public void addLowerThan(String property, Date date) {
        doAddLowerThan(property, date);
    }

    public void addLowerOrEquals(String property, Date date) {
        doAddLowerOrEquals(property, date);
    }

    public void addGreaterThan(String property, Date date) {
        doAddGreaterThan(property, date);
    }

    public void addGreaterOrEquals(String property, Date date) {
        doAddGreaterOrEquals(property, date);
    }

    public void addLowerThan(String property, Number number) {
        doAddLowerThan(property, number);
    }

    public void addLowerOrEquals(String property, Number number) {
        doAddLowerOrEquals(property, number);
    }

    public void addGreaterThan(String property, Number number) {
        doAddGreaterThan(property, number);
    }

    public void addGreaterOrEquals(String property, Number number) {
        doAddGreaterOrEquals(property, number);
    }

    public void addLowerThan(String property, String string) {
        doAddLowerThan(property, string);
    }

    public void addLowerOrEquals(String property, String string) {
        doAddLowerOrEquals(property, string);
    }

    public void addGreaterThan(String property, String string) {
        doAddGreaterThan(property, string);
    }

    public void addGreaterOrEquals(String property, String string) {
        doAddGreaterOrEquals(property, string);
    }

    protected void doAddLowerThan(String property, Object value) {
        Preconditions.checkNotNull(value);
        String hqlParameterName = putHqlParameterWithAvailableName(property, value);
        whereClauses.add(alias + "." + property + " < :" + hqlParameterName);
    }

    protected void doAddLowerOrEquals(String property, Object value) {
        Preconditions.checkNotNull(value);
        String hqlParameterName = putHqlParameterWithAvailableName(property, value);
        whereClauses.add(alias + "." + property + " <= :" + hqlParameterName);
    }

    protected void doAddGreaterThan(String property, Object value) {
        Preconditions.checkNotNull(value);
        String hqlParameterName = putHqlParameterWithAvailableName(property, value);
        whereClauses.add(alias + "." + property + " > :" + hqlParameterName);
    }

    protected void doAddGreaterOrEquals(String property, Object value) {
        Preconditions.checkNotNull(value);
        String hqlParameterName = putHqlParameterWithAvailableName(property, value);
        whereClauses.add(alias + "." + property + " >= :" + hqlParameterName);
    }

    public void addWhereClause(String whereClause) {
        Preconditions.checkArgument(StringUtils.isNotBlank(whereClause));
        whereClauses.add(whereClause);
    }

    public void addCollectionIsEmpty(String property) {
        whereClauses.add(alias + "." + property + " is empty" );
    }

    public void addCollectionIsNotEmpty(String property) {
        whereClauses.add(alias + "." + property + " is not empty" );
    }

    public void addWhereClause(String whereClause, Map<String, Object> hqlParameters) {
        Preconditions.checkNotNull(hqlParameters);
        Set<String> collisiontParameterNames = Sets.newHashSet(Sets.intersection(parameters.keySet(), hqlParameters.keySet()));
        boolean noCollision = collisiontParameterNames.isEmpty();
        if (noCollision) {
            parameters.putAll(hqlParameters);
        } else {

            // add all parameters with no collision
            Set<String> noCollisiontParameterNames = Sets.difference(hqlParameters.keySet(), collisiontParameterNames);
            for (String parameterName : noCollisiontParameterNames) {
                Object parameterValue = hqlParameters.get(parameterName);
                parameters.put(parameterName, parameterValue);
            }
            // resolve all collision parameters
            for (String parameterName : collisiontParameterNames) {
                Object parameterValue = hqlParameters.get(parameterName);

                // resolved parameter name
                String newParameterName = putHqlParameterWithAvailableName(parameterName, parameterValue);

                // replace the :parameterName (with no next alphanumeric caracter)
                whereClause = whereClause.replaceAll(":" + parameterName + "(?!\\w)", ":" + newParameterName);
            }
        }
        addWhereClause(whereClause);
    }

    public void setWhereClauses(Set<String> whereClauses) {
        Preconditions.checkNotNull(whereClauses);
        this.whereClauses = whereClauses;
    }

    public Set<String> getWhereClauses() {
        return whereClauses;
    }

    public void setParameters(Map<String, Object> parameters) {
        Preconditions.checkNotNull(parameters);
        this.parameters = parameters;
    }

    public void setOrderByArguments(Set<String> orderByArguments) {
        Preconditions.checkNotNull(orderByArguments);
        this.orderByArguments = orderByArguments;
    }

    public void setOrderByArguments(String... orderByArguments) {
        Set<String> orderByArgumentsAsSet = Sets.newLinkedHashSet();
        orderByArgumentsAsSet.addAll(Arrays.asList(orderByArguments));
        this.orderByArguments = orderByArgumentsAsSet;
    }

    public String getHqlFromClause() {
        String hqlFromClause = String.format("from %s %s", entityClass.getCanonicalName(), alias);
        return hqlFromClause;
    }

    public String getHqlWhereClause() {
        String hqlWhereClause;
        if (whereClauses.isEmpty()) {
            hqlWhereClause = null;
        } else if (whereClauses.size() == 1) {
            String whereClause = Iterables.getOnlyElement(whereClauses);
            hqlWhereClause = "where " + whereClause;
        } else {
            hqlWhereClause = "where (" + StringUtils.join(whereClauses, ") and (") + ")";
        }
        return hqlWhereClause;
    }

    public String getHqlOrderByClause() {
        String hqlOrderByClause = null;
        if (CollectionUtils.isNotEmpty(orderByArguments)) {
            hqlOrderByClause = "order by " + alias + "." + StringUtils.join(orderByArguments, ", " + alias + ".");
        }
        return hqlOrderByClause;
    }

    public String getHql() {
        String hql = hqlClausesJoiner.join(
                getHqlSelectClause(),
                getHqlFromClause(),
                getHqlWhereClause(),
                getHqlOrderByClause());
        return hql;
    }

    /**
     * Converts a (nested) property name to an HQL argument name.
     * <p/>
     * For example getParameterName("yearlyDeclaration.survey.topiaId") → "yearlyDeclarationSurveyTopiaId"
     *
     * @param propertyName the name of a property, can be a path to a nested property
     * @return a string that can syntactically be used as an HQL parameter name, not prefixed by ':'
     */
    protected String getParameterName(String propertyName) {
        Preconditions.checkArgument(StringUtils.isNotBlank(propertyName));
        String capitalize = WordUtils.capitalize(propertyName, '.');
        String withoutDots = capitalize.replaceAll("\\.", "");
        String parameterName = StringUtils.uncapitalize(withoutDots);
        return parameterName;
    }

    /**
     * Add a parameter in the parameters map searching with the suitable parameter name in order to prevent conflicts.
     *
     * @return the found key where the parameter has been added, suitable to use in the where clause
     */
    protected String putHqlParameterWithAvailableName(String propertyName, Object value) {
        String parameterNamePrefix = getParameterName(propertyName);
        int suffix = 0;
        String parameterName = parameterNamePrefix + suffix;
        while (parameters.containsKey(parameterName)) {
            suffix++;
            parameterName = parameterNamePrefix + suffix;
        }
        parameters.put(parameterName, value);
        return parameterName;
    }

    public Map<String, Object> getHqlParameters() {
        return parameters;
    }

    public boolean isOrderByClausePresent() {
        return CollectionUtils.isNotEmpty(orderByArguments);
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this)
                .append("hql", getHql())
                .append("hqlParameters", getHqlParameters())
                .toString();
    }
}
