package org.nuiton.db.meta;

/*-
 * #%L
 * Nuiton DB Meta
 * %%
 * Copyright (C) 2019 Nuiton
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Charsets;
import com.google.common.collect.Ordering;
import com.google.common.io.Files;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.BufferedWriter;
import java.io.File;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

public class DatabaseMetaCommentExportHelper {

    private static final Log log = LogFactory.getLog(DatabaseMetaCommentExportHelper.class);

    private static final Ordering<TableMeta> TABLE_ORDERING = Ordering.natural().onResultOf(TableMeta::getName);
    private static final Ordering<ColumnMeta> COLUMN_ORDERING = Ordering.natural().onResultOf(ColumnMeta::getName);
    private static final Ordering<ProcedureMeta> PROCEDURE_ORDERING = Ordering.natural().onResultOf(ProcedureMeta::getName);
    private static final Ordering<CustomTypeMeta> CUSTOM_TYPE_ORDERING = Ordering.natural().onResultOf(CustomTypeMeta::getName);

    protected final String schemaPlaceholder;

    public DatabaseMetaCommentExportHelper(String schemaPlaceholder) {
        this.schemaPlaceholder = schemaPlaceholder;
    }

    public DatabaseMetaCommentExportHelper() {
        this("public");
    }

    public void exportCommentsToFile(DatabaseMeta databaseMeta, File file) {
        try {
            try (BufferedWriter writer = Files.newWriter(file, Charsets.UTF_8)) {
                List<String> lines = asCommentLines(databaseMeta);
                for (String line : lines) {
                    writer.write(line);
                    writer.write("\n");
                }
                writer.flush();
            }
            log.debug("File wrote to " + file.getAbsolutePath());
        } catch (Exception eee) {
            log.error("Unable to write file", eee);
        }
    }

    private List<String> asCommentLines(DatabaseMeta databaseMeta) {
        List<String> lines = new LinkedList<>();
        TABLE_ORDERING.sortedCopy(databaseMeta.getTables()).forEach(table -> {
            lines.add(String.format("\n-- TABLE %s", table.getName()));
            List<String> tableLines = asCommentLines(table);
            lines.addAll(tableLines);
        });
        TABLE_ORDERING.sortedCopy(databaseMeta.getViews()).forEach(view -> {
            lines.add(String.format("\n-- VIEW %s", view.getName()));
            List<String> tableLines = asCommentLines(view);
            lines.addAll(tableLines);
        });
        PROCEDURE_ORDERING.sortedCopy(databaseMeta.getProcedures()).forEach(procedure -> {
            lines.add(String.format("\n-- FUNCTION %s", procedure.getName()));
            List<String> tableLines = asCommentLines(procedure);
            lines.addAll(tableLines);
        });
        CUSTOM_TYPE_ORDERING.sortedCopy(databaseMeta.getCustomTypes()).forEach(customType -> {
            lines.add(String.format("\n-- TYPE %s", customType.getName()));
            List<String> tableLines = asCommentLines(customType);
            lines.addAll(tableLines);
        });

        return lines;
    }

    private List<String> asCommentLines(TableMeta tableMeta) {
        List<String> result = new LinkedList<>();
        if (tableMeta.getComment().isPresent()) {
            String line = formatCommentLine(tableMeta.getName(), tableMeta.isView() ? CommentLineType.VIEW : CommentLineType.TABLE, null, tableMeta.getComment().get());
            result.add(line);
        }
        for (ColumnMeta column : COLUMN_ORDERING.sortedCopy(tableMeta.getColumns())) {
            Optional<String> columnLine = asCommentLine(tableMeta, column);
            columnLine.ifPresent(result::add);
        }
        return result;
    }

    private Optional<String> asCommentLine(TableMeta tableMeta, ColumnMeta columnMeta) {
        if (columnMeta.getComment().isPresent()) {
            String line = formatCommentLine(tableMeta.getName(), CommentLineType.COLUMN, columnMeta.getName(), columnMeta.getComment().get());
            return Optional.of(line);
        }
        return Optional.empty();
    }

    private List<String> asCommentLines(ProcedureMeta procedureMeta) {
        List<String> result = new LinkedList<>();
        if (procedureMeta.getComment().isPresent()) {
            String argTypes = procedureMeta.getArgs()
                    .stream()
                    .map(ArgumentMeta::getType)
                    .collect(Collectors.joining( "," ));
            String procedureIdentifier = String.format("%s(%s)", procedureMeta.getName(), argTypes);
            String line = formatCommentLine(procedureIdentifier, CommentLineType.FUNCTION, null, procedureMeta.getComment().get());
            result.add(line);
        }
        return result;
    }

    private List<String> asCommentLines(CustomTypeMeta customTypeMeta) {
        List<String> result = new LinkedList<>();
        if (customTypeMeta.getComment().isPresent()) {
            String line = formatCommentLine(customTypeMeta.getName(), CommentLineType.TYPE, null, customTypeMeta.getComment().get());
            result.add(line);
        }
        return result;
    }

    private String formatCommentLine(String table, CommentLineType type, String colonne, String commentaire) {
        String escaped = commentaire.replaceAll("'", "''");
        String result;
        if (StringUtils.isEmpty(colonne) && CommentLineType.VIEW.equals(type)) {
            result = String.format("COMMENT ON VIEW %s.%s IS '%s';", schemaPlaceholder, table, escaped);
        } else if (StringUtils.isEmpty(colonne) && CommentLineType.FUNCTION.equals(type)) {
            result = String.format("COMMENT ON FUNCTION %s.%s IS '%s';", schemaPlaceholder, table, escaped);
        } else if (StringUtils.isEmpty(colonne) && CommentLineType.TYPE.equals(type)) {
            result = String.format("COMMENT ON TYPE %s.%s IS '%s';", schemaPlaceholder, table, escaped);
        } else if (StringUtils.isEmpty(colonne)) {
            result = String.format("COMMENT ON TABLE %s.%s IS '%s';", schemaPlaceholder, table, escaped);
        } else {
            result = String.format("COMMENT ON COLUMN %s.%s.%s IS '%s';", schemaPlaceholder, table, colonne, escaped);
        }
        return result;
    }

    private enum CommentLineType {
        VIEW,
        FUNCTION,
        TABLE,
        COLUMN,
        TYPE
    }

}
