/*
 * *##% 
 * JRedmine maven plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.jredmine.plugin.announcement;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.changes.model.Action;
import org.apache.maven.plugins.changes.model.Release;
import org.apache.velocity.Template;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.context.Context;
import org.apache.velocity.exception.ResourceNotFoundException;
import org.codehaus.plexus.util.StringUtils;
import org.nuiton.jredmine.model.Attachment;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author chemit
 * @since 1.0.0
 */
public class AnnouncementGenerator {

    /**
     * The token any of urls denoting the base URL for the issue management.
     */
    private static final String URL_TOKEN = "%URL%";
    /**
     * The token in {@link #attachmentLinkTemplate} denoting the attachment ID.
     */
    private static final String ATTACHMENT_TOKEN = "%FILE%";
    /**
     * the redmine url
     */
    protected final String url;
    /**
     * the redmine attachment url template
     */
    protected final String attachmentLinkTemplate;
    /**
     * logger
     */
    private final Log log;

    public AnnouncementGenerator(Log log, String url, String attachmentLinkTemplate) {
        this.log = log;
        this.url = url;
        this.attachmentLinkTemplate = attachmentLinkTemplate;
    }

    public Map<Attachment, String> getAttachmentsUrls(Attachment[] attachments) {

        // transform attachments urls

        boolean hasAttachmentLinks = canGenerateAttachmentLinks();

        Map<Attachment, String> urls = null;

        if (hasAttachmentLinks) {

            urls = new HashMap<Attachment, String>();
            if (hasAttachmentLinks) {
                for (Attachment a : attachments) {
                    String u = parseAttachmentLink(a.getId() + "");
                    urls.put(a, u);
                }
            }
        } else {
            getLog().warn("can not render attachments urls");
        }

        return urls;
    }

    public void doGenerate(VelocityEngine engine, Context context, File out, String templatePath, String templateEncoding)
            throws MojoExecutionException {

        try {

            if (!out.getParentFile().exists()) {
                out.getParentFile().mkdirs();
            }

            Writer writer = new OutputStreamWriter(new FileOutputStream(out), templateEncoding);

            Template velocityTemplate = engine.getTemplate(templatePath, templateEncoding);

            velocityTemplate.merge(context, writer);

            writer.flush();

            writer.close();

        } catch (ResourceNotFoundException rnfe) {
            throw new MojoExecutionException("Resource not found.", rnfe);
        } catch (Exception e) {
            throw new MojoExecutionException(e.toString(), e);
        }
    }

    /**
     * Checks whether links to the issues can be generated.
     *
     * @return <code>true</code> if issue links can be generated, <code>false</code> otherwise.
     */
    public boolean canGenerateAttachmentLinks() {
        return !StringUtils.isBlank(attachmentLinkTemplate) && (!StringUtils.isBlank(url) || attachmentLinkTemplate.indexOf(URL_TOKEN) < 0);
    }

    protected String parseAttachmentLink(String id) {
        String parseLink;
        String issueLink = this.attachmentLinkTemplate;
        parseLink = issueLink.replaceFirst(ATTACHMENT_TOKEN, id);

        if (parseLink.indexOf(URL_TOKEN) >= 0) {
            String u = this.url.substring(0, this.url.lastIndexOf("/"));
            parseLink = parseLink.replaceFirst(URL_TOKEN, u);
        }

        return parseLink;
    }

    /**
     * Get the latest release by matching the supplied releases
     * with the version from the pom.
     *
     * @param releases       list of releases
     * @param releaseVersion the release version
     * @return A <code>Release</code> that matches the next release of the current project
     * @throws MojoExecutionException
     */
    public Release getLatestRelease(List<?> releases, String releaseVersion)
            throws MojoExecutionException {
        boolean isFound;

        Release release;

        String pomVersion = releaseVersion;

        getLog().debug("Found " + releases.size() + " releases.");

        for (Object release1 : releases) {
            release = (Release) release1;
            if (getLog().isDebugEnabled()) {
                getLog().debug("The release: " + release.getVersion() + " has " + release.getActions().size() + " actions.");
            }

            if (release.getVersion() != null && release.getVersion().equals(pomVersion)) {
                isFound = true;
                if (getLog().isDebugEnabled()) {
                    getLog().debug("Found the correct release: " + release.getVersion());
                    logRelease(release);
                }
                return release;
            }
        }

        release = getRelease(releases, pomVersion);
        isFound = (release != null);

        if (!isFound) {
            throw new MojoExecutionException("Couldn't find the release '" + pomVersion + "' among the supplied releases.");
        } else {
        }
        return release;
    }

    protected Release getRelease(List<?> releases, String version) {
        Release release;
        for (Object release1 : releases) {
            release = (Release) release1;
            if (getLog().isDebugEnabled()) {
                getLog().debug("The release: " + release.getVersion() + " has " + release.getActions().size() + " actions.");
            }

            if (release.getVersion() != null && release.getVersion().equals(version)) {
                if (getLog().isDebugEnabled()) {
                    getLog().debug("Found the correct release: " + release.getVersion());
                    logRelease(release);
                }
                return release;
            }
        }
        return null;
    }

    private void logRelease(Release release) {
        Action action;
        for (Object o : release.getActions()) {
            action = (Action) o;
            getLog().debug("o " + action.getType());
            getLog().debug("issue : " + action.getIssue());
            getLog().debug("action : " + action.getAction());
            getLog().debug("dueTo : " + action.getDueTo());
            getLog().debug("dev : " + action.getDev());
        }
    }

    protected Log getLog() {
        return log;
    }
}
