/*
 * #%L
 * JRedmine :: Maven plugin
 * 
 * $Id: UpdateVersionMojo.java 303 2012-07-15 11:14:29Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.4/jredmine-maven-plugin/src/main/java/org/nuiton/jredmine/plugin/UpdateVersionMojo.java $
 * %%
 * Copyright (C) 2009 - 2012 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine.plugin;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.StringUtils;
import org.nuiton.jredmine.model.ModelHelper;
import org.nuiton.jredmine.model.Version;
import org.nuiton.jredmine.model.VersionStatusEnum;
import org.nuiton.plugin.PluginHelper;

import java.text.ParseException;
import java.util.Arrays;
import java.util.Date;

/**
 * Create or update a version on redmine server.
 * <p/>
 * Will add a the version if not existing, otherwise, will update the version.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0.0
 */
@Mojo(name = "update-version", requiresOnline = true, requiresProject = true)
public class UpdateVersionMojo extends AbstractRedmineMojoWithProjectAndVersion implements DryRunAware{

    /**
     * Flag to know if anonymous connexion to redmine server is required.
     * <p/>
     * <b>Note:</b> If set to {@code false}, you should fill {@link #username}
     * and {@link #password} properties.
     *
     * @since 1.1.3
     */
    @Parameter(property = "redmine.anonymous", defaultValue = "false")
    protected boolean anonymous;

    /**
     * The news description to update to redmine server.
     * <p/>
     * Note : if not set, no update of the description will be made.
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.versionDescription")
    protected String versionDescription;

    /**
     * A flag to close the version on redmine (will fix effectiveDate).
     * <p/>
     * The status of the version will be setted to {@code closed} then.
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.closeVersion", defaultValue = "false")
    protected boolean closeVersion;

    /**
     * The effective date to set on the version.
     * <p/>
     * The format of the date is {@code yyyy-mm-dd}.
     * <p/>
     * If not Set - will use current day date only if {@link #closeVersion}
     * flag is on.
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.effectiveDate")
    protected String effectiveDate;

    /**
     * The status to set on the version amoung values {@code open, lock, closed}.
     * <p/>
     * If not Set - will use the {@code closed} value only if
     * {@link #closeVersion} flag is on.
     *
     * @since 1.2.1
     */
    @Parameter(property = "redmine.versionStatus")
    protected String versionStatus;

    /**
     * A flag to skip the goal.
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.skipUpdateVersion", defaultValue = "false")
    protected boolean skipUpdateVersion;

    /**
     * A flag to test plugin but send nothing to redmine.
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.dryRun", defaultValue = "false")
    protected boolean dryRun;

    /**
     * A flag to restirct only one run in a build (for multi-module context).
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.runOnce", defaultValue = "true")
    protected boolean runOnce;

    /** effective date to set */
    private Date date;

    /**
     * new status to apply (stay null if nothing is asked).
     *
     * @since 1.2.1
     */
    private VersionStatusEnum newVersionStatus;

    public UpdateVersionMojo() {
        super(true);
    }

    @Override
    public boolean isAnonymous() {
        return anonymous;
    }

    @Override
    public void setAnonymous(boolean anonymous) {
        this.anonymous = anonymous;
    }

    @Override
    public boolean isDryRun() {
        return dryRun;
    }

    @Override
    public void setDryRun(boolean dryRun) {
        this.dryRun = dryRun;
    }

    @Override
    protected boolean isGoalSkip() {
        return skipUpdateVersion;
    }

    @Override
    protected boolean isRunOnce() {
        return runOnce;
    }

    @Override
    protected boolean checkRunOnceDone() {
        return isRunOnce() && !isExecutionRoot();
    }

    @Override
    protected void init() throws Exception {

        if (versionId == null || versionId.trim().isEmpty()) {
            throw new MojoExecutionException("required a versionId parameter");
        }

        versionId = PluginHelper.removeSnapshotSuffix(versionId);

        runOnceDone = false;

        if (isRunOnce()) {
            runOnceDone = checkRunOnceDone();
            if (runOnceDone) {
                return;
            }
        }

        if (effectiveDate != null && !effectiveDate.trim().isEmpty()) {
            try {
                date = dateFormat.parse(effectiveDate);
            } catch (ParseException e) {
                throw new MojoExecutionException(
                        "could not parse effectivate date " + effectiveDate +
                        " for reason " + e.getMessage(), e);
            }
        } else if (closeVersion) {
            date = new Date();
        }

        if (closeVersion) {

            // by default use the closed status when closing a version
            newVersionStatus = VersionStatusEnum.closed;
        }

        if (StringUtils.isNotEmpty(versionStatus)) {
            try {
                newVersionStatus = VersionStatusEnum.valueOf(versionStatus);
            } catch (IllegalArgumentException e) {
                throw new MojoExecutionException(
                        "could not parse status " + versionStatus +
                        " for reason " + e.getMessage() +
                        ", should be one of values : " +
                        Arrays.toString(VersionStatusEnum.values()), e);
            }

            if (closeVersion) {
                // warns user

                if (getLog().isWarnEnabled()) {
                    getLog().warn("While using the closeVersion flag, you " +
                                  "should not set also the version status " +
                                  ": [" + versionStatus + "]");
                }
            }
        }
        super.init();

    }

    @Override
    protected void doAction() throws Exception {

        if (dryRun) {
            getLog().info("\n  dryRun flag is on, no data will be send!\n");
        }

        // get version

        boolean needCreateVersion;

        Version[] versions = service.getVersions(projectId);
        Version v = ModelHelper.byVersionName(versionId, versions);

        if (v == null) {
            // version must be created
            needCreateVersion = true;
            v = new Version();
            v.setName(versionId);
        } else {

            needCreateVersion = false;
        }

        if (versionDescription != null && !versionDescription.trim().isEmpty()) {
            v.setDescription(versionDescription.trim());
        }

        if (date != null) {
            v.setEffectiveDate(date);
        }

        if (newVersionStatus != null) {

            // change status of the version
            if (isVerbose()) {
                getLog().info("Will set status " + newVersionStatus);

            }
            v.setStatus(newVersionStatus.name());
        }

        releaseVersion = v;

        // prepare version
        if (needCreateVersion) {

            // create version
            getLog().info("create version " + releaseVersion.getName());

            if (!dryRun) {
                service.addVersion(projectId, releaseVersion);
            }

        } else {

            // update version
            getLog().info("udpate version " + releaseVersion.getName());

            if (!dryRun) {
                service.updateVersion(projectId, releaseVersion);
            }
        }

    }
}
