/*
 * #%L
 * JRedmine :: Client
 * 
 * $Id: DefaultRedmineServiceImplementor.java 193 2011-05-20 22:44:58Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.2/jredmine-client/src/main/java/org/nuiton/jredmine/DefaultRedmineServiceImplementor.java $
 * %%
 * Copyright (C) 2009 - 2010 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;
import org.nuiton.io.rest.RestClient;
import org.nuiton.io.rest.RestClientConfiguration;
import org.nuiton.io.rest.RestException;
import org.nuiton.io.rest.RestRequest;
import org.nuiton.jredmine.model.io.xpp3.RedmineXpp3Helper;
import org.nuiton.jredmine.rest.RedmineRestClient;

import java.io.IOException;
import java.io.InputStream;

/**
 * Created: 2 janv. 2010
 *
 * @author Tony Chemit <chemit@codelutin.com> Copyright Code Lutin
 * @version $Revision: 193 $
 *          <p/>
 *          Mise a jour: $Date: 2011-05-21 00:44:58 +0200 (Sat, 21 May 2011) $ par :
 *          $Author: tchemit $
 * @plexus.component role="org.nuiton.jredmine.RedmineServiceImplementor" role-hint="default"
 * @since 1.0.3
 */
public class DefaultRedmineServiceImplementor implements RedmineServiceImplementor {
    /**
     * Logger
     */
    private static final Log log = LogFactory.getLog(DefaultRedmineServiceImplementor.class);
    //FIXME Make this works again (there is a bad internal state somewhere...)
//    /**
//     * client Rest
//     *
//     * @plexus.requirement role="org.nuiton.io.rest.RestClient" role-hint="redmine"
//     */
    protected RestClient session;
    /**
     * xpp3 xpp3Helper to transform xml stream to pojo
     */
    protected final RedmineXpp3Helper xpp3Helper;
    /**
     * internal state to known if service was init
     */
    protected boolean init;

    public DefaultRedmineServiceImplementor() {
        xpp3Helper = new RedmineXpp3Helper();
        if (log.isDebugEnabled()) {
            log.debug("new " + this);
        }
        session = new RedmineRestClient();
    }

    ///////////////////////////////////////////////////////////////////////////
    /// RedmineServiceImplementor implementation
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public boolean isInit() {
        return init;
    }

    @Override
    public RedmineServiceImplementor init(RestClientConfiguration configuration) throws RedmineServiceException {
        session.setConfiguration(configuration);
        RedmineServiceImplementor result = init(session);
        return result;
    }

    @Override
    public RedmineServiceImplementor init(RestClient session) throws RedmineServiceException {
        if (init) {
            throw new IllegalStateException("the client " + this + " was already init!");
        }
        try {
            this.session = session;

            if (!session.isOpen()) {
                session.open();
            }

            init = true;
        } catch (Exception e) {
            throw new RedmineServiceException("could not init service for reason " + e.getMessage(), e);
        }
        return this;
    }

    @Override
    public void destroy() throws RedmineServiceException {
        checkInit();
        try {
            if (session.isOpen()) {
                try {
                    session.close();
                } catch (RestException ex) {
                    throw new RedmineServiceException("has problem while closing Rest client " + ex.getMessage(), ex);
                }
            }
        } finally {
            // can't remove the reference since plexus injects it
//            if (session != null) {
//                session = null;
//            }
            init = false;
        }
    }

    @Override
    public <T> T getData(String requestName, Class<T> type, Object... args) throws RedmineServiceException {
        checkInit();
        InputStream stream = askDataStream(requestName, args);
        T result = getDataFromStream(type, stream);
        return result;
    }

    @Override
    public <T> T[] getDatas(String requestName, Class<T> type, Object... args) throws RedmineServiceException {
        checkInit();
        InputStream stream = askDataStream(requestName, args);
        T[] result = getDatasFromStream(type, stream);
        return result;
    }

    @Override
    public <T> T sendData(String requestName, Class<T> klazz, Object... args) throws RedmineServiceException {
        checkLoggued();
        InputStream stream = sendDataStream(requestName, args);
        T result = getDataFromStream(klazz, stream);
        return result;
    }

    @Override
    public <T> T[] sendDatas(String requestName, Class<T> klazz, Object... args) throws RedmineServiceException {
        checkLoggued();
        InputStream stream = sendDataStream(requestName, args);
        T[] result = getDatasFromStream(klazz, stream);
        return result;
    }

    @Override
    public void checkLoggued() throws IllegalStateException, RedmineServiceLoginException, NullPointerException {
        checkInit();
        checkSessionNotNull(session);
        checkSessionConfigurationNotNull(session);
        if (session.getConfiguration().isAnonymous()) {
            throw new RedmineServiceLoginException("can not access this service in anonymous mode");
        }
    }

    @Override
    public void checkNotLoggued(RestClient session) throws RedmineServiceLoginException, NullPointerException {
//        checkInit();
        checkSessionNotNull(session);
        checkSessionConfigurationNotNull(session);
        if (!session.getConfiguration().isAnonymous()) {
            throw new RedmineServiceLoginException("session must be anonymous in this service");
        }
    }

    protected InputStream askDataStream(String requestName, Object... args) throws RedmineServiceException {

        RestRequest r = getRequest(requestName, args);

        // obtain data from rest client
        try {

            InputStream stream = session.askData(r);
            return stream;
        } catch (Exception e) {
            throw new RedmineServiceException("could not obtain data stream for request " + requestName + " for reason " + e.getMessage(), e);
        }
    }

    protected InputStream sendDataStream(String requestName, Object... args) throws RedmineServiceException {

        RestRequest r = getRequest(requestName, args);

        // obtain data from rest client
        try {

            InputStream stream = session.sendData(r);
            return stream;
        } catch (Exception e) {
            throw new RedmineServiceException("could not send data stream for request " + requestName + " for reason " + e.getMessage(), e);
        }
    }

    protected RestRequest getRequest(String requestName, Object... args) throws RedmineServiceException {
        RestRequest r;
        try {
            r = session.getRequest(requestName, args);
        } catch (Exception e) {
            throw new RedmineServiceException("could not find the request named " + requestName + " for reason " + e.getMessage(), e);
        }
        if (r == null) {
            throw new RedmineServiceException("could not find the request named " + requestName);
        }
        return r;
    }

    protected <T> T getDataFromStream(Class<T> type, InputStream stream) throws RedmineServiceException {
        if (stream == null) {
            return null;
        }

        try {
            T result = xpp3Helper.readObject(type, stream, true);
            return result;
        } catch (IOException ex) {
            throw new RedmineServiceException("could not obtain datas of type " + type + " for reason " + ex.getMessage(), ex);
        } catch (XmlPullParserException ex) {
            throw new RedmineServiceException("could not obtain datas of type " + type + " for reason " + ex.getMessage(), ex);
        }
    }

    protected <T> T[] getDatasFromStream(Class<T> type, InputStream stream) throws RedmineServiceException {
        if (stream == null) {
            return null;
        }
        try {
            T[] result = xpp3Helper.readObjects(type, stream, true);
            return result;
        } catch (IOException ex) {
            throw new RedmineServiceException("could not obtain datas of type " + type + " for reason " + ex.getMessage(), ex);
        } catch (XmlPullParserException ex) {
            throw new RedmineServiceException("could not obtain datas of type " + type + " for reason " + ex.getMessage(), ex);
        }
    }

    protected void checkInit() throws IllegalStateException {
        if (!init) {
            throw new IllegalStateException("the client " + this + " is not init!");
        }
    }

    protected void checkSessionNotNull(RestClient session) {
        if (session == null) {
            throw new NullPointerException("session can not be null");
        }
    }

    protected void checkSessionConfigurationNotNull(RestClient session) {
        if (session.getConfiguration() == null) {
            throw new NullPointerException("session configuration can not be null");
        }
    }
}
