package org.nuiton.jredmine;

import org.nuiton.jredmine.model.*;

/**
 * Contract of all redmine operations which do not requires any login to server.
 * <p/>
 * A default implementation is offered in
 * {@link org.nuiton.jredmine.DefaultRedmineAnonymousService}.
 * <p/>
 * Created: 2 janv. 2010
 *
 * @author Tony Chemit <chemit@codelutin.com> Copyright Code Lutin
 * @version $Revision: 66 $
 *          <p/>
 *          Mise a jour: $Date: 2010-01-03 11:49:37 +0100 (dim., 03 janv. 2010) $ par :
 *          $Author: tchemit $
 * @see org.nuiton.jredmine.DefaultRedmineAnonymousService
 * @since 1.0.3
 */
public interface RedmineAnonymousService {

    /**
     * Obtain all accessible projets.
     *
     * @return all the projects
     * @throws RedmineServiceException if any pb while retriving datas
     * @see org.nuiton.jredmine.model.Project
     */
    Project[] getProjects() throws RedmineServiceException;

    /**
     * Obtain all the priorities defined on a {@link org.nuiton.jredmine.model.Issue}.
     * <p/>
     * <b>Note : </b> The priorities are common for all projects.
     *
     * @return all the issue properties
     * @throws RedmineServiceException if any pb while retriving datas
     * @see org.nuiton.jredmine.model.IssuePriority
     */
    IssuePriority[] getIssuePriorities() throws RedmineServiceException;

    /**
     * Obtain all the statuses defined on a {@link org.nuiton.jredmine.model.Issue}.
     * <p/>
     * <b>Note : </b> The statuses are common for all projects.
     *
     * @return all the issue statuses
     * @throws RedmineServiceException if any pb while retriving datas
     * @see org.nuiton.jredmine.model.IssueStatus
     */
    IssueStatus[] getIssueStatuses() throws RedmineServiceException;

    /**
     * Obtain a project given his name.
     *
     * @param projectName the name of the project
     * @return the project
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Project
     */
    Project getProject(String projectName) throws RedmineServiceException;

    /**
     * Obtain all categories defined on issues for a given project.
     *
     * @param projectName the name of the project
     * @return the categories of issues for the given project.
     * @throws RedmineServiceException if any pb while retriving datas
     * @see org.nuiton.jredmine.model.IssueCategory
     */
    IssueCategory[] getIssueCategories(String projectName) throws RedmineServiceException;

    /**
     * Obtain all trackers defined on a given project.
     *
     * @param projectName the name of the project
     * @return the trackers for the given project.
     * @throws RedmineServiceException if any pb while retriving datas
     * @see org.nuiton.jredmine.model.Tracker
     */
    Tracker[] getTrackers(String projectName) throws RedmineServiceException;

    /**
     * Obtain all news defined on a given project.
     *
     * @param projectName the name of the project
     * @return the news for the given project.
     * @throws RedmineServiceException if any pb while retriving datas
     * @see News
     */
    News[] getNews(String projectName) throws RedmineServiceException;

    /**
     * Obtain all users defined on a given project.
     *
     * @param projectName the name of the project
     * @return the users for the given project.
     * @throws RedmineServiceException if any pb while retriving datas
     * @see User
     * @deprecated since 1.0.3, use now {@link #getProjectMembers(String)}.
     */
    @Deprecated
    User[] getUsers(String projectName) throws RedmineServiceException;

    /**
     * Obtain all users defined on a given project.
     *
     * @param projectName the name of the project
     * @return the users for the given project.
     * @throws RedmineServiceException if any pb while retriving datas
     * @see User
     */
    User[] getProjectMembers(String projectName) throws RedmineServiceException;

    /**
     * Obtain all versions defined on a given project.
     *
     * @param projectName the name of the project
     * @return the versions of the given project.
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Version
     */
    Version[] getVersions(String projectName) throws RedmineServiceException;

    /**
     * Obtain a specific version for a given project.
     *
     * @param projectName the name of the project
     * @param versionName the name of the version
     * @return the version
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Version
     */
    Version getVersion(String projectName, String versionName) throws RedmineServiceException;

    /**
     * Obtain all issues for a given project
     *
     * @param projectName the name of the project
     * @return the issues
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Issue
     * @since 1.0.3
     */
    Issue[] getIssues(String projectName) throws RedmineServiceException;

    /**
     * Obtain all opened issues for a given project
     *
     * @param projectName the name of the project
     * @return the issues
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Issue
     * @since 1.0.3
     */
    Issue[] getOpenedIssues(String projectName) throws RedmineServiceException;

    /**
     * Obtain all closed issues for a given project
     *
     * @param projectName the name of the project
     * @return the issues
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Issue
     * @since 1.0.3
     */
    Issue[] getClosedIssues(String projectName) throws RedmineServiceException;

    /**
     * Obtain all issues for a specific version on a given project
     *
     * @param projectName the name of the project
     * @param versionName the name of the version
     * @return the issues
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Issue
     */
    Issue[] getIssues(String projectName, String versionName) throws RedmineServiceException;

    /**
     * Obtain for a given issue of a given project all the time entries.
     *
     * @param projectName the name of the project
     * @param issueId     the id of the issue
     * @return the time entries of the issue
     * @throws RedmineServiceException if any pb while retrieving time entries of the issue
     * @since 1.0.3
     */
    TimeEntry[] getIssueTimeEntries(String projectName, String issueId) throws RedmineServiceException;

    /**
     * Obtain all attachments for a specific version on a given project
     *
     * @param projectName the name of the project
     * @param versionName the name of the version
     * @return the attachments
     * @throws RedmineServiceException if any pb while retriving datas
     * @see Attachment
     */
    Attachment[] getAttachments(String projectName, String versionName) throws RedmineServiceException;
}
