package jaxx.runtime;

import java.awt.Container;

/**
 * The {@link jaxx.runtime.JAXXContext} contract defines a generic context.
 * <p/>
 * A context contains two king of entries :
 * <p/>
 * <h2>Unamed entry</h2>
 * a such entry maps filter only on the clas of the object of the entry.
 * <p/>
 * To add a <b>unamed</b> entry, use {@link #setContextValue(Object)} and {@link #getContextValue(Class)} to reteave a
 * such entry.
 * <p/>
 * <h2>named entry</h2>
 * a such entry filter on class of the object and on the name of the entry.
 * <p/>
 * To add a <b>named</b> entry, use {@link #setContextValue(Object,String)} and {@link #getContextValue(Class,String)}
 * to reteave a such entry.
 *
 * @author letellier
 * @author chemit
 */
public interface JAXXContext {

    /**
     * Push in the context a new unamed entry.
     * <p/>
     * If a previous entry exists in context (unamed  and same class), it will be removed.
     *
     * @param o the value to push in context
     */
    public <T> void setContextValue(T o);

    /**
     * * Push in the context a new amed entry.
     * <p/>
     * If a previous entry exists in context (same name and class), it will be removed.
     *
     * @param o    the value to push in context
     * @param name the name of the new entry
     */
    public <T> void setContextValue(T o, String name);

    /**
     * Remove from context the value with the given klazz as an unamed entry
     *
     * @param klazz the klazz entry
     */
    public <T> void removeContextValue(Class<T> klazz);

    /**
     * Remove from context the value with the given klazz as an unamed (if name is null) or named entry
     *
     * @param klazz the klazz entry
     * @param name  extra name of the entry
     */

    public <T> void removeContextValue(Class<T> klazz, String name);

    /**
     * Seek for a unamed entry in the context
     * <p/>
     * This is an exemple to call a method in JAXX :
     * <p/>
     * <code>&lt;JButton onActionPerformed='{getContextValue(Action.class).method(args[])}'/&gt;</code>
     *
     * @param clazz the class of unamed entry to seek in context
     * @return the value of the unamed entry for the given class, or <code>null</code> if no such entry.
     */
    public <T> T getContextValue(Class<T> clazz);

    /**
     * Seek for a named entry in the context
     *
     * @param clazz the class of named entry to seek in context
     * @param name  the name of the entry to seek in context
     * @return the value of the named entry for the given class, or <code>null</code> if no such entry.
     */
    public <T> T getContextValue(Class<T> clazz, String name);

    /**
     * Return parent's container corresponding to the Class clazz
     *
     * @param clazz clazz desired
     * @return parent's container
     */
    public <O extends Container> O getParentContainer(Class<O> clazz);

    /**
     * Return parent's container corresponding to the Class clazz
     *
     * @param top   the top container
     * @param clazz desired
     * @return parent's container
     */
    public <O extends Container> O getParentContainer(Object top, Class<O> clazz);
}
