/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.tags.swing;

import jaxx.CompilerException;
import jaxx.compiler.CompiledObject;
import jaxx.compiler.JAXXCompiler;
import jaxx.reflect.ClassDescriptor;
import jaxx.reflect.ClassDescriptorLoader;
import jaxx.tags.DefaultComponentHandler;
import org.w3c.dom.Element;

import javax.swing.JSpinner;
import javax.swing.event.ChangeListener;

public class JSpinnerHandler extends DefaultComponentHandler {
    public static String MINIMUM_PROPERTY = "minimum";
    public static String MAXIMUM_PROPERTY = "maximum";
    public static String VALUE_PROPERTY = "value";

    public JSpinnerHandler(ClassDescriptor beanClass) {
        super(beanClass);
        ClassDescriptorLoader.checkSupportClass(getClass(), beanClass, JSpinner.class);
    }

    public static class CompiledSpinner extends CompiledObject {
        Integer minimum = null;
        Integer maximum = null;
        Integer value = null;

        public CompiledSpinner(String id, ClassDescriptor objectClass, JAXXCompiler compiler) throws CompilerException {
            super(id, objectClass, compiler);
        }
    }

    @Override
    public CompiledObject createCompiledObject(String id, JAXXCompiler compiler) throws CompilerException {
        return new CompiledSpinner(id, getBeanClass(), compiler);
    }

    @Override
    protected void configureProxyEventInfo() {
        super.configureProxyEventInfo();
        addProxyEventInfo("getValue", ChangeListener.class, "model");
    }

    @Override
    public ClassDescriptor getPropertyType(CompiledObject object, String propertyName, JAXXCompiler compiler) throws CompilerException {
        if (propertyName.equals(MINIMUM_PROPERTY) || propertyName.equals(MAXIMUM_PROPERTY) ||
                propertyName.equals(VALUE_PROPERTY)) {
            return ClassDescriptorLoader.getClassDescriptor(Integer.class);
        }
        return super.getPropertyType(object, propertyName, compiler);
    }

    @Override
    public void setProperty(CompiledObject object, String name, Object value, JAXXCompiler compiler) throws CompilerException {
        if (name.equals(MINIMUM_PROPERTY)) {
            ((CompiledSpinner) object).minimum = (Integer) value;
        } else if (name.equals(MAXIMUM_PROPERTY)) {
            ((CompiledSpinner) object).maximum = (Integer) value;
        } else if (name.equals(VALUE_PROPERTY)) {
            ((CompiledSpinner) object).value = (Integer) value;
        } else {
            super.setProperty(object, name, value, compiler);
        }
    }

    @Override
    protected void closeComponent(CompiledObject object, Element tag, JAXXCompiler compiler) throws CompilerException {
        CompiledSpinner spinner = (CompiledSpinner) object;
        if (spinner.minimum != null || spinner.maximum != null || spinner.value != null) {
            if (spinner.getConstructorParams() != null) {
                compiler.reportError("constructorParams and minimum/maximum may not both be specified for the same JSpinner");
            }
            if (spinner.minimum == null) {
                spinner.minimum = Math.min(0, spinner.maximum != null ? spinner.maximum.intValue() : 0);
            }
            if (spinner.maximum == null) {
                spinner.maximum = Math.max(100, spinner.minimum.intValue());
            }
            if (spinner.value == null) {
                spinner.value = spinner.minimum;
            }
            spinner.setConstructorParams("new SpinnerNumberModel(" + spinner.value + ", " + spinner.minimum + ", " + spinner.maximum + ", 1)");
        }

        super.closeComponent(object, tag, compiler);
    }
}