/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.compiler;

import jaxx.CompilerException;
import jaxx.UnsupportedAttributeException;
import jaxx.UnsupportedTagException;
import jaxx.css.Rule;
import jaxx.css.Stylesheet;
import jaxx.css.StylesheetHelper;
import jaxx.parser.ParseException;
import jaxx.reflect.ClassDescriptor;
import jaxx.reflect.ClassDescriptorLoader;
import jaxx.reflect.FieldDescriptor;
import jaxx.reflect.MethodDescriptor;
import jaxx.runtime.ComponentDescriptor;
import jaxx.runtime.JAXXObjectDescriptor;
import jaxx.tags.DefaultObjectHandler;
import jaxx.tags.TagHandler;
import jaxx.tags.TagManager;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.AttributesImpl;
import org.xml.sax.helpers.XMLFilterImpl;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.ErrorListener;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.sax.SAXSource;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Modifier;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Compiles JAXX files into Java classes.
 * <p/>
 * use {@link Generator} ... todo finish javadoc
 */
public class JAXXCompiler {

    /** log */
    protected static final Log log = LogFactory.getLog(JAXXCompiler.class);

    /**
     * True to throw exceptions when we encounter unresolvable classes, false to ignore.
     * This is currently set to false until JAXX has full support for inner classes
     * (including enumerations), because currently they don't always resolve (but will
     * generally compile without error anyway).
     */
    public static final boolean STRICT_CHECKS = false;
    public static final String JAXX_NAMESPACE = "http://www.jaxxframework.org/";
    public static final String JAXX_INTERNAL_NAMESPACE = "http://www.jaxxframework.org/internal";

    /** Maximum length of an inlinable creation method. */
    public static final int INLINE_THRESHOLD = 300;

    protected final DefaultObjectHandler firstPassClassTagHandler;

    protected List<String> staticImports = new ArrayList<String>();

    /*---------------------------------------------------------------------------------*/
    /*-- compiler fields --------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    /** flag to detec if an error occurs while compiling jaxx file */
    protected boolean failed;

    /** Object corresponding to the root tag in the document. */
    protected CompiledObject root;

    /** Contains strings of the form "javax.swing." */
    protected Set<String> importedPackages = new HashSet<String>();

    /** Contains strings of the form "javax.swing.Timer" */
    protected Set<String> importedClasses = new HashSet<String>();

    /** Keeps track of open components (components still having children added). */
    protected Stack<CompiledObject> openComponents = new Stack<CompiledObject>();

    /** Sequence number used to create automatic variable names. */
    protected int autogenID = 0;

    protected List<DataBinding> dataBindings = new ArrayList<DataBinding>();

    protected SymbolTable symbolTable = new SymbolTable();

    /** Base directory used for path resolution (normally the directory in which the .jaxx file resides). */
    protected File baseDir;

    /** .jaxx file being compiled. */
    protected File src;

    /** Parsed XML of src file. */
    protected Document document;

    /** Name of class being compiled. */
    protected String outputClassName;

    protected ScriptManager scriptManager = new ScriptManager(this);

    /** Combination of all stylesheets registered using {@link #registerStylesheet}. */
    protected Stylesheet stylesheet;

    /** Contains all attributes defined inline on class tags. */
    protected List<Rule> inlineStyles = new ArrayList<Rule>();

    /**
     * Maps objects (expressed in Java code) to event listener classes (e.g. MouseListener) to Lists of EventHandlers.  The final list
     * contains all event  handlers of a particular type attached to a particular object (again, as represented by a Java expression).
     */
    protected Map<String, Map<ClassDescriptor, List<EventHandler>>> eventHandlers = new HashMap<String, Map<ClassDescriptor, List<EventHandler>>>();

    /** Maps of uniqued id for objects used in compiler */
    protected Map<Object, String> uniqueIds = new HashMap<Object, String>();

    /** Map of event handler method names used in compiler */
    protected Map<EventHandler, String> eventHandlerMethodNames = new HashMap<EventHandler, String>();

    /** ClassLoader which searches the user-specified class path in addition to the normal class path */
    protected ClassLoader classLoader;

    /**
     * A list of Runnables which will be run after the first compilation pass.  This is primarily used
     * to trigger the creation of CompiledObjects, which cannot be created during the first pass and must be
     * created in document order.
     */
    protected List<Runnable> initializers = new ArrayList<Runnable>();

    /** left brace matcher */
    protected Matcher leftBraceMatcher = Pattern.compile("^(\\{)|[^\\\\](\\{)").matcher("");

    /** right brace matcher */
    protected Matcher rightBraceMatcher = Pattern.compile("^(\\})|[^\\\\](\\})").matcher("");

    /** extra interfaces which can by passed to root object via the 'implements' attribute */
    private String[] extraInterfaces;

    /** a flag to generate a abstract class */
    private boolean abstractClass;

    /** the possible generic type of the class */
    private String genericType;

    /** thepossible generic type of the super class */
    private String superGenericType;

    /** Extra code to be added to the instance initializer. */
    protected StringBuffer initializer = new StringBuffer();

    /** Extra code to be added at the end of the instance initializer. */
    protected StringBuffer lateInitializer = new StringBuffer();

    /** Extra code to be added to the class body. */
    protected StringBuffer bodyCode = new StringBuffer();

    /** Code to initialize data bindings. */
    protected StringBuffer initDataBindings = new StringBuffer();

    /** Body of the applyDataBinding method. */
    protected StringBuffer applyDataBinding = new StringBuffer();

    /** Body of the removeDataBinding method. */
    protected StringBuffer removeDataBinding = new StringBuffer();

    /** Body of the processDataBinding method. */
    protected StringBuffer processDataBinding = new StringBuffer();

    /** true if a main() method has been declared in a script */
    protected boolean mainDeclared;

    /** the file to be generated */
    protected JavaFile javaFile;


    protected CompilerOptions options;

    /** Used for error reporting purposes, so we can report the right line number. */
    protected Stack<Element> tagsBeingCompiled = new Stack<Element>();

    /** Used for error reporting purposes, so we can report the right source file. */
    protected Stack<File> sourceFiles = new Stack<File>();

    /** Maps object ID strings to the objects themselves.  These are created during the second compilation pass. */
    protected Map<String, CompiledObject> objects = new LinkedHashMap<String, CompiledObject>();

    /** Maps objects to their ID strings.  These are created during the second compilation pass. */
    protected Map<CompiledObject, String> ids = new LinkedHashMap<CompiledObject, String>();

    protected CompiledObjectDecorator defaultDecorator;
    
    /*---------------------------------------------------------------------------------*/
    /*-- Constructor methods ----------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    protected JAXXCompiler(ClassLoader classLoader, DefaultObjectHandler firstPassClassTagHandler, String... staticImports) {
        this.firstPassClassTagHandler = firstPassClassTagHandler;
        this.options = new CompilerOptions();
        this.classLoader = classLoader;
        if (staticImports == null || staticImports.length == 0) {
            staticImports = new String[0];
        }
        this.staticImports = Arrays.asList(staticImports);
        addImport("java.lang.*");
    }

    /**
     * Creates a new JAXXCompiler.
     *
     * @param baseDir                  classpath location
     * @param src                      location of file to compile
     * @param outputClassName          the out file name
     * @param options                  options to pass to javac
     * @param firstPassClassTagHandler handler to use for first pass
     * @param staticImports            statics imports
     */
    protected JAXXCompiler(File baseDir, File src, String outputClassName, CompilerOptions options, DefaultObjectHandler firstPassClassTagHandler, String... staticImports) {
        this.baseDir = baseDir;
        this.src = src;
        this.firstPassClassTagHandler = firstPassClassTagHandler;
        this.staticImports = Arrays.asList(staticImports);
        sourceFiles.push(src);
        this.outputClassName = outputClassName;
        this.options = options;
        addImport(outputClassName.substring(0, outputClassName.lastIndexOf(".") + 1) + "*");
        if (staticImports == null || staticImports.length == 0) {
            staticImports = new String[]{"java.lang.*"};
        }
        for (Object staticImport : staticImports) {
            addImport((String) staticImport);
        }
        // add extra imports from options
        if (options.getExtraImports() != null) {
            for (String extraImport : options.getExtraImports()) {
                addImport(extraImport);
            }
        }
        defaultDecorator = CompiledObjectDecorator.getDecorator(options.getDefaultDecoratorClass());
        if (defaultDecorator == null) {
            log.error("could not find default decorator : "+options.getDefaultDecoratorClass());
            throw new IllegalArgumentException("could not find default decorator : " + options.getDefaultDecoratorClass());
        }
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Initializer methods -----------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    public void runInitializers() {
        for (Runnable runnable : initializers) {
            if (log.isDebugEnabled()) {
                log.debug(runnable);
            }
            try {
                runnable.run();
            } catch (Exception e) {
                //TC - 20081018 report error and quit
                reportError(e.getMessage());
                return;
            }
        }
        initializers.clear();
    }


    /**
     * Registers a <code>Runnable</code> which will be executed after the first
     * compilation pass is complete.
     *
     * @param r runnable to register
     */
    public void registerInitializer(Runnable r) {
        initializers.add(r);
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Compile methods --------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    public void compileFirstPass(final Element tag) throws IOException {
        tagsBeingCompiled.push(tag);

        String namespace = tag.getNamespaceURI();
        String fullClassName;
        String localName = tag.getLocalName();
        boolean namespacePrefix = tag.getPrefix() != null;
        // resolve class tags into fully-qualified class name
        if (namespace != null && namespace.endsWith("*")) {
            String packageName = namespace.substring(0, namespace.length() - 1);
            if (localName.startsWith(packageName)) // class name is fully-qualified already
            {
                fullClassName = TagManager.resolveClassName(localName, this);
            } else { // namespace not included in class name, probably need the namespace to resolve
                fullClassName = TagManager.resolveClassName(packageName + localName, this);
                if (fullClassName == null && !namespacePrefix) // it was just a default namespace, try again without using the namespace
                {
                    fullClassName = TagManager.resolveClassName(localName, this);
                }
            }
        } else {
            fullClassName = TagManager.resolveClassName(localName, this);
        }

        if (fullClassName != null) { // we are definitely dealing with a class tag
            addDependencyClass(fullClassName);
            namespace = fullClassName.substring(0, fullClassName.lastIndexOf(".") + 1) + "*";
            if (symbolTable.getSuperclassName() == null) {
                symbolTable.setSuperclassName(fullClassName);
            }
            String id = tag.getAttribute("id");
            if (id.length() > 0) {
                symbolTable.getClassTagIds().put(id, fullClassName);
                if (tag.getAttributeNode("javaBean") != null) {
                    // add java bean support for this property
                    String capitalizeName = org.apache.commons.lang.StringUtils.capitalize(id);
                    // add method
                    symbolTable.getScriptMethods().add(new MethodDescriptor("get" + capitalizeName, Modifier.PUBLIC, fullClassName, new String[0], classLoader));
                    if (Boolean.class.getName().equals(fullClassName)) {
                        symbolTable.getScriptMethods().add(new MethodDescriptor("is" + capitalizeName, Modifier.PUBLIC, fullClassName, new String[0], classLoader));
                    }
                    symbolTable.getScriptMethods().add(new MethodDescriptor("set" + capitalizeName, Modifier.PUBLIC, "void", new String[]{fullClassName}, classLoader));
                }
            }
        }
        // during the first pass, we can't create ClassDescriptors for JAXX files because they may not have been processed yet
        // (and we can't wait until they have been processed because of circular dependencies).  So we don't do any processing
        // during the first pass which requires having a ClassDescriptor;  here we determine whether we have a class tag or not
        // (class tag namespaces end in "*") and use a generic handler if so.  The real handler is used during the second pass.
        TagHandler handler = (namespace != null && namespace.endsWith("*")) ? firstPassClassTagHandler : TagManager.getTagHandler(tag.getNamespaceURI(), localName, namespacePrefix, this);
        if (handler != firstPassClassTagHandler && handler instanceof DefaultObjectHandler) {
            fullClassName = ((DefaultObjectHandler) handler).getBeanClass().getName();
            //namespace = fullClassName.substring(0, fullClassName.lastIndexOf(".") + 1) + "*";
            handler = firstPassClassTagHandler;
        }
        if (handler == firstPassClassTagHandler) {
            final String finalClassName = fullClassName;
            registerInitializer(new Runnable() { // register an initializer which will create the CompiledObject after pass 1

                public void run() {
                    DefaultObjectHandler handler = (DefaultObjectHandler) TagManager.getTagHandler(null, finalClassName, JAXXCompiler.this);
                    if (handler == null) {
                        throw new CompilerException("Internal error: missing TagHandler for '" + finalClassName + "'");
                    }
                    handler.registerCompiledObject(tag, JAXXCompiler.this);
                }
            });
        }
        if (handler != null) {
            try {
                handler.compileFirstPass(tag, this);
            }
            catch (CompilerException e) {
                reportError(e);
            }
        } else {
            reportError("Could not find a Java class corresponding to: <" + tag.getTagName() + ">");
            failed = true;
        }

        Element finished = tagsBeingCompiled.pop();
        if (finished != tag) {
            throw new RuntimeException("internal error: just finished compiling " + tag + ", but top of tagsBeingCompiled stack is " + finished);
        }
    }


    public void compileSecondPass(Element tag) throws IOException {
        tagsBeingCompiled.push(tag);

        TagHandler handler = TagManager.getTagHandler(tag.getNamespaceURI(), tag.getLocalName(), tag.getPrefix() != null, this);
        if (handler != null) {
            handler.compileSecondPass(tag, this);
        } else {
            reportError("Could not find a Java class corresponding to: <" + tag.getTagName() + ">");
            assert false : "can't-happen error: error should have been reported during the fast pass and caused an abort";
            failed = true;
        }

        Element finished = tagsBeingCompiled.pop();
        if (finished != tag) {
            throw new RuntimeException("internal error: just finished compiling " + tag + ", but top of tagsBeingCompiled stack is " + finished);
        }

    }

    protected void compileFirstPass() throws IOException {
        try {
            InputStream in = new FileInputStream(src);
            document = parseDocument(in);
            in.close();
            compileFirstPass(document.getDocumentElement());
        }
        catch (SAXParseException e) {
            reportError(e.getLineNumber(), "Invalid XML: " + e.getMessage());
        }
        catch (SAXException e) {
            reportError(null, "Error parsing XML document: " + e);
        }
    }

    protected void compileSecondPass() throws IOException {
        if (!tagsBeingCompiled.isEmpty()) {
            throw new RuntimeException("Internal error: starting pass two, but tagsBeingCompiled is not empty: " + tagsBeingCompiled);
        }
        compileSecondPass(document.getDocumentElement());
    }

    /*---------------------------------------------------------------------------------*/
    /*-- CompiledObject methods -------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    public void openComponent(CompiledObject component) throws CompilerException {
        openComponent(component, null);
    }

    public void openComponent(CompiledObject component, String constraints) throws CompilerException {
        CompiledObject parent = getOpenComponent();
        openInvisibleComponent(component);
        if (parent != null && !component.isOverride()) {
            parent.addChild(component, constraints, this);
        }
    }

    public void openInvisibleComponent(CompiledObject component) {
        if (!ids.containsKey(component)) {
            registerCompiledObject(component);
        }
        openComponents.push(component);
    }

    public CompiledObject getOpenComponent() {
        if (openComponents.isEmpty()) {
            return null;
        }
        return openComponents.peek();
    }

    public void closeComponent(CompiledObject component) {
        if (openComponents.pop() != component) {
            throw new IllegalArgumentException("can only close the topmost open object");
        }
    }

    public void registerCompiledObject(CompiledObject object) {
        assert JAXXCompilerLaunchor.get().symbolTables.values().contains(symbolTable) : "attempting to register CompiledObject before pass 1 is complete";
        if (root == null) {
            root = object;
        }

        String id = object.getId();
        if (ids.containsKey(object)) {
            reportError("object '" + object + "' is already registered with id '" + ids.get(object) + "', cannot re-register as '" + id + "'");
        }
        if (objects.containsKey(id) && !(objects.get(id) instanceof Element)) {
            reportError("id '" + id + "' is already registered to component " + objects.get(id));
        }
        objects.put(id, object);
        ids.put(object, id);
        if (object.getDecorator() == null) {
            // use compiler decorator
            object.setDecorator(defaultDecorator);
        }
    }

    public CompiledObject getCompiledObject(String id) {
        runInitializers();
        assert JAXXCompilerLaunchor.get().symbolTables.values().contains(symbolTable) : "attempting to retrieve CompiledObject before pass 1 is complete";
        return objects.get(id);
    }


    public boolean inlineCreation(CompiledObject object) {
        return object.getId().startsWith("$") && object.getInitializationCode(this).length() < INLINE_THRESHOLD;
    }

    public void checkOverride(CompiledObject object) throws CompilerException {
        if (object.getId().startsWith("$")) {
            return;
        }
        ClassDescriptor ancestor = root.getObjectClass();
        if (ancestor == object.getObjectClass()) {
            return;
        }
        while (ancestor != null) {
            try {
                FieldDescriptor f = ancestor.getDeclaredFieldDescriptor(object.getId());
                if (!f.getType().isAssignableFrom(object.getObjectClass())) {
                    reportError("attempting to redefine superclass member '" + object.getId() + "' as incompatible type (was " + f.getType() + ", redefined as " + object.getObjectClass() + ")");
                }
                object.setOverride(true);
                object.setOverrideType(f.getType());
                break;
            }
            catch (NoSuchFieldException e) {
                ancestor = ancestor.getSuperclass();
            }
        }
    }

    /*---------------------------------------------------------------------------------*/
    /*-- DataBinding methods ----------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    /**
     * Examine an attribute value for data binding expressions.  Returns a 'cooked' expression which
     * can be used to determine the resulting value.  It is expected that this expression will be used
     * as the source expression in a call to {@link #registerDataBinding}.
     * If the attribute value does not invoke data binding, this method returns <code>null</code>
     *
     * @param stringValue the string value of the property from the XML
     * @param type        the type of the property, from the <code>JAXXPropertyDescriptor</code>
     * @return a processed version of the expression
     * @throws jaxx.CompilerException ?
     */
    public String processDataBindings(String stringValue, ClassDescriptor type) throws CompilerException {
        int pos = getNextLeftBrace(stringValue, 0);
        if (pos != -1) {
            StringBuffer expression = new StringBuffer();
            int lastPos = 0;
            while (pos != -1 && pos < stringValue.length()) {
                if (pos > lastPos) {
                    if (expression.length() > 0) {
                        expression.append(" + ");
                    }
                    expression.append('"');
                    expression.append(escapeJavaString(stringValue.substring(lastPos, pos)));
                    expression.append('"');
                }

                if (expression.length() > 0) {
                    expression.append(" + ");
                }
                expression.append('(');
                int pos2 = getNextRightBrace(stringValue, pos + 1);
                if (pos2 == -1) {
                    reportError("unmatched '{' in expression: " + stringValue);
                    return "";
                }
                expression.append(stringValue.substring(pos + 1, pos2));
                expression.append(')');
                pos2++;
                if (pos2 < stringValue.length()) {
                    pos = getNextLeftBrace(stringValue, pos2);
                    lastPos = pos2;
                } else {
                    pos = stringValue.length();
                    lastPos = pos;
                }
            }
            if (lastPos < stringValue.length()) {
                if (expression.length() > 0) {
                    expression.append(" + ");
                }
                expression.append('"');
                expression.append(escapeJavaString(stringValue.substring(lastPos)));
                expression.append('"');
            }
            return type == ClassDescriptorLoader.getClassDescriptor(String.class) ? "String.valueOf(" + expression + ")" : expression.toString();
        }
        return null;
    }


    public void registerDataBinding(String src, String dest, String assignment) {
        try {
            src = checkJavaCode(src);
            dataBindings.add(new DataBinding(src, dest, assignment, this));
        }
        catch (CompilerException e) {
            reportError("While parsing data binding for '" + dest.substring(dest.lastIndexOf(".") + 1) + "': " + e.getMessage());
        }
    }

    public void registerEventHandler(EventHandler handler) {
        String objectCode = handler.getObjectCode();
        Map<ClassDescriptor, List<EventHandler>> listeners = eventHandlers.get(objectCode);
        if (listeners == null) {
            listeners = new HashMap<ClassDescriptor, List<EventHandler>>();
            eventHandlers.put(objectCode, listeners);
        }
        ClassDescriptor listenerClass = handler.getListenerClass();
        List<EventHandler> handlerList = listeners.get(listenerClass);
        if (handlerList == null) {
            handlerList = new ArrayList<EventHandler>();
            listeners.put(listenerClass, handlerList);
        }
        handlerList.add(handler);
    }

    public String getEventHandlerMethodName(EventHandler handler) {
        String result = eventHandlerMethodNames.get(handler);
        if (result == null) {
            if (getOptions().isOptimize()) {
                result = "$ev" + eventHandlerMethodNames.size();
            } else {
                //TC-20090309 must get the goal property from the event id
                // to make possible inheritance
                String id = handler.getEventId().substring(0,handler.getEventId().indexOf("."));
                
                result = "do" + org.apache.commons.lang.StringUtils.capitalize(handler.getListenerMethod().getName()) + "__on__" + id;
                //result = "do" + org.apache.commons.lang.StringUtils.capitalize(handler.getListenerMethod().getName()) + "__on__" + handler.getObjectCode();
            }
            eventHandlerMethodNames.put(handler, result);
        }
        return result;
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Script methods ---------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    public void addScriptField(FieldDescriptor field) {
        symbolTable.getScriptFields().add(field);
    }

    public void addScriptMethod(MethodDescriptor method) {
        if (method.getName().equals("main") && method.getParameterTypes().length == 1 && method.getParameterTypes()[0].getName().equals("[Ljava.lang.String;")) {
            setMainDeclared(true);
        }
        symbolTable.getScriptMethods().add(method);
    }

    public void registerScript(String script) throws CompilerException {
        registerScript(script, null);
    }

    public void registerScript(String script, File sourceFile) throws CompilerException {
        if (sourceFile != null) {
            sourceFiles.push(sourceFile);
        }
        script = script.trim();
        if (!"".equals(script) && !script.endsWith("}") && !script.endsWith(";")) {
            script += ";";
        }
        scriptManager.registerScript(script);

        if (sourceFile != null) {
            File pop = sourceFiles.pop();
            if (pop != sourceFile) {
                throw new RuntimeException("leaving registerScript(), but " + sourceFile + " was not the top entry on the stack (found " + pop + " instead)");
            }
        }
    }

    public String preprocessScript(String script) throws CompilerException {
        return scriptManager.preprocessScript(script);
    }

    /*---------------------------------------------------------------------------------*/
    /*-- StyleSheet methods -----------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    public void applyStylesheets() {
        for (Object o : new ArrayList<CompiledObject>(objects.values())) {
            CompiledObject object = (CompiledObject) o;
            TagManager.getTagHandler(object.getObjectClass()).applyStylesheets(object, this);
        }
    }

    public void registerStylesheet(Stylesheet stylesheet) {
        if (this.stylesheet == null) {
            this.stylesheet = stylesheet;
        } else {
            this.stylesheet.add(stylesheet.getRules());
        }
    }

    public void addInlineStyle(CompiledObject object, String propertyName, boolean dataBinding) {
        inlineStyles.add(StylesheetHelper.inlineAttribute(object, propertyName, dataBinding));
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Report methods ---------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    public void reportWarning(String warning) {
        Element currentTag = null;
        if (!tagsBeingCompiled.isEmpty()) {
            currentTag = tagsBeingCompiled.peek();
        }
        reportWarning(currentTag, warning, 0);
    }


    public void reportWarning(Element tag, String warning, int lineOffset) {
        String lineNumber = null;
        if (tag != null) {
            String lineAttr = tag.getAttributeNS(JAXX_INTERNAL_NAMESPACE, "line");
            if (lineAttr.length() > 0) {
                lineNumber = lineAttr;
            }
        }
        File src = sourceFiles.peek();
        try {
            src = src.getCanonicalFile();
        }
        catch (IOException e) {
            // ignore ?
        }

        System.err.print(src);
        if (lineNumber != null) {
            System.err.print(":" + ((sourceFiles.size() == 1) ? Integer.parseInt(lineNumber) + lineOffset : lineOffset + 1));
        }
        System.err.println(": Warning: " + warning);
        JAXXCompilerLaunchor.get().warningCount++;
    }


    public void reportError(String error) {
        Element currentTag = null;
        if (!tagsBeingCompiled.isEmpty()) {
            currentTag = tagsBeingCompiled.peek();
        }
        reportError(currentTag, error);
    }

    public void reportError(CompilerException ex) {
        reportError(null, ex);
    }

    public void reportError(String extraMessage, CompilerException ex) {
        String message = ex.getMessage();
        if (ex.getClass() == UnsupportedAttributeException.class || ex.getClass() == UnsupportedTagException.class) {
            message = ex.getClass().getName().substring(ex.getClass().getName().lastIndexOf(".") + 1) + ": " + message;
        }
        int lineOffset;
        if (ex instanceof ParseException) {
            lineOffset = Math.max(0, ((ParseException) ex).getLine() - 1);
        } else {
            lineOffset = 0;
        }
        Element currentTag = null;
        if (!tagsBeingCompiled.isEmpty()) {
            currentTag = tagsBeingCompiled.peek();
        }
        reportError(currentTag, extraMessage != null ? extraMessage + message : message, lineOffset);
    }

    public void reportError(Element tag, String error) {
        reportError(tag, error, 0);
    }

    public void reportError(Element tag, String error, int lineOffset) {
        int lineNumber = 0;
        if (tag != null) {
            String lineAttr = tag.getAttributeNS(JAXX_INTERNAL_NAMESPACE, "line");
            if (lineAttr.length() > 0) {
                lineNumber = Integer.parseInt(lineAttr);
            }
        }
        lineNumber = Math.max(lineNumber, 1) + lineOffset;
        reportError(lineNumber, error);
    }

    public void reportError(int lineNumber, String error) {
        File src = sourceFiles.isEmpty() ? null : sourceFiles.peek();
        try {
            if (src != null) {
                src = src.getCanonicalFile();
            }
        }
        catch (IOException e) {
            // ignore ?
        }

        System.err.print(src != null ? src.getPath() : "<unknown source>");
        if (lineNumber > 0) {
            System.err.print(":" + lineNumber);
        }
        System.err.println(": " + error);
        JAXXCompilerLaunchor.get().errorCount++;
        failed = true;
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Getter methods ---------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    public Map<String, CompiledObject> getObjects() {
        return objects;
    }

    public List<DataBinding> getDataBindings() {
        return dataBindings;
    }

    /*public List<CompiledBeanValidator> getValidators() {
        return validators;
    }*/

    public Map<String, Map<ClassDescriptor, List<EventHandler>>> getEventHandlers() {
        return eventHandlers;
    }

    public CompilerOptions getOptions() {
        return options;
    }

    public String getOutputClassName() {
        return outputClassName;
    }

    public File getBaseDir() {
        return baseDir;
    }

    public Set<String> getImportedClasses() {
        return importedClasses;
    }

    public Set<String> getImportedPackages() {
        return importedPackages;
    }

    public Iterator<CompiledObject> getObjectCreationOrder() {
        return objects.values().iterator();
    }

    public CompiledObject getRootObject() {
        return root;
    }

    public Stack<File> getSourceFiles() {
        return sourceFiles;
    }

    public ScriptManager getScriptManager() {
        return scriptManager;
    }

    public SymbolTable getSymbolTable() {
        return symbolTable;
    }

    public Stylesheet getStylesheet() {
        Stylesheet merged = new Stylesheet();
        if (stylesheet != null) {
            merged.add(stylesheet.getRules());
        }
        merged.add(inlineStyles.toArray(new Rule[inlineStyles.size()]));
        return merged;
    }

    public FieldDescriptor[] getScriptFields() {
        List<FieldDescriptor> scriptFields = symbolTable.getScriptFields();
        return scriptFields.toArray(new FieldDescriptor[scriptFields.size()]);
    }

    public MethodDescriptor[] getScriptMethods() {
        List<MethodDescriptor> scriptMethods = symbolTable.getScriptMethods();
        return scriptMethods.toArray(new MethodDescriptor[scriptMethods.size()]);
    }

    public MethodDescriptor getScriptMethod(String methodName) {
        for (MethodDescriptor m :symbolTable.getScriptMethods()) {
           if (methodName.equals(m.getName()))  {
               return m;
           }
        }
        return null;
    }

    public boolean isFailed() {
        return failed;
    }

    /**
     * Returns a <code>ClassLoader</code> which searches the user-specified class path in addition
     * to the normal system class path.
     *
     * @return <code>ClassLoader</code> to use while resolving class references
     */
    public ClassLoader getClassLoader() {
        if (classLoader == null) {
            if (options.getClassLoader() != null) {
                classLoader = options.getClassLoader();
            } else {
                String classPath = options.getClassPath();
                if (classPath == null) {
                    classPath = ".";
                }
                String[] paths = classPath.split(File.pathSeparator);
                URL[] urls = new URL[paths.length];
                for (int i = 0; i < paths.length; i++) {
                    try {
                        urls[i] = new File(paths[i]).toURI().toURL();
                    }
                    catch (MalformedURLException e) {
                        throw new RuntimeException(e);
                    }
                }
                classLoader = new URLClassLoader(urls, getClass().getClassLoader());
            }
        }

        return classLoader;
    }

    public JAXXObjectDescriptor getJAXXObjectDescriptor() {
        runInitializers();
        CompiledObject[] components = new ArrayList<CompiledObject>(objects.values()).toArray(new CompiledObject[objects.size()]);
        assert initializers.isEmpty() : "there are pending initializers remaining";
        assert root != null : "root object has not been defined";
        assert Arrays.asList(components).contains(root) : "root object is not registered";
        ComponentDescriptor[] descriptors = new ComponentDescriptor[components.length];
        // as we print, sort the array so that component's parents are always before the components themselves
        for (int i = 0; i < components.length; i++) {
            CompiledObject parent = components[i].getParent();
            while (parent != null) {
                boolean found = false;
                for (int j = i + 1; j < components.length; j++) { // found parent after component, swap them
                    if (components[j] == parent) {
                        components[j] = components[i];
                        components[i] = parent;
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    break;
                }
                parent = components[i].getParent();
            }
            int parentIndex = -1;
            if (parent != null) {
                for (int j = 0; j < i; j++) {
                    if (components[j] == parent) {
                        parentIndex = j;
                        break;
                    }
                }
            }
            descriptors[i] = new ComponentDescriptor(components[i].getId(), components[i] == root ? outputClassName : components[i].getObjectClass().getName(),
                    components[i].getStyleClass(), parentIndex != -1 ? descriptors[parentIndex] : null);
        }

        Stylesheet stylesheet = getStylesheet();
        if (stylesheet == null) {
            stylesheet = new Stylesheet();
        }

        return new JAXXObjectDescriptor(descriptors, stylesheet);
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Setter methods ---------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    public void setFailed(boolean failed) {
        this.failed = failed;
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Buffer ------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    public StringBuffer getInitializer() {
        return initializer;
    }

    public StringBuffer getLateInitializer() {
        return lateInitializer;
    }

    public StringBuffer getBodyCode() {
        return bodyCode;
    }

    public StringBuffer getInitDataBindings() {
        return initDataBindings;
    }

    public StringBuffer getApplyDataBinding() {
        return applyDataBinding;
    }

    public StringBuffer getRemoveDataBinding() {
        return removeDataBinding;
    }

    public StringBuffer getProcessDataBinding() {
        return processDataBinding;
    }

    public boolean isMainDeclared() {
        return mainDeclared;
    }

    public void setMainDeclared(boolean mainDeclared) {
        this.mainDeclared = mainDeclared;
    }

    public void appendInitializerCode(String code) {
        getInitializer().append(code);
    }

    public void appendBodyCode(String code) {
        getBodyCode().append(code);
    }

    public void appendInitDataBindings(String code) {
        getInitDataBindings().append(code);
    }

    public void appendProcessDataBinding(String code) {
        getProcessDataBinding().append(code);
    }

    public void appendApplyDataBinding(String code) {
        getApplyDataBinding().append(code);
    }

    public void appendRemoveDataBinding(String code) {
        getRemoveDataBinding().append(code);
    }

    public void appendLateInitializer(String code) {
        getLateInitializer().append(code);
    }

    public boolean haveProcessDataBinding() {
        return getProcessDataBinding().length() > 0;
    }

    public boolean haveApplyDataBinding() {
        return getApplyDataBinding().length() > 0;
    }

    public boolean haveRemoveDataBinding() {
        return getRemoveDataBinding().length() > 0;
    }

    public void addMethodToJavaFile(JavaMethod method) {
        getJavaFile().addMethod(method);
    }

    public boolean hasMethod(String methodName) {
        JavaMethod[] methods = getJavaFile().getMethods();
        for (JavaMethod method : methods) {
            if (methodName.equals(method.getName())) {
                return true;
            }
        }
        return false;
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Other methods ----------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    public void addImport(String text) {
        if (text.endsWith("*")) {
            importedPackages.add(text.substring(0, text.length() - 1));
        } else {
            importedClasses.add(text);
        }

        if (!text.equals("*")) {
            getJavaFile().addImport(text);
        }
    }

    public void addDependencyClass(String className) {
        if (!JAXXCompilerLaunchor.get().jaxxFileClassNames.contains(className)) {
            URL jaxxURL = getClassLoader().getResource(className.replace('.', '/') + ".jaxx");
            URL classURL = getClassLoader().getResource(className.replace('.', '/') + ".class");
            if (jaxxURL != null && classURL != null) {
                try {
                    File jaxxFile = URLtoFile(jaxxURL);
                    File classFile = URLtoFile(classURL);
                    if (classFile.lastModified() > jaxxFile.lastModified()) {
                        return; // class file is newer, no need to recompile
                    }
                }
                catch (Exception e) {
                    // do nothing
                }
            }

            if (jaxxURL != null && jaxxURL.toString().startsWith("file:")) {
                File jaxxFile = URLtoFile(jaxxURL);
                try {
                    jaxxFile = jaxxFile.getCanonicalFile();
                }
                catch (IOException ex) {
                    // ignore ?
                }
                assert jaxxFile.getName().equalsIgnoreCase(className.substring(className.lastIndexOf(".") + 1) + ".jaxx") :
                        "expecting file name to match " + className + ", but found " + jaxxFile.getName();
                if (jaxxFile.getName().equals(className.substring(className.lastIndexOf(".") + 1) + ".jaxx")) { // check case match
                    if (JAXXCompilerLaunchor.get().currentPass != JAXXCompilerLaunchor.LifeCycle.compile_first_pass) {
                        throw new AssertionError("Internal error: adding dependency class " + className + " during second compilation pass");
                    }
                    JAXXCompilerLaunchor.get().jaxxFileClassNames.add(className);
                    JAXXCompilerLaunchor.get().jaxxFiles.add(jaxxFile);
                }
            }
        }
    }

    /**
     * Verifies that a snippet of Java code parses correctly.  A warning is generated if the string has enclosing
     * curly braces.
     *
     * @param javaCode the Java code snippet to test
     * @return a "cooked" version of the string which has enclosing curly braces removed.
     * @throws jaxx.CompilerException if the code cannot be parsed
     */
    public String checkJavaCode(String javaCode) {
        javaCode = scriptManager.trimScript(javaCode);
        scriptManager.checkParse(javaCode);
        return javaCode;
    }

    /**
     * Check that a reference exists in symbol table on second compil pass
     *
     * @param tag       the current tag
     * @param reference the required reference
     * @param strict    flag to report an error if reference was not found
     * @param attribute (if not null reference the attribute where is defined the reference)
     * @return <code>true</code> if reference was found, <code>false</code> otherwise and add an error in compiler
     */
    public boolean checkReference(Element tag, String reference, boolean strict, String attribute) {
        String component = getSymbolTable().getClassTagIds().get(reference);
        if (component == null) {
            if (strict) {
                String msg;
                if (attribute != null) {
                    msg = "tag '" + tag.getLocalName() + "' could not find the reference '" + reference + "' on attribute [" + attribute + "]";
                } else {
                    msg = "tag '" + tag.getLocalName() + "' could not find the reference '" + reference + "'";
                }
                reportError(msg);
            }
            return false;
        }
        return true;
    }

    protected int getNextLeftBrace(String string, int pos) {
        leftBraceMatcher.reset(string);
        return leftBraceMatcher.find(pos) ? Math.max(leftBraceMatcher.start(1), leftBraceMatcher.start(2)) : -1;
    }

    protected int getNextRightBrace(String string, int pos) {
        leftBraceMatcher.reset(string);
        rightBraceMatcher.reset(string);
        int openCount = 1;
        int rightPos;
        while (openCount > 0) {
            pos++;
            int leftPos = leftBraceMatcher.find(pos) ? Math.max(leftBraceMatcher.start(1), leftBraceMatcher.start(2)) : -1;
            rightPos = rightBraceMatcher.find(pos) ? Math.max(rightBraceMatcher.start(1), rightBraceMatcher.start(2)) : -1;
            assert leftPos == -1 || leftPos >= pos;
            assert rightPos == -1 || rightPos >= pos;
            if (leftPos != -1 && leftPos < rightPos) {
                pos = leftPos;
                openCount++;
            } else if (rightPos != -1) {
                pos = rightPos;
                openCount--;
            } else {
                openCount = 0;
            }
        }
        return pos;
    }

    public String[] parseParameterList(String parameters) throws CompilerException {
        List<String> result = new ArrayList<String>();
        StringBuffer current = new StringBuffer();
        int state = 0; // normal
        for (int i = 0; i < parameters.length(); i++) {
            char c = parameters.charAt(i);
            switch (state) {
                case 0: // normal
                    switch (c) {
                        case '"':
                            current.append(c);
                            state = 1;
                            break; // in quoted string
                        case '\\':
                            current.append(c);
                            state = 2;
                            break; // immediately after backslash
                        case ',':
                            if (current.length() > 0) {
                                result.add(current.toString());
                                current.setLength(0);
                                break;
                            } else {
                                reportError("error parsing parameter list: " + parameters);
                            }
                        default:
                            current.append(c);
                    }
                    break;
                case 1: // in quoted string
                    switch (c) {
                        case '"':
                            current.append(c);
                            state = 0;
                            break; // normal
                        case '\\':
                            current.append(c);
                            state = 3;
                            break; // immediate after backslash in quoted string
                        default:
                            current.append(c);
                    }
                    break;
                case 2: // immediately after backslash
                    current.append(c);
                    state = 0; // normal
                    break;
                case 3: // immediately after backslash in quoted string
                    current.append(c);
                    state = 1; // in quoted string
                    break;
            }
        }
        if (current.length() > 0) {
            result.add(current.toString());
        }
        return result.toArray(new String[result.size()]);
    }

    public String getAutoId(ClassDescriptor objectClass) {
        if (options.getOptimize()) {
            return "$" + Integer.toString(autogenID++, 36);
        } else {
            String name = objectClass.getName();
            name = name.substring(name.lastIndexOf(".") + 1);
            return "$" + name + autogenID++;
        }
    }

    public String getUniqueId(Object object) {
        String result = uniqueIds.get(object);
        if (result == null) {
            result = "$u" + uniqueIds.size();
            uniqueIds.put(object, result);
        }
        return result;
    }

    public void setExtraInterfaces(String[] extraInterfaces) {
        this.extraInterfaces = extraInterfaces;
    }

    public String[] getExtraInterfaces() {
        return extraInterfaces;
    }

    public boolean isAbstractClass() {
        return abstractClass;
    }

    public void setAbstractClass(boolean abstractClass) {
        this.abstractClass = abstractClass;
    }

    public String getGenericType() {
        return genericType;
    }

    public void setGenericType(String genericType) {
        this.genericType = genericType;
    }

    public String getSuperGenericType() {
        return superGenericType;
    }

    public void setSuperGenericType(String superGenericType) {
        this.superGenericType = superGenericType;
    }

    public void addSimpleField(JavaField javaField) {
        getJavaFile().addSimpleField(javaField);
    }

    public JavaFile getJavaFile() {
        if (javaFile == null) {
            javaFile = new JavaFile();
        }
        return javaFile;
    }

    public void generateCode(Iterable<Generator> generatorIterator) throws IOException {
        File dest;
        if (getOptions().getTargetDirectory() != null) {
            dest = new File(getOptions().getTargetDirectory(), getOutputClassName().replace('.', File.separatorChar) + ".java");
        } else {
            dest = new File(getBaseDir(), getOutputClassName().substring(getOutputClassName().lastIndexOf(".") + 1) + ".java");
        }
        if (dest.exists() && !dest.setLastModified(System.currentTimeMillis())) {
            log.warn("could not touch file " + dest);
        }
        try {
            PrintWriter out = new PrintWriter(new FileWriter(dest));
            int dotPos = getOutputClassName().lastIndexOf(".");
            String packageName = dotPos != -1 ? getOutputClassName().substring(0, dotPos) : null;
            String simpleClassName = getOutputClassName().substring(dotPos + 1);
            CompiledObject compiledObject = getRootObject();
            for (Generator generator : generatorIterator) {
                generator.finalizeCompiler(compiledObject, this, javaFile, packageName, simpleClassName);
            }

            for (CompiledObject object : getObjects().values()) {
                object.finalizeCompiler();
            }

            for (Generator generator : generatorIterator) {
                generator.prepareJavaFile(compiledObject, this, javaFile, packageName, simpleClassName);
            }
            out.println(javaFile.toString(getLineSeparator()));
            out.close();
        } catch (RuntimeException e) {
            // file could not be generated, so delete it...
            if (!dest.delete()) {
                log.warn("could not delete file " + dest);
            }
            throw e;
        } catch (ClassNotFoundException e) {
            // file could not be generated, so delete it...
            if (!dest.delete()) {
                log.warn("could not delete file " + dest);
            }
            throw new CompilerException(e);
        }
    }

    /** line separator cached value */
    protected static String lineSeparator = System.getProperty("line.separator", "\n");

    /**
     * Returns the system line separator string.
     *
     * @return the string used to separate lines
     */
    public static String getLineSeparator() {
        return lineSeparator;
    }

    // 1.5 adds getCanonicalName; unfortunately we can't depend on 1.5 features yet
    public static String getCanonicalName(Class clazz) {
        if (clazz.isArray()) {
            String canonicalName = getCanonicalName(clazz.getComponentType());
            if (canonicalName != null) {
                return canonicalName + "[]";
            }
            return null;
        }
        return clazz.getName().replace('$', '.');
    }

    public static String getCanonicalName(ClassDescriptor clazz) {
        if (clazz.isArray()) {
            String canonicalName = getCanonicalName(clazz.getComponentType());
            if (canonicalName != null) {
                return canonicalName + "[]";
            }
            return null;
        }
        return clazz.getName().replace('$', '.');
    }

    public static String getCanonicalName(CompiledObject compiled) {
        ClassDescriptor clazz = compiled.getObjectClass();
        if (clazz.isArray()) {
            String canonicalName = getCanonicalName(clazz.getComponentType());
            if (canonicalName != null) {
                if (compiled.getGenericTypesLength() > 0) {
                    canonicalName += compiled.getGenericTypes();
                }
                return canonicalName + "[]";
            }
            return null;
        }

        String canonicalName = clazz.getName().replace('$', '.');
        if (compiled.getGenericTypesLength() > 0) {
            canonicalName += compiled.getGenericTypes();
        }
        return canonicalName;
    }

    /**
     * Escapes a string using standard Java escape sequences, generally in preparation to including it in a string literal
     * in a compiled Java file.
     *
     * @param raw the raw string to be escape
     * @return a string in which all 'dangerous' characters have been replaced by equivalent Java escape sequences
     */
    public static String escapeJavaString(String raw) {
        StringBuffer out = new StringBuffer(raw);
        for (int i = 0; i < out.length(); i++) {
            char c = out.charAt(i);
            if (c == '\\' || c == '"') {
                out.insert(i, '\\');
                i++;
            } else if (c == '\n') {
                out.replace(i, i + 1, "\\n");
                i++;
            } else if (c == '\r') {
                out.replace(i, i + 1, "\\r");
                i++;
            } else if (c < 32 || c > 127) {
                String value = Integer.toString((int) c, 16);
                while (value.length() < 4) {
                    value = "0" + value;
                }
                out.replace(i, i + 1, "\\u" + value);
                i += 5;
            }
        }
        return out.toString();
    }

    public static File URLtoFile(URL url) {
        return URLtoFile(url.toString());
    }

    public static File URLtoFile(String urlString) {
        if (!urlString.startsWith("file:")) {
            throw new IllegalArgumentException("url must start with 'file:'");
        }
        urlString = urlString.substring("file:".length());
        if (urlString.startsWith("/") && System.getProperty("os.name").startsWith("Windows")) {
            urlString = urlString.substring(1);
        }
        try {
            return new File(URLDecoder.decode(urlString.replace('/', File.separatorChar), "utf-8"));
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    public static SAXParser getSAXParser() {
        try {
            SAXParserFactory factory = SAXParserFactory.newInstance();
            factory.setNamespaceAware(true);
            SAXParser parser;
            parser = factory.newSAXParser();
            return parser;
        }
        catch (SAXException e) {
            throw new RuntimeException(e);
        }
        catch (ParserConfigurationException e) {
            throw new RuntimeException(e);
        }
    }

    public static Document parseDocument(InputStream in) throws IOException, SAXException {
        try {
            TransformerFactory factory = TransformerFactory.newInstance();
            Transformer transformer = factory.newTransformer();
            transformer.setErrorListener(new ErrorListener() {
                public void warning(TransformerException ex) throws TransformerException {
                    throw ex;
                }

                public void error(TransformerException ex) throws TransformerException {
                    throw ex;
                }

                public void fatalError(TransformerException ex) throws TransformerException {
                    throw ex;
                }
            });

            DOMResult result = new DOMResult();
            transformer.transform(new SAXSource(new XMLFilterImpl(getSAXParser().getXMLReader()) {
                Locator locator;

                @Override
                public void setDocumentLocator(Locator locator) {
                    this.locator = locator;
                }

                @Override
                public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException {
                    AttributesImpl resultAtts = new AttributesImpl(atts);
                    resultAtts.addAttribute(JAXX_INTERNAL_NAMESPACE, "line", "internal:line", "CDATA", String.valueOf(locator.getLineNumber()));
                    getContentHandler().startElement(uri, localName, qName, resultAtts);
                }
            }, new InputSource(in)), result);
            return (Document) result.getNode();
        }
        catch (TransformerConfigurationException e) {
            throw new RuntimeException(e);
        }
        catch (TransformerException e) {
            Throwable ex = e;
            while (ex.getCause() != null) {
                ex = ex.getCause();
            }
            if (ex instanceof IOException) {
                throw (IOException) ex;
            }
            if (ex instanceof SAXException) {
                throw (SAXException) ex;
            }
            if (ex instanceof RuntimeException) {
                throw (RuntimeException) ex;
            }
            throw new RuntimeException(ex);
        }
    }
}
