package jaxx.compiler;

import java.lang.reflect.Modifier;
import java.util.Iterator;
import java.util.Set;

/**
 * A decorator to place on a root compiled object to process javaHelp on the file.
 * 
 * @author tony
 * @since 1.2
 */
public class HelpRootCompiledObjectDecorator extends DefaultCompiledObjectDecorator {

    /**
     * the list of discovered helpId
     */
    protected static Set<String> helpIds = new java.util.HashSet<String>();

    protected String getBrokerFQN(JAXXCompiler compiler) {
        String helpBrokerFQN = compiler.getOptions().getHelpBrokerFQN();
        return helpBrokerFQN;
    }

    protected String getHelpId(CompiledObject o) {
        String helpID = null;
        if (o.hasClientProperties()) {
            helpID = o.getClientProperty("help");
        }
        return helpID;
    }

    @Override
    public void finalizeCompiler(JAXXCompiler compiler, CompiledObject root, CompiledObject object, JavaFile javaFile, String packageName, String className, String fullClassName) {
        super.finalizeCompiler(compiler, root, object, javaFile, packageName, className, fullClassName);
        CompilerOptions options = compiler.getOptions();

        if (options.isGenerateHelp()) {

            // add JaxxHelpUI interface
            Class<?> validatorInterface = jaxx.runtime.JaxxHelpUI.class;
            String helpBrokerFQN = getBrokerFQN(compiler);
            javaFile.addInterface(JAXXCompiler.getCanonicalName(validatorInterface) + "<" + helpBrokerFQN + ">");

            javaFile.addMethod(JavaMethod.newMethod(Modifier.PUBLIC, "void", "registerHelpId",
                    "broker.installUI(component, helpId);",
                    new JavaArgument(helpBrokerFQN, "broker"),
                    new JavaArgument("Component", "component"),
                    new JavaArgument("String", "helpId")));

            javaFile.addMethod(JavaMethod.newMethod(Modifier.PUBLIC, "void", "showHelp",
                    "getBroker().showHelp(this, helpId);",
                    new JavaArgument("String", "helpId")));

            StringBuilder buffer = new StringBuilder();

            String lineSeparator = JAXXCompiler.getLineSeparator();

            if (options.isGenerateHelp()) {

                // add code to init javax help system
                Iterator<CompiledObject> itr = compiler.getObjectCreationOrder();

                for (; itr.hasNext();) {
                    CompiledObject o = itr.next();
                    String helpID = getHelpId(o);
                    if (helpID != null) {
                        buffer.append(lineSeparator);
                        // detects a helpId to register
                        buffer.append("registerHelpId(_broker, " + o.getJavaCode() + ", " + helpID + ");");
                        //keep the helpID for helpSet generation
                        helpIds.add(helpID);
                    }
                }
            }
            if (buffer.length() > 0) {

                StringBuilder extraCode = new StringBuilder(helpBrokerFQN).append(" _broker = getBroker();");

                buffer.append(lineSeparator).append("_broker.prepareUI(this);");
                buffer.append(lineSeparator);

                // add the calls
                compiler.appendLateInitializer(extraCode.toString());
                compiler.appendLateInitializer(buffer.toString());
            }
        }
    }

    public static Set<String> getHelpIds() {
        return helpIds;
    }
}
