/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.compiler;

import jaxx.CompilerException;
import jaxx.UnsupportedAttributeException;
import jaxx.parser.JavaParser;
import jaxx.parser.JavaParserConstants;
import jaxx.parser.JavaParserTreeConstants;
import jaxx.parser.SimpleNode;
import jaxx.reflect.ClassDescriptor;
import jaxx.reflect.ClassDescriptorLoader;
import jaxx.reflect.FieldDescriptor;
import jaxx.reflect.MethodDescriptor;
import jaxx.tags.DefaultObjectHandler;
import jaxx.tags.TagManager;
import jaxx.types.TypeManager;

import java.beans.Introspector;
import java.beans.PropertyChangeListener;
import java.io.StringReader;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Represents a Java expression which fires a <code>PropertyChangeEvent</code> when it can be
 * determined that its value may have changed.  Events are fired on a "best effort" basis, and events
 * may either be fired too often (the value has not actually changed) or not often enough (the value
 * changed but no event was fired).
 */
public class DataSource {
    private class NULL {
    } // type attached to "null" constants in parsed expressions

    private String id;

    /** The Java source code for the expression. */
    private String source;

    /** The current <code>JAXXCompiler</code>. */
    private JAXXCompiler compiler;

    /** List of symbols which this data source expression depends on. */
    private List<String> dependencySymbols = new ArrayList<String>();

    private StringBuffer addListenerCode = new StringBuffer();
    private StringBuffer removeListenerCode = new StringBuffer();
    private boolean compiled;

    /** the delegate of property to be required */
    private String objectCode;

    /**
     * Creates a new data source.  After creating a <code>DataSource</code>, use {@link #compile}
     * to cause it to function at runtime.
     *
     * @param id       the DataSource's id
     * @param source   the Java source code for the data source expression
     * @param compiler the current <code>JAXXCompiler</code>
     */
    public DataSource(String id, String source, JAXXCompiler compiler) {
        this.id = id;
        this.source = source;
        this.compiler = compiler;
    }


    public String getId() {
        return id;
    }

    public String getSource() {
        return source;
    }

    public String getObjectCode() {
        return objectCode;
    }

    /**
     * Compiles the data source expression and listener.  This method calls methods in <code>JAXXCompiler</code>
     * to add the Java code that performs the data source setup.  Adding listeners to <code>DataSource</code> is
     * slightly more complicated than with ordinary classes, because <code>DataSource</code> only exists at compile
     * time.  You must pass in a Java expression which evaluates to  a <code>PropertyChangeListener</code>;  this
     * expression will be compiled and evaluated at runtime to yield the <code>DataSource's</code> listener.
     *
     * @param propertyChangeListenerCode Java code snippet which evaluates to a <code>PropertyChangeListener</code>
     * @return <code>true</code> if the expression has dependencies, <code>false</code> otherwise
     * @throws CompilerException if a compilation error occurs
     */
    public boolean compile(String propertyChangeListenerCode) throws CompilerException {
        if (compiled) {
            throw new IllegalStateException(this + " has already been compiled");
        }
        String id = compiler.getAutoId(ClassDescriptorLoader.getClassDescriptor(getClass()));
        JavaParser p = new JavaParser(new StringReader(source + ";"));
        while (!p.Line()) {
            SimpleNode node = p.popNode();
            scanNode(node, id);
        }

        if (dependencySymbols.size() > 0) {
            //TC 20081108 prefer add a real JavaField instead of raw code
            //compiler.appendBodyCode("private PropertyChangeListener " + id + " = " + propertyChangeListenerCode + ";\n");
            compiler.addSimpleField(new JavaField(Modifier.PRIVATE, PropertyChangeListener.class.getName(), id, propertyChangeListenerCode));
        }

        compileListeners();
        compiled = true;

        return dependencySymbols.size() > 0;
    }

    /** @return a list of symbols on which this data source depends. */
    public Collection<String> getDependencies() {
        return Collections.unmodifiableList(dependencySymbols);
    }

    /**
     * Examines a node to identify any dependencies it contains.
     *
     * @param node       node to scan
     * @param listenerId id of listener
     * @throws jaxx.CompilerException ?
     */
    private void scanNode(SimpleNode node, String listenerId) throws CompilerException {
        switch (node.getId()) {
            case JavaParserTreeConstants.JJTMETHODDECLARATION:
                break;
            case JavaParserTreeConstants.JJTFIELDDECLARATION:
                break;

            default:
                int count = node.jjtGetNumChildren();
                for (int i = 0; i < count; i++) {
                    scanNode(node.getChild(i), listenerId);
                }
                determineNodeType(node, listenerId);
        }
    }

    private ClassDescriptor determineLiteralType(SimpleNode node) {
        assert node.getId() == JavaParserTreeConstants.JJTLITERAL;
        if (node.jjtGetNumChildren() == 1) {
            int id = node.getChild(0).getId();
            if (id == JavaParserTreeConstants.JJTBOOLEANLITERAL) {
                return ClassDescriptorLoader.getClassDescriptor(boolean.class);
            }
            if (id == JavaParserTreeConstants.JJTNULLLITERAL) {
                return ClassDescriptorLoader.getClassDescriptor(NULL.class);
            }
            throw new RuntimeException("Expected BooleanLiteral or NullLiteral, found " + JavaParserTreeConstants.jjtNodeName[id]);
        }
        int id = node.firstToken.kind;
        switch (id) {
            case JavaParserConstants.INTEGER_LITERAL:
                if (node.firstToken.image.toLowerCase().endsWith("l")) {
                    return ClassDescriptorLoader.getClassDescriptor(long.class);
                }
                return ClassDescriptorLoader.getClassDescriptor(int.class);
            case JavaParserConstants.CHARACTER_LITERAL:
                return ClassDescriptorLoader.getClassDescriptor(char.class);
            case JavaParserConstants.FLOATING_POINT_LITERAL:
                if (node.firstToken.image.toLowerCase().endsWith("f")) {
                    return ClassDescriptorLoader.getClassDescriptor(float.class);
                }
                return ClassDescriptorLoader.getClassDescriptor(double.class);
            case JavaParserConstants.STRING_LITERAL:
                return ClassDescriptorLoader.getClassDescriptor(String.class);
            default:
                throw new RuntimeException("Expected literal token, found " + JavaParserConstants.tokenImage[id]);
        }
    }

    /**
     * Scans through a compound symbol (foo.bar.baz) to identify and track all trackable pieces of it.
     *
     * @param symbol       symbol to scan
     * @param contextClass current class context
     * @param isMethod     flag to search a method
     * @param listenerId   id of the listener
     * @return the type of the symbol (or null if it could not be determined).
     */
    private ClassDescriptor scanCompoundSymbol(String symbol, ClassDescriptor contextClass, boolean isMethod, String listenerId) {
        String[] tokens = symbol.split("\\s*\\.\\s*");
        StringBuffer currentSymbol = new StringBuffer();
        StringBuffer tokensSeenSoFar = new StringBuffer();
        boolean accepted; // if this ends up false, it means we weren't able to figure out
        // which object the method is being invoked on
        boolean recognizeClassNames = true;
        for (int j = 0; j < tokens.length - (isMethod ? 1 : 0); j++) {
            accepted = false;

            if (tokensSeenSoFar.length() > 0) {
                tokensSeenSoFar.append('.');
            }
            tokensSeenSoFar.append(tokens[j]);
            if (currentSymbol.length() > 0) {
                currentSymbol.append('.');
            }
            currentSymbol.append(tokens[j]);

            if (currentSymbol.indexOf(".") == -1) {
                String memberName = currentSymbol.toString();
                CompiledObject object = compiler.getCompiledObject(memberName);
                if (object != null) {
                    contextClass = object.getObjectClass();
                    currentSymbol.setLength(0);
                    accepted = true;
                    recognizeClassNames = false;
                } else {
                    try {
                        FieldDescriptor field = contextClass.getFieldDescriptor(memberName);
                        trackMemberIfPossible(tokensSeenSoFar.toString(), contextClass, field.getName(), false, listenerId);
                        contextClass = field.getType();
                        currentSymbol.setLength(0);
                        accepted = true;
                        recognizeClassNames = false;
                    }
                    catch (NoSuchFieldException e) {
                        if (j == 0 || j == 1 && tokens[0].equals(compiler.getRootObject().getId())) { // still in root context
                            FieldDescriptor[] newFields = compiler.getScriptFields();
                            for (FieldDescriptor newField : newFields) {
                                if (newField.getName().equals(memberName)) {
                                    addListener(tokensSeenSoFar.toString(),
                                            null,
                                            "addPropertyChangeListener(\"" + memberName + "\", " + listenerId + ");" + JAXXCompiler.getLineSeparator(),
                                            "removePropertyChangeListener(\"" + memberName + "\", " + listenerId + ");" + JAXXCompiler.getLineSeparator());
                                    contextClass = newField.getType();
                                    assert contextClass != null : "script field '" + memberName + "' is defined, but has type null";
                                    currentSymbol.setLength(0);
                                    accepted = true;
                                    recognizeClassNames = false;
                                    break;
                                }
                            }
                        }
                    }
                }
            }
            if (currentSymbol.length() > 0 && recognizeClassNames) {
                contextClass = TagManager.resolveClass(currentSymbol.toString(), compiler);
                if (contextClass != null) {
                    currentSymbol.setLength(0);
                    //accepted = true;
                    //recognizeClassNames = false;
                    // TODO: for now we don't handle statics
                    return null;
                }
            }
            if (!accepted) {
                return null;
            }
        }

        return contextClass;
    }

    /**
     * Adds type information to nodes where possible, and as a side effect adds event listeners to nodes which
     * can be tracked.
     *
     * @param expression the node to scan
     * @param listenerId id of the listener
     * @return the class descriptor of the return type or null
     */
    private ClassDescriptor determineExpressionType(SimpleNode expression, String listenerId) {
        assert expression.getId() == JavaParserTreeConstants.JJTPRIMARYEXPRESSION;
        SimpleNode prefix = expression.getChild(0);
        if (prefix.jjtGetNumChildren() == 1) {
            int type = prefix.getChild(0).getId();
            if (type == JavaParserTreeConstants.JJTLITERAL || type == JavaParserTreeConstants.JJTEXPRESSION) {
                prefix.setJavaType(prefix.getChild(0).getJavaType());
            } else if (type == JavaParserTreeConstants.JJTNAME && expression.jjtGetNumChildren() == 1) // name with no arguments after it
            {
                prefix.setJavaType(scanCompoundSymbol(prefix.getText().trim(), compiler.getRootObject().getObjectClass(), false, listenerId));
            }
        }

        if (expression.jjtGetNumChildren() == 1) {
            return prefix.getJavaType();
        }

        ClassDescriptor contextClass = prefix.getJavaType();
        if (contextClass == null) {
            contextClass = compiler.getRootObject().getObjectClass();
        }
        String lastNode = prefix.getText().trim();

        for (int i = 1; i < expression.jjtGetNumChildren(); i++) {
            SimpleNode suffix = expression.getChild(i);
            if (suffix.jjtGetNumChildren() == 1 && suffix.getChild(0).getId() == JavaParserTreeConstants.JJTARGUMENTS) {
                if (suffix.getChild(0).jjtGetNumChildren() == 0) { // at the moment only no-argument methods are trackable
                    contextClass = scanCompoundSymbol(lastNode, contextClass, true, listenerId);
                    if (contextClass == null) {
                        return null;
                    }
                    int dotPos = lastNode.lastIndexOf(".");
                    String objectCode = dotPos == -1 ? "" : lastNode.substring(0, dotPos);
                    for (int j = i - 2; j >= 0; j--) {
                        objectCode = expression.getChild(j).getText() + objectCode;
                    }
                    if (objectCode.length() == 0) {
                        objectCode = compiler.getRootObject().getJavaCode();
                    }
                    String methodName = lastNode.substring(dotPos + 1).trim();
                    try {
                        MethodDescriptor method = contextClass.getMethodDescriptor(methodName);
                        trackMemberIfPossible(objectCode, contextClass, method.getName(), true, listenerId);
                        return method.getReturnType();
                    }
                    catch (NoSuchMethodException e) {
                        // happens for methods defined in the current JAXX file via scripts
                        String propertyName = null;
                        if (methodName.startsWith("is")) {
                            propertyName = Introspector.decapitalize(methodName.substring("is".length()));
                        } else if (methodName.startsWith("get")) {
                            propertyName = Introspector.decapitalize(methodName.substring("get".length()));
                        }
                        if (propertyName != null) {
                            MethodDescriptor[] newMethods = compiler.getScriptMethods();
                            for (MethodDescriptor newMethod : newMethods) {
                                if (newMethod.getName().equals(methodName)) {
                                    addListener(compiler.getRootObject().getId(),
                                            null,
                                            "addPropertyChangeListener(\"" + propertyName + "\", " + listenerId + ");" + JAXXCompiler.getLineSeparator(),
                                            "removePropertyChangeListener(\"" + propertyName + "\", " + listenerId + ");" + JAXXCompiler.getLineSeparator());
                                    contextClass = newMethod.getReturnType();
                                    break;
                                }
                            }
                        }
                    }
                }
            }
            lastNode = suffix.getText().trim();
            if (lastNode.startsWith(".")) {
                lastNode = lastNode.substring(1);
            }
        }

        return null;
    }

    private void trackMemberIfPossible(String objectCode, ClassDescriptor objectClass, String memberName, boolean method, String listenerId) {
        if (objectClass.isInterface()) // might be technically possible to track in some cases, but for now
        {
            return;                    // we can't create a DefaultObjectHandler for interfaces
        }

        DefaultObjectHandler handler = TagManager.getTagHandler(objectClass);
        try {
            if (handler.isMemberBound(memberName)) {
                addListener(objectCode + "." + memberName + (method ? "()" : ""),
                        objectCode,
                        handler.getAddMemberListenerCode(objectCode, id, memberName, listenerId, compiler),
                        handler.getRemoveMemberListenerCode(objectCode, id, memberName, listenerId, compiler));
            }
        }
        catch (UnsupportedAttributeException e) {
            // ignore -- this is thrown for methods like toString(), for which there is no tracking and
            // no setting support
        }
    }

    /**
     * Adds type information to nodes where possible, and as a side effect adds event listeners to nodes which
     * can be tracked.
     *
     * @param node       node to scan
     * @param listenerId the listener id
     */
    private void determineNodeType(SimpleNode node, String listenerId) {
        ClassDescriptor type = null;
        if (node.jjtGetNumChildren() == 1) {
            type = node.getChild(0).getJavaType();
        }
        switch (node.getId()) {
            case JavaParserTreeConstants.JJTCLASSORINTERFACETYPE:
                type = ClassDescriptorLoader.getClassDescriptor(Class.class);
                break;
            case JavaParserTreeConstants.JJTPRIMARYEXPRESSION:
                type = determineExpressionType(node, listenerId);
                break;
            case JavaParserTreeConstants.JJTLITERAL:
                type = determineLiteralType(node);
                break;
            case JavaParserTreeConstants.JJTCASTEXPRESSION:
                type = TagManager.resolveClass(node.getChild(0).getText(), compiler);
                break;
        }
        node.setJavaType(type);
    }

    private void addListener(String dependencySymbol, String objectCode, String addCode, String removeCode) {
        this.objectCode = objectCode;
        if (!dependencySymbols.contains(dependencySymbol)) {
            dependencySymbols.add(dependencySymbol);
            if (objectCode != null) {
                addListenerCode.append("if (").append(objectCode).append(" != null) {").append(JAXXCompiler.getLineSeparator());
                addListenerCode.append("    ");
            }
            addListenerCode.append("    ").append(addCode);
            if (objectCode != null) {
                addListenerCode.append("    ").append("}");
            }

            if (objectCode != null) {
                removeListenerCode.append("if (").append(objectCode).append(" != null) {").append(JAXXCompiler.getLineSeparator());
                removeListenerCode.append("    ");
            }
            removeListenerCode.append("    ").append(removeCode);
            if (objectCode != null) {
                removeListenerCode.append("    }");
            }
        }
    }

    private void compileListeners() {
        String javaCodeId = TypeManager.getJavaCode(id);
        if (addListenerCode.length() > 0) {
            if (compiler.haveApplyDataBinding()) {
                compiler.appendApplyDataBinding(" else ");
            }
            compiler.appendApplyDataBinding("if (" + javaCodeId + ".equals($binding)) {" + JAXXCompiler.getLineSeparator());
            compiler.appendApplyDataBinding("    " + addListenerCode + JAXXCompiler.getLineSeparator());
            compiler.appendApplyDataBinding("}");
            //if (compiler.applyDataBinding.length() > 0)
            //    compiler.applyDataBinding.append("else ");
            //compiler.applyDataBinding.append("if ($binding.equals(").append(TypeManager.getJavaCode(id)).append(")) {").append(JAXXCompiler.getLineSeparator());
            //compiler.applyDataBinding.append("    ").append(addListenerCode).append(JAXXCompiler.getLineSeparator());
            //compiler.applyDataBinding.append("}").append(JAXXCompiler.getLineSeparator());
        }

        if (removeListenerCode.length() > 0) {
            if (compiler.haveRemoveDataBinding()) {
                compiler.appendRemoveDataBinding(" else ");
            }
            compiler.appendRemoveDataBinding("if (" + javaCodeId + ".equals($binding)) {" + JAXXCompiler.getLineSeparator());
            compiler.appendRemoveDataBinding("    " + removeListenerCode + JAXXCompiler.getLineSeparator());
            compiler.appendRemoveDataBinding("}");
            //if (compiler.removeDataBinding.length() > 0)
            //    compiler.removeDataBinding.append("else ");
            //compiler.removeDataBinding.append("if ($binding.equals(").append(TypeManager.getJavaCode(id)).append(")) {").append(JAXXCompiler.getLineSeparator());
            //compiler.removeDataBinding.append("    ").append(removeListenerCode).append(JAXXCompiler.getLineSeparator());
            //compiler.removeDataBinding.append("}").append(JAXXCompiler.getLineSeparator());
        }
    }
}
