/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.tags.swing;

import jaxx.CompilerException;
import jaxx.UnsupportedAttributeException;
import jaxx.compiler.CompiledObject;
import jaxx.compiler.JAXXCompiler;
import jaxx.reflect.ClassDescriptor;
import jaxx.reflect.ClassDescriptorLoader;
import jaxx.runtime.swing.JAXXButtonGroup;
import jaxx.tags.DefaultComponentHandler;
import jaxx.types.TypeManager;

import javax.swing.AbstractButton;
import javax.swing.event.ChangeListener;

public class JRadioButtonHandler extends DefaultComponentHandler {
    private static final String VALUE_PROPERTY = JAXXButtonGroup.VALUE_CLIENT_PROPERTY.substring(1);
    private static final String BUTTON_GROUP_PROPERTY = JAXXButtonGroup.BUTTON8GROUP_CLIENT_PROPERTY.substring(1);

    public JRadioButtonHandler(ClassDescriptor beanClass) {
        super(beanClass);
        ClassDescriptorLoader.checkSupportClass(getClass(), beanClass, AbstractButton.class);
    }

    @Override
    protected void configureProxyEventInfo() {
        super.configureProxyEventInfo();
        addProxyEventInfo("isSelected", ChangeListener.class, "model");
    }

    @Override
    public ClassDescriptor getPropertyType(CompiledObject object, String name, JAXXCompiler compiler) throws CompilerException {
        if (name.equals(BUTTON_GROUP_PROPERTY)) {
            return null; // accepts either a String or a ButtonGroup
        } else if (name.equals(VALUE_PROPERTY)) {
            return ClassDescriptorLoader.getClassDescriptor(Object.class);
        } else {
            return super.getPropertyType(object, name, compiler);
        }
    }

    @Override
    public boolean isMemberBound(String name) throws UnsupportedAttributeException {
        return !(name.equals(BUTTON_GROUP_PROPERTY) || name.equals(VALUE_PROPERTY)) && super.isMemberBound(name);
    }

    // handle buttonGroup assignment in addition block rather than initialization block
    @Override
    public void setProperty(CompiledObject object, String name, Object value, JAXXCompiler compiler) {
        if (name.equals(BUTTON_GROUP_PROPERTY)) {
            object.appendAdditionCode(getSetPropertyCode(object.getJavaCode(), name, TypeManager.getJavaCode(value), compiler));
        } else {
            super.setProperty(object, name, value, compiler);
        }
    }

    @Override
    public String getSetPropertyCode(String id, String name, String valueCode, JAXXCompiler compiler) throws CompilerException {
        if (name.equals(BUTTON_GROUP_PROPERTY)) {
            if (valueCode.startsWith("\"") && valueCode.endsWith("\"")) {
                valueCode = valueCode.substring(1, valueCode.length() - 1);
                CompiledObject buttonGroup = compiler.getCompiledObject(valueCode);
                if (buttonGroup == null) {
                    buttonGroup = new CompiledObject(valueCode, ClassDescriptorLoader.getClassDescriptor(JAXXButtonGroup.class), compiler);
                    compiler.registerCompiledObject(buttonGroup);
                }
            }
            return "{ javax.swing.ButtonGroup $buttonGroup = " + valueCode + "; " + id + ".putClientProperty(\"$buttonGroup\", $buttonGroup); $buttonGroup.add(" + id + "); }\n";
        } else if (name.equals(VALUE_PROPERTY)) {
            return "{ " + id + ".putClientProperty(\"" + JAXXButtonGroup.VALUE_CLIENT_PROPERTY + "\", " + valueCode + ");  Object $buttonGroup = " + id + ".getClientProperty(\"" + JAXXButtonGroup.BUTTON8GROUP_CLIENT_PROPERTY + "\");" +
                    " if ($buttonGroup instanceof jaxx.runtime.swing.JAXXButtonGroup) { ((jaxx.runtime.swing.JAXXButtonGroup) $buttonGroup).updateSelectedValue(); } }\n";
        } else {
            return super.getSetPropertyCode(id, name, valueCode, compiler);
        }
    }

}