/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.compiler;

/**
 * Represents an argument to a <code>JavaMethod</code>.
 *
 * @see JavaMethod
 */
public class JavaArgument {
    private String name;
    private String type;
    private boolean isFinal;


    /**
     * Creates a new <code>JavaArgument</code> with the specified name and type.  For example, the method <code>main()</code>
     * might have a <code>JavaArgument</code> with a name of <code>"arg"</code> and a type of <code>"java.lang.String[]"</code>.
     *
     * @param type the argument's type, as it would appear in Java source code
     * @param name the argument's name
     */
    public JavaArgument(String type, String name) {
        this(type, name, false);
    }


    /**
     * Creates a new <code>JavaArgument</code> with the specified name, type, and finality.  For example, the method <code>main()</code>
     * might have a <code>JavaArgument</code> with a name of <code>"arg"</code> and a type of <code>"java.lang.String[]"</code>.  The
     * <code>isFinal</code> parameter allows the presence of the <code>final</code> keyword on the argument to be controlled.
     *
     * @param type    the argument's type, as it would appear in Java source code
     * @param name    the argument's name
     * @param isFinal <code>true</code> if the argument should be marked final
     */
    public JavaArgument(String type, String name, boolean isFinal) {
        this.type = type;
        this.name = name;
        this.isFinal = isFinal;
    }


    /**
     * Returns the argument's name.
     *
     * @return the name of the argument
     */
    public String getName() {
        return name;
    }


    /**
     * Returns the argument's type as it would be represented in Java source code.
     *
     * @return the argument's type
     */
    public String getType() {
        return type;
    }


    /**
     * Returns <code>true</code> if the <code>final</code> keyword should appear before the argument.
     *
     * @return <code>true</code> if the argument is final
     */
    public boolean isFinal() {
        return isFinal;
    }


    /**
     * Returns the Java source code for this argument.
     *
     * @return the Java source code for this argument
     */
    @Override
    public String toString() {
        String result = type + ' ' + name;
        return isFinal ? "final " + result : result;
    }
}