/*
 * #%L
 * EUGene :: Maven plugin
 * 
 * $Id: ModelChainedFileWriter.java 1243 2013-05-05 08:32:43Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/eugene/tags/eugene-2.7.3/eugene-maven-plugin/src/main/java/org/nuiton/eugene/plugin/writer/ModelChainedFileWriter.java $
 * %%
 * Copyright (C) 2006 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.eugene.plugin.writer;

import org.apache.commons.lang3.StringUtils;
import org.nuiton.eugene.DefaultTemplateConfiguration;
import org.nuiton.eugene.ModelPropertiesUtil;
import org.nuiton.eugene.ModelReader;
import org.nuiton.eugene.Template;
import org.nuiton.eugene.TemplateConfiguration;
import org.nuiton.eugene.models.Model;
import org.nuiton.eugene.writer.ChainedFileWriterConfiguration;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

/**
 * To write model files from zargo files.
 *
 * @author tchemit
 * @since 2.0.0
 */
public class ModelChainedFileWriter extends BaseChainedFileWriter {

    public static final String PROP_GENERATED_PACKAGES = "generatedPackages";

    public static final String PROP_EXCLUDE_TEMPLATES = "excludeTemplates";

    public static final String PROP_TEMPLATES = "templates";

    public static final String PROP_TEMPLATES_LIST = "templatesList";

    public static final String PROP_DEFAULT_PACKAGE = "defaultPackage";

    public static final String PROP_MODEL_READER = "modelReader";

    public static final String PROP_MODEL_PROPERTIES_PROVIDER = "modelPropertiesProvider";

    public static final String PROP_READER = "reader";

    public static final String PROP_TEMPLATE_CONFIGURATION = "templateConfiguration";

    public ModelChainedFileWriter() {
        super(
                PROP_TEMPLATES, "templates",
                PROP_TEMPLATES_LIST, "templatesList",
                PROP_EXCLUDE_TEMPLATES, "excludeTemplates",
                PROP_READER, "reader",
                PROP_MODEL_READER, "modelReader",
                PROP_GENERATED_PACKAGES, "generatedPackages",
                PROP_DEFAULT_PACKAGE, "defaultPackage",
                PROP_TEMPLATE_CONFIGURATION, "templateConfiguration",
                PROP_MODEL_PROPERTIES_PROVIDER, "modelPropertiesProvider"
        );
    }

    @Override
    public String getInputProtocol() {
        return "model";
    }

    @Override
    public String getOutputProtocol(String modelType) {
        // nothing after java files
        return null;
    }

    @Override
    public boolean acceptModel(String modelType) {
        // accept all models
        return acceptObjectModelOrStateModel(modelType);
    }

    @Override
    public boolean acceptInclude(String include) {
        return include.startsWith("model:") ||
               include.endsWith(".objectmodel") ||
               include.endsWith(".statemodel");
    }

    @Override
    public String getDefaultIncludes() {
        return "**/*.*model";
    }

    @Override
    public String getDefaultInputDirectory() {
        return "src/main/models";
    }

    @Override
    public String getDefaultOutputDirectory() {
        return "java";
    }

    @Override
    public String getDefaultTestInputDirectory() {
        return "src/test/models";
    }

    @Override
    public String getDefaultTestOutputDirectory() {
        return "test-java";
    }

    public String getDefaultPackage() {
        return getProperty(PROP_DEFAULT_PACKAGE, String.class);
    }

    public String[] getExcludeTemplates() {
        return getProperty(PROP_EXCLUDE_TEMPLATES, String[].class);
    }

    public String getGeneratedPackages() {
        return getProperty(PROP_GENERATED_PACKAGES, String.class);
    }

    @SuppressWarnings({"unchecked"})
    public List<Template<Model>> getTemplatesList() {
        return getProperty(PROP_TEMPLATES_LIST, List.class);
    }

    public String getTemplates() {
        return getProperty(PROP_TEMPLATES, String.class);
    }

    public TemplateConfiguration getTemplateConfiguration() {
        return getProperty(PROP_TEMPLATE_CONFIGURATION, TemplateConfiguration.class);
    }

    protected ModelReader<?> getModelReader() {
        return getProperty(PROP_MODEL_READER, ModelReader.class);
    }

    protected String getReader() {
        return getProperty(PROP_READER, String.class);
    }

    protected ModelPropertiesUtil.ModelPropertiesProvider getModelPropertiesProvider() {
        return getProperty(PROP_MODEL_PROPERTIES_PROVIDER, ModelPropertiesUtil.ModelPropertiesProvider .class);
    }

    @Override
    protected void initWriter(ChainedFileWriterConfiguration configuration) {
        super.initWriter(configuration);

        // obtain a reader
        ClassLoader classLoader = configuration.getClassLoader();
        ClassLoader loader = classLoader;
        if (getModelReader() == null) {

            if (getReader() != null) {
                // use a specific reader
                String reader = getReader();
                try {
                    ClassLoader fixedClassLoader = loader;
                    ModelReader<?> modelReader = (ModelReader<?>)
                            Class.forName(reader, true,
                                          fixedClassLoader).newInstance();
                    //TODO : should check that the reader is compatible with
                    //TODO : given modelType
                    properties.put(PROP_MODEL_READER, modelReader);
                } catch (Exception eee) {
                    throw new IllegalStateException("could not obtain reader "
                                                    + reader, eee);
                }
            } else {
                String modelType = configuration.getModelType();
                ModelReader<?> modelReader = configuration.getModelHelper().getModelReader(modelType, "xml");
                        configuration.getModelHelper().getModelReaders().get(modelType);
                if (modelReader == null) {
                    throw new IllegalStateException(
                            "could not find a model reader for modelType : " +
                            modelType + ", availables readers : " +
                            configuration.getModelHelper().getModelReaders().values());
                }
                properties.put(PROP_MODEL_READER, modelReader);
            }
        }

        boolean verbose = configuration.isVerbose();

        // gets the provider of safe tag values and stereotypes
        ModelPropertiesUtil.ModelPropertiesProvider propertiesProvider =
                getModelPropertiesProvider();
        
        // affect it to the model reader
        getModelReader().setModelPropertiesProvider(propertiesProvider);

        // set the verbose level of the model reader
        getModelReader().setVerbose(verbose);

        List<Template<?>> templatesList = initTemplates(configuration);

        properties.put(PROP_TEMPLATES_LIST, templatesList);
    }

    protected List<Template<?>> initTemplates(ChainedFileWriterConfiguration configuration) {

        ClassLoader loader = configuration.getClassLoader();

        boolean verbose = configuration.isVerbose();

        Properties templateProperties = new Properties();

        templateProperties.put(Template.PROP_DEFAULT_PACKAGE, getDefaultPackage());
        templateProperties.put(Template.PROP_ENCODING, configuration.getEncoding());
        templateProperties.put(Template.PROP_VERBOSE, verbose);
        templateProperties.put(Template.PROP_OVERWRITE, configuration.isOverwrite());
        templateProperties.put(Template.PROP_CLASS_LOADER, loader);
        templateProperties.put(Template.PROP_EXCLUDE_TEMPLATES, configuration.getProperties().get(PROP_EXCLUDE_TEMPLATES));

        String generatedPackages = getGeneratedPackages();
        if (StringUtils.isEmpty(generatedPackages)) {
            if (verbose) {
                getLog().info("generating all packages");
            }
        } else {
            templateProperties.put(Template.PROP_GENERATED_PACKAGES,
                                   generatedPackages);
            if (verbose) {
                getLog().info("generating only for packages " + generatedPackages);
            }
        }

        // init templates

        List<Template<?>> templatesList = new ArrayList<Template<?>>();
        if (StringUtils.isEmpty(getTemplates())) {
            throw new IllegalArgumentException("No template specified, use the templates parameter");
        }
        String[] templatesNames = getTemplates().split(",");
        for (String templateName : templatesNames) {
            // remove trailing spaces
            templateName = templateName.trim();
            Template<?> template =
                    configuration.getModelTemplates().get(templateName);

            if (template == null) {
                getLog().warn("template [" + templateName + "] is not " +
                              "registred via plexus, try to load it directly");
                try {
                    template = (Template<?>) Class.forName(
                            templateName, true, loader).newInstance();
                } catch (Exception e) {
                    throw new IllegalStateException(
                            "Can't obtain template [" + templateName +
                            "] for reason " + e.getMessage(), e);
                }
            }

            if (verbose) {
                getLog().info("will use the template [" + templateName + "]");
            }

            // will use this template
            templatesList.add(template);

            // set the properties of the template
            template.setConfiguration(
                    new DefaultTemplateConfiguration(templateProperties));
        }
        return templatesList;
    }

    @Override
    public void generate(ChainedFileWriterConfiguration configuration,
                         File outputDir,
                         Map<File, List<File>> filesByRoot,
                         Map<File, List<File>> resourcesByFile) throws IOException {

        Set<File> modelFiles = new HashSet<File>();
        for (List<File> files : filesByRoot.values()) {
            modelFiles.addAll(files);
        }
        File[] filesToRead = modelFiles.toArray(new File[modelFiles.size()]);

        if (configuration.isVerbose()) {
            getLog().info("Will read " + filesToRead.length + " model(s).");
        }
        

        
        // read memory model from all files models
        Model model = getModelReader().read(filesToRead);

        // get the last modified source timestamp from reader
        long lastModifiedSource = getModelReader().getLastModifiedSource();

        // apply all templates to the model

        for (Template<Model> template : getTemplatesList()) {
            getLog().info("Apply generator " + template.getClass().getSimpleName());

            // set the lastModified source property
            template.setProperty(Template.PROP_LAST_MODIFIED_SOURCE, lastModifiedSource);

            template.setProperty(Template.PROP_WRITER_REPORT, getWriterReport());

            if (configuration.isVerbose()) {
                getLog().info(" overwrite          = " + template.isOverwrite());
                getLog().info(" encoding           = " + template.getEncoding());
                getLog().info(" lastModifiedSource = " + template.getLastModifiedSource());
                getLog().info(" exclude            = " + template.getExcludeTemplates());
                getLog().info(" output directory   = " + outputDir);
            }

            // apply template
            template.applyTemplate(model, outputDir);
        }
    }

}
