/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: SendingViewManager.java 21 2011-05-09 16:43:58Z sletellier $
 * $HeadURL: http://svn.chorem.org/svn/vradi/tags/vradi-0.6/vradi-swing/src/main/java/org/chorem/vradi/ui/email/helpers/SendingViewManager.java $
 * %%
 * Copyright (C) 2009 - 2010 Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.vradi.ui.email.helpers;

import jaxx.runtime.swing.ErrorDialogUI;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.vradi.VradiContext;
import org.chorem.vradi.VradiEntitiesListener;
import org.chorem.vradi.beans.FormPagedResult;
import org.chorem.vradi.beans.SendingHelper;
import org.chorem.vradi.entities.Sending;
import org.chorem.vradi.entities.Session;
import org.chorem.vradi.entities.Status;
import org.chorem.vradi.entities.User;
import org.chorem.vradi.entities.XmlStream;
import org.chorem.vradi.services.VradiService;
import org.chorem.vradi.ui.email.EmailHandler;
import org.chorem.vradi.ui.email.SendingViewUI;
import org.chorem.vradi.ui.email.renderers.EmailFormsHighlighter;
import org.chorem.vradi.ui.helpers.UIHelper;
import org.chorem.vradi.ui.offer.OfferTable;
import org.chorem.vradi.ui.offer.models.OfferListTableModel;
import org.chorem.vradi.ui.task.VradiTask;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.decorator.HighlighterFactory;
import org.nuiton.wikitty.WikittyProxy;

import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;
import java.awt.Cursor;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

public class SendingViewManager implements VradiEntitiesListener<Sending> {

    /** Logger */
    static private final Log log = LogFactory.getLog(SendingViewManager.class);

    protected Session session;

    protected Sending sending;

    protected SendingViewUI ui;

    protected EmailFormsHighlighter emailFormsHighlighter;

    protected OfferListTableModel listTableModel;

    private static SendingViewManager instance;

    protected SendingViewManager() {

        // register each tree on wikitty service
        VradiContext.get().getVradiNotifier().addVradiListener(Sending.class, this);
    }

    // Singleton
    public static SendingViewManager getInstance() {
        if (instance == null) {
            instance = new SendingViewManager();
        }
        return instance;
    }

    public void openUI(SendingViewUI ui, Session session, Sending sending) {

        if (sending == null) {
            return;
        }

        this.ui = ui;
        WikittyProxy wikittyProxy = VradiService.getWikittyProxy();
        this.sending = wikittyProxy.restore(Sending.class, sending.getWikittyId());
        this.session = session;

        // Init session ui
        ui.setBean(sending);

        ui.setSession(session);

        // listTable cellRenderer
        // TODO SLE 30/09/2010 : Refactor it !
        OfferListTableModel.OfferListTableCellRenderer renderer =
                new OfferListTableModel.OfferListTableCellRenderer();

        OfferTable listTable = ui.getListTable();

        listTableModel = (OfferListTableModel) listTable.getModel();

        emailFormsHighlighter = new EmailFormsHighlighter(listTableModel);
        updateDatas();

        listTable.setModel(listTableModel);

        listTable.setDefaultRenderer(Boolean.class, renderer);
        listTable.setDefaultRenderer(Date.class, renderer);
        listTable.setDefaultRenderer(Double.class, renderer);
        listTable.setDefaultRenderer(String.class, renderer);
        listTable.setDefaultRenderer(XmlStream.class, renderer);
        listTable.setDefaultRenderer(Status.class, renderer);

        // listTable highlighter
        Highlighter highlighter =
                HighlighterFactory.createAlternateStriping(
                        HighlighterFactory.NOTEPAD, HighlighterFactory.GENERIC_GRAY);

        listTable.addHighlighter(highlighter);
        listTable.addHighlighter(emailFormsHighlighter);

        // Add mouse listener
        MouseListener offerListTableMouseListener = getOfferListTableMouseListener();
        listTable.addMouseListener(offerListTableMouseListener);

        // init ui
        ui.init();
    }

    protected void updateDatas() {
        if (sending == null) {
            return;
        }
        WikittyProxy wikittyProxy = VradiService.getWikittyProxy();

        FormPagedResult formPage = new FormPagedResult();

        Set<String> formsIdsToDisplay = new HashSet<String>();

        // Get forms
        Set<String> forms = SendingHelper.extractForms(wikittyProxy, sending.getGroupForms());
        if (forms != null) {
            formsIdsToDisplay.addAll(forms);
        }

        // Get deleted
        Set<String> deletedForms = SendingHelper.extractForms(wikittyProxy, sending.getDeletedGroupForms());
        Set<String> addedForms = SendingHelper.extractForms(wikittyProxy, sending.getAddedGroupForms());
        if (deletedForms != null) {
            formsIdsToDisplay.addAll(deletedForms);
        }
        if (!formsIdsToDisplay.isEmpty()) {
            formPage.setFormsToShow(new ArrayList<String>(formsIdsToDisplay));
        }

        emailFormsHighlighter.setFormsAdded(addedForms);
        emailFormsHighlighter.setFormsDeleted(deletedForms);
        listTableModel.setFormPagedResult(formPage);

        emailFormsHighlighter.notifyChange();
    }

    protected void refresh() {

        SwingUtilities.invokeLater(new Thread() {

            @Override
            public void run() {
                WikittyProxy wikittyProxy = VradiService.getWikittyProxy();
                sending = wikittyProxy.restore(Sending.class, sending.getWikittyId());

                ui.setBean(sending);

                updateDatas();
            }
        });
    }

    @Override
    public void entitiesAdded(Set<Sending> added) {
        refresh();
    }

    @Override
    public void entitiesRemoved(Set<String> removed) {
    }

    /**
     * Implementation du click (droit/gauche) sur la table des offres résultats/
     * <p/>
     * Comportement :
     * <ul>
     * <li>double clic : ouvre l'offre
     * <li>clic droit : changer le status, ajouter a une session de mail
     * </ul>
     *
     * @return le mouse listener
     */
    public MouseListener getOfferListTableMouseListener() {
        // double click on listTable shows selected offer
        return new MouseAdapter() {

            @Override
            public void mouseClicked(MouseEvent e) {
                JXTable listTable = (JXTable) e.getSource();

                // double clic
                if (e.getButton() == MouseEvent.BUTTON1 &&
                    e.getClickCount() == 2) {
                    try {
                        ui.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

                        EmailHandler handler = UIHelper.getHandler(ui, EmailHandler.class);
                        handler.selectOffer(ui, listTable);
                    } catch (Exception ex) {
                        if (log.isErrorEnabled()) {
                            log.error("Can't open offer", ex);
                        }
                        ErrorDialogUI.showError(ex);
                    } finally {
                        ui.setCursor(null);
                    }

                    // clic droit
                } else if (e.getButton() == MouseEvent.BUTTON3) {
                    int[] selectedRows = listTable.getSelectedRows();
                    int rowClicked = listTable.rowAtPoint(e.getPoint());
                    boolean rowClickedSelected = false;
                    if (selectedRows == null || selectedRows.length == 0) {
                        listTable.getSelectionModel().setSelectionInterval(rowClicked, rowClicked);
                        rowClickedSelected = true;
                    } else {
                        int i = 0;
                        while (i < selectedRows.length && !rowClickedSelected) {
                            rowClickedSelected = rowClicked == selectedRows[i++];
                        }
                    }

                    if (rowClickedSelected) {
                        showOfferListContextMenu(ui, listTable, ui.getOfferListTableModel(), e);
                    }
                }
            }
        };
    }

    /**
     * Le menu contextuel lors d'un clic droit sur la table.
     *
     * @param ui    sending view ui
     * @param table of offers
     * @param model of table
     * @param e     event du clic
     */
    protected void showOfferListContextMenu(final SendingViewUI ui, JXTable table, OfferListTableModel model, MouseEvent e) {

        // Enable menu only if session is active
        boolean enabled = ui.getSession() != null && ui.isNotClosed();

        JPopupMenu menu = new JPopupMenu();

        // Menu to add user or group sending
        JMenuItem removeFormFromSession = new JMenuItem(_("vradi.email.removeFormFromSession"));

        removeFormFromSession.setEnabled(enabled);

        if (enabled) {
            removeFormFromSession.addActionListener(new RemoveAllFormFromSession(ui, table, model));
        }

        menu.add(removeFormFromSession);

        // Menu to add user or group sending
        JMenuItem removeFormFromSessionForUser = new JMenuItem(_("vradi.email.removeFormFromSessionForUser"));

        removeFormFromSessionForUser.setEnabled(enabled);

        if (enabled) {
            removeFormFromSessionForUser.addActionListener(new RemoveFormFromSessionForUser(ui, table, model));
        }

        menu.add(removeFormFromSessionForUser);

        int[] selectedRows = table.getSelectedRows();
        final List<String> formsIdsToUpdate = new ArrayList<String>();
        for (int row : selectedRows) {
            row = table.convertRowIndexToModel(row);

            FormPagedResult formPage = listTableModel.getFormPagedResult();
            List<String> formsToShow = formPage.getFormsIdsToShow();

            String formId = formsToShow.get(row);
            if (formId != null) {

                Set<String> deletedFormsIds = SendingHelper.extractForms(VradiService.getWikittyProxy(), sending.getDeletedGroupForms());
                if (deletedFormsIds != null) {
                    if (deletedFormsIds.contains(formId)) {
                        formsIdsToUpdate.add(formId);
                    }

                }
            }
        }

        if (!formsIdsToUpdate.isEmpty()) {

            // Display popup menu
            JMenuItem revertDeletion = new JMenuItem(_("vradi.email.forms.revertDeletion"));
            revertDeletion.addActionListener(new ActionListener() {

                @Override
                public void actionPerformed(ActionEvent e) {
                    EmailHandler handler = UIHelper.getHandler(ui, EmailHandler.class);
                    handler.revertDeletion(ui, sending, formsIdsToUpdate);
                }
            });

            revertDeletion.setEnabled(session != null);

            menu.add(revertDeletion);
        }

        menu.show(e.getComponent(), e.getX(), e.getY());
    }

    protected abstract class RemoveFormFromSession implements ActionListener {

        protected List<String> formsIdsToUpdate = new ArrayList<String>();

        protected SendingViewUI ui;

        protected User user;

        protected Session session;

        protected EmailHandler handler;

        protected RemoveFormFromSession(SendingViewUI ui, JXTable listTable, OfferListTableModel tableModel) {
            this.ui = ui;
            handler = UIHelper.getHandler(ui, EmailHandler.class);

            Sending sending = ui.getBean();
            user = VradiService.getWikittyProxy().restore(User.class, sending.getUser());

            session = VradiService.getVradiDataService().getSessionBySending(sending.getWikittyId());

            int[] selectedRows = listTable.getSelectedRows();

            for (int row : selectedRows) {
                row = listTable.convertRowIndexToModel(row);

                FormPagedResult formPage = tableModel.getFormPagedResult();
                List<String> formsIdsToShow = formPage.getFormsIdsToShow();

                String formId = formsIdsToShow.get(row);
                if (formId != null) {
                    formsIdsToUpdate.add(formId);
                }
            }
        }
    }

    protected class RemoveAllFormFromSession extends RemoveFormFromSession {

        public RemoveAllFormFromSession(SendingViewUI ui, JXTable listTable, OfferListTableModel tableModel) {
            super(ui, listTable, tableModel);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            if (handler.askToRemoveAllForm(ui, EmailDataHelper.convertFormsIdsToString(formsIdsToUpdate))) {
                VradiTask<Void> removeFormsFromSessionTask = new VradiTask<Void>(ui) {
                    @Override
                    public Void doAction() throws Exception {
                        VradiService.getVradiStorageService().removeFormsFromSession(session.getWikittyId(), formsIdsToUpdate);
                        return null;
                    }
                };
                removeFormsFromSessionTask.execute();
            }
        }
    }

    protected class RemoveFormFromSessionForUser extends RemoveFormFromSession {

        public RemoveFormFromSessionForUser(SendingViewUI ui, JXTable listTable, OfferListTableModel tableModel) {
            super(ui, listTable, tableModel);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            if (handler.askToRemoveUserForm(ui, EmailDataHelper.convertFormsIdsToString(formsIdsToUpdate), user.getName())) {
                handler.removeUserSending(ui, session, formsIdsToUpdate, new Object[]{user}, false);
            }
        }
    }
}
