/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: VradiMain.java 21 2011-05-09 16:43:58Z sletellier $
 * $HeadURL: http://svn.chorem.org/svn/vradi/tags/vradi-0.6/vradi-swing/src/main/java/org/chorem/vradi/VradiMain.java $
 * %%
 * Copyright (C) 2009 - 2010 Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.vradi;

import jaxx.runtime.SwingUtil;
import jaxx.runtime.decorator.Decorator;
import jaxx.runtime.decorator.DecoratorProvider;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.renderer.DecoratorProviderListCellRenderer;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.beanutils.converters.DateConverter;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.vradi.entities.Client;
import org.chorem.vradi.entities.Form;
import org.chorem.vradi.entities.FormLink;
import org.chorem.vradi.entities.Group;
import org.chorem.vradi.entities.Status;
import org.chorem.vradi.entities.Thesaurus;
import org.chorem.vradi.entities.User;
import org.chorem.vradi.entities.VradiUser;
import org.chorem.vradi.entities.XmlStream;
import org.chorem.vradi.services.VradiException;
import org.chorem.vradi.services.VradiService;
import org.chorem.vradi.ui.ChangeLogHandler;
import org.chorem.vradi.ui.ChangeLogUI;
import org.chorem.vradi.ui.VradiMainUI;
import org.chorem.vradi.ui.VradiMainUIHandler;
import org.chorem.vradi.ui.helpers.UIHelper;
import org.chorem.vradi.ui.login.LoginHandler;
import org.chorem.vradi.ui.login.LoginUI;
import org.chorem.vradi.ui.renderers.NumberListCellRenderer;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.bundle.I18nBundle;
import org.nuiton.i18n.bundle.I18nBundleEntry;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.StringUtil;
import org.nuiton.util.converter.ConverterUtil;
import org.nuiton.wikitty.entities.WikittyExtension;

import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import static org.nuiton.i18n.I18n._;

public class VradiMain {

    /** Logger */
    private static Log log = LogFactory.getLog(VradiMain.class);

    public static void main(String[] args) {
        final long startingTime = System.nanoTime();
        log.info("Vradi start at " + new Date() + " args: " + Arrays.toString(args));

        try {
            // init root context
            final VradiContext context = init(args);
            log.info(_("vradi.init.context.done", StringUtil.convertTime(startingTime, System.nanoTime())));

            SwingUtil.checkJAXXContextEntry(context, SwingUtil.newContextEntryDef(ApplicationConfig.class));
            SwingUtil.checkJAXXContextEntry(context, SwingUtil.newContextEntryDef(VradiAction.class));

            final ApplicationConfig config = context.getContextValue(ApplicationConfig.class);
            config.doAction(0);

            final LoginHandler handler = UIHelper.getHandler(context, LoginHandler.class);
            final LoginUI loginUI = new LoginUI(context);

            loginUI.addPropertyChangeListener("bean", new PropertyChangeListener() {
                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    if (evt.getNewValue() != null) {
                        // on affiche l'ui principale
                        context.setContextValue(evt.getNewValue());

                        // dispose loginUI
                        loginUI.removePropertyChangeListener(this);
                        loginUI.dispose();

                        try {

                            ChangeLogHandler changeLogHandler = UIHelper.getHandler(context, ChangeLogHandler.class);
                            String changeLog = changeLogHandler.getChangeLog();
                            if (!changeLog.isEmpty()) {
                                ChangeLogUI clUI = new ChangeLogUI(context);
                                clUI.getContent().setText(changeLog);
                                clUI.setLocationRelativeTo(null);
                                clUI.setVisible(true);
                            }
                        } catch (Exception eee) {
                            log.error("Can't open change log ui : ", eee);
                            ErrorDialogUI.showError(eee);
                        }

                        // show mainUI
                        VradiMainUIHandler uiHandler = UIHelper.getHandler(context, VradiMainUIHandler.class);
                        VradiMainUI mainUI = uiHandler.initUI(context, VradiConfigHelper.isFullScreen(config));
                        mainUI.setCursor(null);
                        mainUI.setLocationRelativeTo(null);
                        mainUI.setVisible(true);

                        log.info(_("vradi.init.ui.done", StringUtil.convertTime(startingTime, System.nanoTime())));
                    }
                }
            });

            SwingUtilities.invokeLater(new Runnable() {
                @Override
                public void run() {

                    String login = VradiConfigHelper.getLogin(config);
                    // password here is md5 encoded
                    String password = VradiConfigHelper.getPassword(config);
                    VradiUser vradiUser = null;

                    if (StringUtils.isNotBlank(login) && StringUtils.isNotBlank(password)) {
                        try {
                            vradiUser = handler.getVradiStorageService().loginUser(login, password);
                        } catch (VradiException e) {
                            log.error(e.getMessage());
                        }
                    }

                    if (vradiUser != null) {
                        loginUI.setBean(vradiUser);
                    } else {
                        loginUI.setLocationRelativeTo(null);
                        loginUI.setVisible(true);
                    }
                }
            });

        } catch (Exception e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            System.exit(1);
        }
    }

    public static VradiContext init(String... args) throws Exception {
        // to enable javassist on webstart, must remove any securityManager,
        // see if this can be dangerous (should not be since jnlp is signed ?)
        // moreover it speeds up the loading :)
        System.setSecurityManager(null);

        long t0 = System.nanoTime();

        DefaultI18nInitializer initializer = new DefaultI18nInitializer("vradi-swing-i18n");

        I18n.init(initializer, null);

        Runtime.getRuntime().addShutdownHook(new ShutdownHook());

        // init root context
        VradiContext context = VradiContext.init();

        // Init available locale
        initLocalesAvailables(initializer);

        // register decorator one for all
        DecoratorProvider decoratorProvider = new DecoratorProvider() {
            @Override
            protected void loadDecorators() {

                // FIXME EC-20100420 all ${name} are hard coded and non refectorable
                registerMultiJXPathDecorator(Client.class, "${name}$s", "", "");
                registerMultiJXPathDecorator(User.class, VradiConstants.USER_ONLY_DECORATOR, "${name}$s", "", "");
                registerMultiJXPathDecorator(Group.class, "${name}$s", "", "");
                registerMultiJXPathDecorator(Status.class, "${name}$s", "", "");
                registerMultiJXPathDecorator(WikittyExtension.class, "${name}$s", "", "");
                registerMultiJXPathDecorator(XmlStream.class, "${name}$s", "", "");
                registerMultiJXPathDecorator(Thesaurus.class, "${name}$s", "", "");
                // ${tagsAsString}##
                registerMultiJXPathDecorator(File.class, "${name}$s", "", "");
                //format:date($today, 'MM/dd/yyyy')
                registerMultiJXPathDecorator(Form.class, "${objet}$s", "", "");
                registerMultiJXPathDecorator(FormLink.class, "${name}$s", "", "");
                registerDecorator(new Decorator<User>(User.class) {

                    private static final long serialVersionUID = 1L;

                    @Override
                    public String toString(Object bean) {
                        User user = (User) bean;
                        Client client = VradiService.getWikittyProxy().restore(Client.class, user.getClient());
                        String clientString = "";
                        if (client != null) {
                            clientString = client.getName() + " - ";
                        }
                        return clientString + user.getName();
                    }
                });
            }
        };
        context.setContextValue(decoratorProvider);
        context.setContextValue(new DecoratorProviderListCellRenderer(decoratorProvider));

        context.setContextValue(new NumberListCellRenderer());

        // init config
        ApplicationConfig config = VradiConfig.getConfig(args);
        context.setContextValue(config);

        long t00 = System.nanoTime();

        // init i18n
        Locale locale = VradiConfigHelper.getLocale(config);
        I18n.setDefaultLocale(locale);
        log.info("language : " + locale);

        if (log.isDebugEnabled()) {
            log.debug("i18n loading time : " + StringUtil.convertTime(t00, System.nanoTime()));
        }

        log.info(_("vradi.message.config.loaded", VradiConfigHelper.getVersionAsString(config)));

        // prepare ui look&feel and load ui properties
        try {
            SwingUtil.initNimbusLoookAndFeel();

            // set dismiss delay for one hour
            ToolTipManager.sharedInstance().setDismissDelay(1000 * 60 * 60);

            // Show big button on splitPane
            UIDefaults defaults = UIManager.getLookAndFeelDefaults();
            defaults.put("SplitPane.oneTouchButtonSize", 10);

            // Display verticals line on JTree
            defaults.put("Tree.drawHorizontalLines", Boolean.TRUE);
            defaults.put("Tree.drawVerticalLines", Boolean.TRUE);
            defaults.put("Tree.showRootHandles", Boolean.TRUE);
            defaults.put("Tree.leftChildIndent", 12);
            defaults.put("Tree.rightChildIndent", 12);
            defaults.put("Tree.scrollsOnExpand", Boolean.TRUE);

        } catch (Exception e) {
            // could not find nimbus look-and-feel
            log.warn(_("vradi.warning.nimbus.landf"));
        } catch (Throwable e) {
            log.warn(_("vradi.warning.no.ui"));
        }

        // init date converters
        Converter converter = ConverterUtil.getConverter(Date.class);
        if (converter != null) {
            ConvertUtils.deregister(Date.class);

            DateConverter dateConverter = new DateConverter();
            dateConverter.setUseLocaleFormat(true);
            ConvertUtils.register(dateConverter, Date.class);
        }

        // chargement de la configuration des uis
        SwingUtil.loadUIConfig("/" + config.getConfigFileName(), null);

        // init service proxies
        VradiService.initServiceProxies(config);

        // Init notifier
        context.setContextValue(new VradiNotifier());

        if (log.isDebugEnabled()) {
            log.debug("init done in " + StringUtil.convertTime(t0, System.nanoTime()));
        }

        return context;
    }

    public static void initLocalesAvailables(DefaultI18nInitializer initializer) {
        List<Locale> localesAvailables = new ArrayList<Locale>();
        try {
            I18nBundle[] i18nBundles = initializer.resolvBundles();
            for (I18nBundle bundle : i18nBundles) {
                for (Object aBundle : bundle) {
                    I18nBundleEntry entry = (I18nBundleEntry) aBundle;
                    Locale locale = entry.getLocale();
                    localesAvailables.add(locale);

                    log.info("Language found : " + locale.getLanguage());
                }
            }
        } catch (Exception e) {
            log.error("failled to resolve i18n bundles");
        }
        VradiContext.get().setLocalesAvailables(localesAvailables);
    }

    public static class ShutdownHook extends Thread {

        public ShutdownHook() {
            super("shutdown vradi");
        }

        @Override
        public void run() {
            try {
                super.run();
                VradiContext.get().close();
                // force to kill main thread

                log.info(_("vradi.init.closed", new Date()));
                Runtime.getRuntime().halt(0);
            } catch (Exception ex) {
                log.error("error while closing " + ex.getMessage(), ex);
                ErrorDialogUI.showError(ex);
                Runtime.getRuntime().halt(1);
            }
        }
    }
}
