/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.votecounting.business;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Contexte du dépouillement du sondage (PATTERN STRATEGY).
 *
 * @author fdesbois
 * @version $Id: Context.java 2731 2009-08-18 12:22:46Z nrannou $
 */
public class Context {

    /** log. */
    private static final Log log = LogFactory.getLog(Context.class);

    /**
     * Booléen permettant de savoir si la méthode doit calculer la valeur des
     * choix en fonction des groupes
     */
    private boolean groupCounting;
    /**
     * Liste des choix liés au contexte et résultats
     */
    private List<Choice> choices, results;
    /**
     * Méthode de dépouillement choisie pour ce contexte
     */
    private Method method;

    /**
     * Constructeur
     *
     * @param method : Méthode de dépouillement
     * @param groupCounting : Dépouillement pour les groupes ou non
     */
    public Context(Method method, boolean groupCounting) {
        this.groupCounting = groupCounting;
        this.choices = new ArrayList<Choice>();
        this.results = new ArrayList<Choice>();
        this.method = method;
    }

    /**
     * Renvoie un choix de la map
     *
     * @param idChoice : identifiant du choix
     * @return choix
     */
    public Choice getChoice(String idChoice) {
        for (Choice choice : this.choices) {
            if (choice.getIdChoice().equals(idChoice))
                return choice;
        }
        return null;
    }

    /**
     * Renvoie la liste des choix
     *
     * @return liste des choix
     */
    public List<Choice> getChoices() {
        return this.choices;
    }

    /**
     * Renvoie la liste des résultats
     *
     * @return liste des choix résultats
     */
    public List<Choice> getResults() {
        return this.results;
    }

    /**
     * Ajout d'un nouveau choix au contexte
     *
     * @param idChoice : identifiant du choix
     */
    public void addChoice(String idChoice) {
        this.choices.add(new Choice(idChoice));
    }

    /**
     * Ajout d'un groupe pour l'ensemble des choix du contexte
     *
     * @param idGroup : identifiant du groupe
     * @param weight : poids du groupe
     */
    public void addGroup(String idGroup, double weight) {
        for (Choice choice : this.choices) {
            choice.addGroup(idGroup, weight);
        }
    }

    /**
     * Execution de la méthode de dépouillement et calcul des choix résultats
     *
     * @return liste de choix résultats
     */
    public boolean execute() {
        try {
            method.executeMethod(choices, groupCounting);
        } catch (Exception e) {
            log.error("L'exécution du dépouillement a échoué", e);
            return false;
        }
        this.results.clear();
        Choice result = new Choice("");
        for (Choice curr : this.choices) {
            if (curr.value() > result.value()) {
                result = curr;
                this.results.clear();
            } else if (curr.value() == result.value()) {
                this.results.add(curr);
            }
        }
        this.results.add(result);
        return true;
    }

}