/*
 * #%L
 * Lima :: Swing
 * 
 * $Id: VatChartTreeTableModel.java 3777 2014-04-14 07:29:08Z sbavencoff $
 * $HeadURL: https://svn.chorem.org/lima/tags/lima-0.7.3/lima-swing/src/main/java/org/chorem/lima/ui/vatchart/VatChartTreeTableModel.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package org.chorem.lima.ui.vatchart;

import static org.nuiton.i18n.I18n.t;

import java.util.List;

import javax.swing.tree.TreePath;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.LimaException;
import org.chorem.lima.business.api.VatStatementService;
import org.chorem.lima.entity.VatStatement;
import org.chorem.lima.entity.VatStatementImpl;
import org.chorem.lima.service.LimaServiceFactory;
import org.jdesktop.swingx.treetable.AbstractTreeTableModel;

public class VatChartTreeTableModel extends AbstractTreeTableModel {

    /** log. */
    private static final Log log = LogFactory.getLog(VatChartViewHandler.class);

    /** Services. */
    protected final VatStatementService vatStatementService;

    /** Model constructor. Initiate account service used here. */
    public VatChartTreeTableModel() {
        //create root for the tree
        super(new VatStatementImpl());
        // Gets factory service 
        vatStatementService =
                LimaServiceFactory.getService(
                        VatStatementService.class);
    }


    @Override
    public int getColumnCount() {
        return 2;
    }

    @Override
    public String getColumnName(int column) {
        String res = null;
        switch (column) {
            case 0:
                res = t("lima.table.label");
                break;
            case 1:
                res = t("lima.table.account");
                break;
        }
        return res;
    }

    @Override
    public int getChildCount(Object node) {
        int result = 0;
        if (node == getRoot()) {
            try {
                result = vatStatementService.
                        getChildrenVatStatement(null).size();
            } catch (LimaException eee) {
                log.debug("Can't count child", eee);
            }
        } else {
            VatStatement parentVatStatementHeader =
                    (VatStatement) node;
            try {
                result = vatStatementService.getChildrenVatStatement(
                        parentVatStatementHeader).size();
            } catch (LimaException eee) {
                log.debug("Can't count child", eee);
            }
        }
        return result;
    }

    @Override
    public Object getChild(Object parent, int index) {
        Object result = null;
        if (parent == getRoot()) {
            try {
                List<VatStatement> vatStatements =
                        vatStatementService.getChildrenVatStatement(null);
                result = vatStatements.get(index);
            } catch (LimaException eee) {
                log.debug("Can't get child", eee);
            }
        } else {
            VatStatement parentVatStatement =
                    (VatStatement) parent;
            try {
                List<VatStatement> vatStatements = vatStatementService.
                        getChildrenVatStatement(parentVatStatement);
                result = vatStatements.get(index);
            } catch (LimaException eee) {
                log.debug("Can't get child", eee);
            }
        }
        return result;
    }

    @Override
    public int getIndexOfChild(Object parent, Object child) {
        int result = 0;

        if (parent == getRoot()) {
            try {
                List<VatStatement> vatStatements =
                        vatStatementService.getChildrenVatStatement(null);
                result = vatStatements.indexOf(child);
            } catch (LimaException eee) {
                log.debug("Can't get index child", eee);
            }
        } else {
            VatStatement parentVatStatement =
                    (VatStatement) parent;
            try {
                List<VatStatement> vatStatements = vatStatementService.
                        getChildrenVatStatement(parentVatStatement);
                result = vatStatements.indexOf(child);
            } catch (LimaException eee) {
                log.debug("Can't get index child", eee);
            }
        }
        return result;
    }

    @Override
    public Object getValueAt(Object node, int column) {
        Object result = "n/a";
        VatStatement vatStatement = (VatStatement) node;
        switch (column) {
            case 0:
                result = vatStatement.getLabel();
                break;
            case 1:
                result = vatStatement.getAccounts();
                break;
        }
        return result;
    }

    @Override
    public boolean isCellEditable(Object node, int column) {
        return column == 1 && isLeaf(node);
    }

    @Override
    public boolean isLeaf(Object node) {
        return getChildCount(node) == 0;
    }


    /** Refresh VatStatementChart. */
    public void refreshTree() throws LimaException {

        modelSupport.fireNewRoot();
    }


    /**
     * Add VatStatement(path can be null).
     *
     * @param path
     * @param vatStatement
     * @throws LimaException
     */
    public void addVatStatement(TreePath path, VatStatement vatStatement) throws LimaException {
        // Calling account service
        //fixed bug when path is null
        VatStatement parentVatStatementHeader = null;
        if (path != null) {
            parentVatStatementHeader =
                    (VatStatement) path.getLastPathComponent();
        }
        vatStatementService.createVatStatement(
                parentVatStatementHeader, vatStatement);
        modelSupport.fireTreeStructureChanged(path);
    }


    /**
     * Update vatStatement
     *
     * @param path
     * @param vatStatement
     * @throws LimaException
     */
    public void updateVatStatement(TreePath path, VatStatement vatStatement) throws LimaException {

        vatStatementService.updateVatStatement(vatStatement);
        modelSupport.fireTreeStructureChanged(path);
    }

    /**
     * Remove vatStatement
     *
     * @param path
     * @param vatStatement
     * @throws LimaException
     */
    public void removeVatStatementObject(TreePath path, VatStatement vatStatement) throws LimaException {
        // Calling account service
        int index = getIndexOfChild(
                path.getParentPath().getLastPathComponent(), vatStatement);
        vatStatementService.removeVatStatement(vatStatement);
        modelSupport.fireChildRemoved(path.getParentPath(), index, vatStatement);
    }


    @Override
    public void setValueAt(Object value, Object node, int column) {
        VatStatement vatStatement = (VatStatement) node;
        vatStatement.setAccounts((String) value);
        try {
            vatStatementService.updateVatStatement(vatStatement);
        } catch (LimaException eee) {
            log.debug("Can't update vatStatement", eee);
        }
    }


}
