/*
 * #%L
 * Lima Swing
 * 
 * $Id: FiscalPeriodViewHandler.java 3777 2014-04-14 07:29:08Z sbavencoff $
 * $HeadURL: https://svn.chorem.org/lima/tags/lima-0.7.3/lima-swing/src/main/java/org/chorem/lima/ui/fiscalperiod/FiscalPeriodViewHandler.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.fiscalperiod;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.LimaException;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.business.api.FinancialTransactionService;
import org.chorem.lima.business.api.FiscalPeriodService;
import org.chorem.lima.entity.EntryBook;
import org.chorem.lima.entity.EntryBookImpl;
import org.chorem.lima.entity.FinancialTransaction;
import org.chorem.lima.entity.FiscalPeriod;
import org.chorem.lima.entity.FiscalPeriodImpl;
import org.chorem.lima.service.LimaServiceFactory;
import org.nuiton.util.DateUtil;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import static org.nuiton.i18n.I18n.t;

/**
 * Handler pour la gestion des exercices (creation/fermeture).
 *
 * @author chatellier
 * @version $Revision: 3777 $
 * 
 * Last update : $Date: 2014-04-14 09:29:08 +0200 (Mon, 14 Apr 2014) $
 * By : $Author: sbavencoff $
 */
public class FiscalPeriodViewHandler implements ServiceListener {

    private static final Log log = LogFactory.getLog(FiscalPeriodViewHandler.class);

    protected FiscalPeriodView view;

    /** Service. */
    protected FiscalPeriodService fiscalPeriodService;

    protected FinancialTransactionService financialTransactionService;

    /**
     * Constructor.
     * 
     * @param view managed view
     */
    public FiscalPeriodViewHandler(FiscalPeriodView view) {
        this.view = view;

        fiscalPeriodService = LimaServiceFactory.getService(FiscalPeriodService.class);
        financialTransactionService = LimaServiceFactory.getService(FinancialTransactionService.class);

        LimaServiceFactory.addServiceListener(FiscalPeriodService.class, this);
    }

    public void init() {

        InputMap inputMap = view.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        ActionMap actionMap = view.getActionMap();

        // add action on Ctrl + N
        String binding = "new-FiscalPeriod";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_N, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                addFiscalPeriod();
            }
        });

        // add action on Ctrl + M
        binding = "update-FiscalPeriod";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_M, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                updateFiscalPeriod();
            }
        });

        // add action on Delete
        binding = "remove-FiscalPeriod";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                deleteFiscalPeriod();
            }
        });

        // add action on Ctrl + B
        binding = "close-FiscalPeriod";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_B, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                blockFiscalPeriod();
            }
        });


        loadAllFiscalPeriod();
    }

    /**
     * Reload fiscal period list from service and update ui.
     */
    public void loadAllFiscalPeriod() {
        List<FiscalPeriod> periods = fiscalPeriodService.getAllFiscalPeriods();

        FiscalPeriodTableModel tableModel = view.getFiscalPeriodTableModel();
        tableModel.setValues(periods);
    }

    /**
     * Display add period form and perform add.
     */
    public void addFiscalPeriod() {

        FiscalPeriodTableModel tableModel = view.getFiscalPeriodTableModel();

        // set begin date picker
        Date beginDate = null;
        Date endDate;
        try {
            FiscalPeriod result = fiscalPeriodService.getLastFiscalPeriod();
            if (result != null){
                beginDate = DateUtils.addDays(result.getEndDate(), 1);
            }
        }
        catch (LimaException eee){
            if(log.isDebugEnabled()){
                log.debug("Can't get last fiscal period", eee);
            }

        }
        if (beginDate == null) {
            Calendar calendarBegin = Calendar.getInstance();
            // set begindate to JAN 1 - 0:00.000 of this years
            beginDate = calendarBegin.getTime();
            beginDate = DateUtils.truncate(beginDate, Calendar.YEAR);
        }

        // get end date
        endDate = DateUtils.addYears(beginDate, 1);
        endDate = DateUtils.addDays(endDate, -1);


        AddPeriod addPeriodDialog = new AddPeriod(view);
        addPeriodDialog.setTitle(t("lima.ui.fiscalperiod.addfiscalperiodtitle"));
        addPeriodDialog.getBeginDatePicker().setDate(beginDate);
        addPeriodDialog.getEndDatePicker().setDate(endDate);
        addPeriodDialog.setLocationRelativeTo(view);
        addPeriodDialog.setVisible(true);

        if (checkFiscalPeriod(addPeriodDialog)) {
            FiscalPeriod fiscalPeriod = new FiscalPeriodImpl();
            fiscalPeriod.setBeginDate(addPeriodDialog.getBeginDatePicker().getDate());
            fiscalPeriod.setEndDate(addPeriodDialog.getEndDatePicker().getDate());
            fiscalPeriod = fiscalPeriodService.createFiscalPeriod(fiscalPeriod);
            tableModel.addValue(fiscalPeriod);
        }
    }

    protected boolean checkFiscalPeriod(AddPeriod addPeriodDialog) {
        Date beginDate = addPeriodDialog.getBeginDatePicker().getDate();
        Date endDate = addPeriodDialog.getEndDatePicker().getDate();

        String title = addPeriodDialog.getTitle();

        if (!addPeriodDialog.isValidate()) {
            return false;
        } else if (endDate.before(beginDate)) {
            JOptionPane.showMessageDialog(view,
                    t("lima.ui.fiscalperiod.addfiscalperiod.endAfterBegin"),
                    title,
                    JOptionPane.ERROR_MESSAGE);
            return false;
        } else {
            //check if fiscalperiod have 12 months, ask a confirmation
            int nbMonth = DateUtil.getDifferenceInMonths(beginDate, endDate);
            int response = JOptionPane.YES_OPTION;
            if (nbMonth != 12) {
                response = JOptionPane.showConfirmDialog(view,
                        t("lima.ui.fiscalperiod.addfiscalperiod.morethan12"),
                        title,
                        JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
            }
            return (response == JOptionPane.YES_OPTION);


        }


    }

    /**
     * Display add period form and perform add.
     */
    public void updateFiscalPeriod() {

        FiscalPeriodTable table = view.getFiscalPeriodTable();
        FiscalPeriodTableModel tableModel = view.getFiscalPeriodTableModel();

        int selectedRow = table.getSelectedRow();

        if (selectedRow >=0 ) {

            FiscalPeriod selectedFiscalPeriod = tableModel.get(selectedRow);

            if (!selectedFiscalPeriod.isLocked()) {

                AddPeriod addPeriodDialog = new AddPeriod(view);
                addPeriodDialog.setTitle(t("lima.ui.fiscalperiod.modifyfiscalperiodtitle"));
                addPeriodDialog.getBeginDatePicker().setDate(selectedFiscalPeriod.getBeginDate());
                addPeriodDialog.getEndDatePicker().setDate(selectedFiscalPeriod.getEndDate());
                addPeriodDialog.setModifyPeriod(true);
                addPeriodDialog.setLocationRelativeTo(view);
                addPeriodDialog.setVisible(true);

                if (checkFiscalPeriod(addPeriodDialog)) {
                    selectedFiscalPeriod.setEndDate(addPeriodDialog.getEndDatePicker().getDate());
                    fiscalPeriodService.updateEndDate(selectedFiscalPeriod);
                    tableModel.fireTableRowsUpdated(selectedRow, selectedRow);
                }
            }
        }
    }

    /**
     * Undisplay period and perform delete.
     */
    public void deleteFiscalPeriod() {

        FiscalPeriodTable table = view.getFiscalPeriodTable();
        FiscalPeriodTableModel tableModel = view.getFiscalPeriodTableModel();

        int selectedRow = table.getSelectedRow();

        if (selectedRow >=0 ) {

            FiscalPeriod selectedFiscalPeriod = tableModel.get(selectedRow);

            if (!selectedFiscalPeriod.isLocked()
                    && financialTransactionService.getAllFinancialTransactions(selectedFiscalPeriod).isEmpty()) {

                //check if the user want to delete the fiscal year
                int response = JOptionPane.showConfirmDialog(view,
                        t("lima.ui.fiscalperiod.delete.confirmation"),
                        t("lima.ui.fiscalperiod.delete.title"), JOptionPane.YES_NO_OPTION,
                        JOptionPane.WARNING_MESSAGE);

                if (response == JOptionPane.YES_OPTION) {
                    fiscalPeriodService.deleteFiscalPeriod(selectedFiscalPeriod);

                    tableModel.remove(selectedFiscalPeriod);
                    view.setBlockEnabled(false);
                    view.setDeleteEnabled(false);
                }
            }
        }
    }

    /**
     * Block selected fiscal period,
     * and make new fiscal period and retained earnings
     * after user confirmations
     */
    public void blockFiscalPeriod() {
        FiscalPeriodTable table = view.getFiscalPeriodTable();
        final FiscalPeriodTableModel tableModel = view.getFiscalPeriodTableModel();

        final int selectedRow = table.getSelectedRow();

        if (selectedRow >= 0) {

            final FiscalPeriod selectedFiscalPeriod = tableModel.get(selectedRow);

            if (!selectedFiscalPeriod.isLocked()) {

                //check if the user want to block the fiscal year
                int answerBlock = JOptionPane.showConfirmDialog(view,
                        t("lima.ui.fiscalperiod.block.confirmation"),
                        t("lima.ui.fiscalperiod.block.title"), JOptionPane.YES_NO_OPTION,
                        JOptionPane.WARNING_MESSAGE);

                if (answerBlock == JOptionPane.YES_OPTION) {

                    //report if they are at least one transaction to be report
                    if (fiscalPeriodService.isRetainedEarnings(selectedFiscalPeriod)) {

                        // find next fiscal year
                        FiscalPeriod lastFiscalPeriod = fiscalPeriodService.getLastFiscalPeriod();
                        if (selectedFiscalPeriod.equals(lastFiscalPeriod)) {

                            //check if the user wants to create a new fiscal year
                            int answerCreate = JOptionPane.showConfirmDialog(view,
                                    t("lima.ui.fiscalperiod.block.newyear"),
                                    t("lima.ui.fiscalperiod.block.title"), JOptionPane.YES_NO_OPTION,
                                    JOptionPane.QUESTION_MESSAGE);

                            // create a new fiscal year
                            if (answerCreate == JOptionPane.YES_OPTION) {
                                addFiscalPeriod();
                            }
                        }


                        lastFiscalPeriod = fiscalPeriodService.getLastFiscalPeriod();
                        // check if next fiscal year don't exist
                        if (selectedFiscalPeriod.equals(lastFiscalPeriod)) {

                            JOptionPane.showMessageDialog(view,
                                    t("lima.ui.fiscalperiod.block.cantBalanceIfNextPeriodNotExist"),
                                    t("lima.ui.fiscalperiod.block.title"),
                                    JOptionPane.ERROR_MESSAGE);

                        } else {

                            //check if the user wants to report datas
                            int answerRetainedEarnings = JOptionPane.showConfirmDialog(view,
                                    t("lima.ui.fiscalperiod.block.addretainedearnings"),
                                    t("lima.ui.fiscalperiod.block.title"), JOptionPane.YES_NO_OPTION,
                                    JOptionPane.QUESTION_MESSAGE);

                            if (answerRetainedEarnings == JOptionPane.YES_OPTION) {
                                //Sets EntryBook
                                EntryBook newEntryBook = new EntryBookImpl();
                                RetainedEarningsEntryBookForm entryBookForm =
                                        new RetainedEarningsEntryBookForm(view);
                                entryBookForm.setEntryBook(newEntryBook);

                                // jaxx constructor don't call super() ?
                                entryBookForm.setLocationRelativeTo(view);
                                entryBookForm.setVisible(true);
                                final EntryBook entryBook = entryBookForm.getEntryBook();

                                if (entryBook != null && StringUtils.isNotBlank(entryBook.getCode())) {

                                    final RetainedEarningsWait retainedEarningsWait = new RetainedEarningsWait(view);
                                    retainedEarningsWait.setLocationRelativeTo(view);

                                    new SwingWorker<FiscalPeriod, Void>() {
                                        @Override
                                        protected FiscalPeriod doInBackground() throws Exception {
                                            FiscalPeriod fiscalPeriodBlocked = fiscalPeriodService.retainedEarningsAndBlockFiscalPeriod(selectedFiscalPeriod, entryBook, true);
                                            tableModel.setValue(selectedRow, fiscalPeriodBlocked);

                                            view.setBlockEnabled(false);
                                            view.setDeleteEnabled(false);

                                            return fiscalPeriodBlocked;
                                        }

                                        @Override
                                        protected void done() {
                                            retainedEarningsWait.setVisible(false);
                                        }
                                    }.execute();
                                    retainedEarningsWait.setVisible(true);

                                } else {

                                    JOptionPane.showMessageDialog(view,
                                            t("lima.ui.fiscalperiod.block.cantBalanceNotBook"),
                                            t("lima.ui.fiscalperiod.block.title"),
                                            JOptionPane.ERROR_MESSAGE);
                                }

                            } else {

                                JOptionPane.showMessageDialog(view,
                                        t("lima.ui.fiscalperiod.block.cantBlockNotBalance"),
                                        t("lima.ui.fiscalperiod.block.title"),
                                        JOptionPane.ERROR_MESSAGE);
                            }
                        }

                    } else {

                        FiscalPeriod fiscalPeriodBlocked = fiscalPeriodService.blockFiscalPeriod(selectedFiscalPeriod);
                        tableModel.setValue(selectedRow, fiscalPeriodBlocked);

                        view.setBlockEnabled(false);
                        view.setDeleteEnabled(false);

                    }
                }
            }
        }
    }

    @Override
    public void notifyMethod(String serviceName, String methodeName) {
        //refresh on import datas
        if (methodeName.contains("importAll")) {
            loadAllFiscalPeriod();
        }
    }
    
    /**
     * Manage the differents buttons for fiscal period
     * */
    protected void onSelectionChanged(ListSelectionEvent listSelectionEvent){
        if(!listSelectionEvent.getValueIsAdjusting()) {
            DefaultListSelectionModel listSelectionModel = (DefaultListSelectionModel)listSelectionEvent.getSource();

            int selectedRow = listSelectionEvent.getFirstIndex();
            if (!listSelectionModel.isSelectedIndex(selectedRow)) {
                selectedRow = listSelectionEvent.getLastIndex();
            }

            FiscalPeriodTable table = view.getFiscalPeriodTable();
            FiscalPeriodTableModel tableModel = view.getFiscalPeriodTableModel();
            if (tableModel.getRowCount() != selectedRow) {
                FiscalPeriod fiscalPeriodAt = tableModel.get(selectedRow);

                List<FinancialTransaction> financialTransactionList = financialTransactionService.getAllFinancialTransactions(fiscalPeriodAt);

                boolean enableBlock = false;
                boolean enableDelete = false;

                if (log.isDebugEnabled()) {
                    log.debug("reloadEnablingButton");
                }

              if (selectedRow != -1){
                  if (log.isDebugEnabled()) {
                      log.debug("selectedFiscalPeriod != null");
                  }

                  //no action possible for fiscal period closed
                  if (!fiscalPeriodAt.getLocked()){
                      if (log.isDebugEnabled()) {
                          log.debug("selectedFiscalPeriod not locked");
                      }
                      enableBlock = true;

                      //only an empty and open fiscal period may be deleted
                      enableDelete = financialTransactionList.size() <= 0;
                   }
                }
                view.setBlockEnabled(enableBlock);
                view.setDeleteEnabled(enableDelete);
            }
        }
    }  
        
}
