/*
 * #%L
 * Lima Swing
 * 
 * $Id: FiscalPeriodTableModel.java 3229 2011-07-18 14:17:31Z vsalaun $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.5/lima-swing/src/main/java/org/chorem/lima/ui/fiscalperiod/FiscalPeriodTableModel.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.fiscalperiod;

import static org.nuiton.i18n.I18n._;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;


import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.table.AbstractTableModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.FinancialTransactionServiceMonitorable;
import org.chorem.lima.business.FiscalPeriodServiceMonitorable;
import org.chorem.lima.business.ImportServiceMonitorable;
import org.chorem.lima.business.LimaException;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.business.utils.FiscalPeriodComparator;
import org.chorem.lima.entity.FinancialTransaction;
import org.chorem.lima.entity.FiscalPeriod;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.util.ErrorHelper;
import org.nuiton.util.Resource;

/**
 * TODO add comment here.
 * 
 * @author chatellier
 * @version $Revision: 3229 $
 * 
 * Last update : $Date: 2011-07-18 16:17:31 +0200 (Mon, 18 Jul 2011) $
 * By : $Author: vsalaun $
 */
public class FiscalPeriodTableModel extends AbstractTableModel implements ServiceListener{

    /** serialVersionUID. */
    private static final long serialVersionUID = 77027335135838258L;

    private static final Log log = LogFactory.getLog(FiscalPeriodTableModel.class);

    protected FiscalPeriodServiceMonitorable fiscalPeriodService;
    
    protected FinancialTransactionServiceMonitorable financialTransactionService;
    
    /** data cache */
    protected List<Object> cacheDataList;

    public FiscalPeriodTableModel() {
        fiscalPeriodService =
            LimaServiceFactory.getInstance().getService(
                    FiscalPeriodServiceMonitorable.class);
        fiscalPeriodService.addListener(this);
            LimaServiceFactory.getInstance().getService(
                ImportServiceMonitorable.class).addListener(this);
        financialTransactionService =
            LimaServiceFactory.getInstance().getService(
                    FinancialTransactionServiceMonitorable.class);
        financialTransactionService.addListener(this);
        cacheDataList = null;
        }
    
    protected List<Object> getDataList() {
        List<Object> results = new ArrayList<Object>();
        try {
            List<FiscalPeriod> fiscalPeriod =
                fiscalPeriodService.getAllFiscalPeriods();
            Collections.sort(fiscalPeriod, new FiscalPeriodComparator());
            results.addAll(fiscalPeriod);
        }
        catch (LimaException eee) {
            if (log.isErrorEnabled()) {
                log.debug("Can't update model", eee);
            }
            ErrorHelper.showErrorDialog(_("lima.fiscalperiod.listerror"), eee);
        }
        return results;
    }

    public void refresh(){
        cacheDataList = getDataList();
        fireTableDataChanged();
    }
    
    @Override
    public int getRowCount() {
        int result = 0;
        
        if (cacheDataList != null) {
            result = cacheDataList.size();
        }

        return result;
    }

    @Override
    public int getColumnCount() {
        return 2;
    }

    
    public Object getElementAt(int row){
        
        Object currentRow = null;
        
        if (cacheDataList != null) {
            currentRow = cacheDataList.get(row);
        }
        
        return currentRow;
    }
    
    @Override
    public String getColumnName(int columnIndex) {
        
        String result = "n/a";
        
        switch(columnIndex) {
        case 0 :
            result = _("lima.table.fiscalperiod");
            break;
        case 1:
            result = _("lima.table.closure");
            break;
        }

        return result;
    }

    @Override
    public Class<?> getColumnClass(int columnIndex) {
        // both String
        return String.class;
    }


    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return false;
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {

        Object result = "n/a";

        List<Object> periods;
        periods = cacheDataList;
        FiscalPeriod fiscalPeriod = (FiscalPeriod)periods.get(rowIndex);
        
        if (fiscalPeriod != null){
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("dd MMM yyyy");                

            switch (columnIndex) {
            case 0:
                result = simpleDateFormat.
                        format(fiscalPeriod.getBeginDate())
                    + " - " + simpleDateFormat.
                            format(fiscalPeriod.getEndDate());
                break;
            case 1:
                if(fiscalPeriod.getLocked()){
                    result = _("lima.common.closed");
                }
                else{
                    result = _("lima.common.open");
                }
                break;
            }
        }

        return result;
    }
    
    public FiscalPeriod getFiscalPeriodAtRow(int row) throws LimaException {
        FiscalPeriod fiscalPeriod = null;
        fiscalPeriod = (FiscalPeriod) cacheDataList.get(row);        
        return fiscalPeriod;
    }
    
    /**
     * @param period
     */
    public void addFiscalPeriod(FiscalPeriod period) throws LimaException {
        
        try {
            // Calling fiscal period service
            fiscalPeriodService.createFiscalPeriod(period);
            //on recharge la liste
            cacheDataList = getDataList();
            int row = getDataList().indexOf(period);
            fireTableRowsInserted(row, row);
        }
        catch (LimaException eee) {
            if (log.isErrorEnabled()) {
                log.debug("Can't create FiscalPeriod", eee);
            }
            ErrorHelper.showErrorDialog(_("lima.fiscalperiod.creationerror"), eee);
        }
    }
    
    public void blockFiscalPeriod(FiscalPeriod fiscalPeriod) throws LimaException {
        List<FinancialTransaction> financialTransactionsUnbal = 
            financialTransactionService.getAllInexactFinancialTransactions(fiscalPeriod);
        if (financialTransactionsUnbal.size() > 0) {
            JFrame f = new JFrame();
                f.setIconImage(Resource.getIcon("icons/lima.png").getImage());
                JOptionPane.showMessageDialog(
                f,
                _("lima.fiscalperiod.unbalancedtransactions"),
                _("lima.common.error"),
                JOptionPane.ERROR_MESSAGE);
            f.dispose();
        } else {
            fiscalPeriodService.blockFiscalPeriod(fiscalPeriod);
            refresh();
        }
    }

    @Override
    public void notifyMethod(String serviceName, String methodeName) {
        //refresh on import datas
        if (methodeName.contains("importAll")){
            refresh();
        }        
    }
}
