/*
 * #%L
 * Lima :: Swing
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2008 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package org.chorem.lima.ui.celleditor;

import java.awt.Component;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.math.BigDecimal;
import java.util.EventObject;

import javax.swing.AbstractCellEditor;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.event.AncestorEvent;
import javax.swing.event.AncestorListener;
import javax.swing.table.TableCellEditor;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.LimaContext;
import org.chorem.lima.business.LimaException;

public class BigDecimalTableCellEditor extends AbstractCellEditor
        implements TableCellEditor, FocusListener, AncestorListener, KeyListener {

    private static final Log log = LogFactory.getLog(BigDecimalTableCellEditor.class);
    private static final long serialVersionUID = 1L;
    protected final JTextField textField;
    private boolean keyPressed;

    /**
     * constructor
     */
    public BigDecimalTableCellEditor() {
        textField = new JTextField();
        textField.setEditable(true);
        textField.setHorizontalAlignment(SwingConstants.RIGHT);
        textField.addFocusListener(this);
        textField.addAncestorListener(this);
        textField.addKeyListener(this);
        keyPressed = false;
    }
    
    @Override
    public Component getTableCellEditorComponent(JTable table, Object value,
            boolean isSelected, int row, int column) {
        textField.setText(value.toString());
        return textField;
    }

    // the editor should return a BigDecimal
    public Object getCellEditorValue() {
        String textValue = textField.getText().trim();
        //replace all comma by full stop
        textValue = textValue.replaceAll(",", ".");
        if (textValue.equals("")) {
            return BigDecimal.ZERO;
        } else {
            //set 0 in case the user type in a non number format
            BigDecimal bdValue = BigDecimal.ZERO;
            try {
                bdValue = new BigDecimal(textValue);
            } catch (NumberFormatException eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't set " + textValue + " to a BigDecimal", eee);
                }
            }
            //round half up the number using the scale given by the configuration
            bdValue = bdValue.setScale(LimaContext.getContext().getConfig().getScale(),
                BigDecimal.ROUND_HALF_UP);
            return bdValue;
        }
    }
    
    @Override
    public boolean isCellEditable(EventObject evt) {
        return !(evt instanceof MouseEvent) || ((MouseEvent) evt).getClickCount() == 2;
    }
    
    /**
     * Listeners
     */
    @Override
    public void focusGained(FocusEvent e) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                textField.selectAll();
           }
        });
    }
    
    @Override
    public void focusLost(FocusEvent e) {
        keyPressed = false;
    }
    
    @Override
    public void ancestorAdded(final AncestorEvent e) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                textField.requestFocus();
           }
        });
    }

    @Override
    public void ancestorRemoved(AncestorEvent e) {
    }

    @Override
    public void ancestorMoved(AncestorEvent e) {
    }

    @Override
    public void keyTyped(KeyEvent e) {

    }

    @Override
    public void keyPressed(KeyEvent e) {
	}

    @Override
    public void keyReleased(KeyEvent e) {
        // replace all the cell content only if all the text has been selected
        if (keyPressed == false
                && textField.getSelectionStart() == 0
                && textField.getSelectionEnd() == textField.getText().length()) {
            // delete the the cell content
            if (e.getKeyChar() == KeyEvent.VK_BACK_SPACE) {
                textField.setText("");
                keyPressed = true;
            // replace the content by the char typed in
            } else if (String.valueOf(e.getKeyChar()).matches("[a-zA-z0-9]")) {
                textField.setText(String.valueOf(e.getKeyChar()));
                keyPressed = true;
            }
        }
    }

}
