/*
 * #%L
 * Lima Swing
 * 
 * $Id: AccountTreeTableModel.java 3051 2010-11-29 14:57:16Z echatellier $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.4.3/lima-swing/src/main/java/org/chorem/lima/ui/account/AccountTreeTableModel.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.account;

import static org.nuiton.i18n.I18n._;
import java.util.List;
import javax.swing.tree.TreePath;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.AccountServiceMonitorable;
import org.chorem.lima.business.LimaBusinessException;
import org.chorem.lima.business.LimaException;
import org.chorem.lima.entity.Account;
import org.chorem.lima.entity.AccountImpl;
import org.chorem.lima.service.LimaServiceFactory;
import org.jdesktop.swingx.treetable.AbstractTreeTableModel;

/**
 * Tree table model for account edition.
 * 
 * @author ore
 * @author chatellier
 * @version $Revision: 3051 $
 * 
 * Last update : $Date: 2010-11-29 15:57:16 +0100 (lun., 29 nov. 2010) $
 * By : $Author: echatellier $
 */
public class AccountTreeTableModel extends AbstractTreeTableModel {
    
    /** log. */
    private static final Log log = LogFactory.getLog(AccountViewHandler.class);
    
    /** Account service. */
    protected final AccountServiceMonitorable accountService;

    /**
     * Model constructor. Init account service used here.
     */
    public AccountTreeTableModel() {
        // root
        super(new AccountImpl());
        // Gets factory service        
        accountService =
            LimaServiceFactory.getInstance().getService(
                    AccountServiceMonitorable.class);
        }


    @Override
    public int getColumnCount() {
        return 2;
    }

    @Override
    public String getColumnName(int column) {
        String res = null;
        switch (column) {
            case 0:
                res = _("lima.table.number");
                break;
            case 1:
                res = _("lima.table.label");
                break;
        }
        return res;
    }

    @Override
    public int getChildCount(Object node) {
        int result = 0;
        if (node == getRoot()) {
            try {
                result = accountService.getChildrenAccounts(null).size();
            } catch (LimaException eee) {
                log.debug("Can't count child", eee);
            }
        }
        else {
            Account parentAccount = (Account) node;
            try {
                result = accountService.getChildrenAccounts(parentAccount).size();
            } catch (LimaException eee) {
                log.debug("Can't count child", eee);
            }
        }
        return result;
    }

    @Override
    public Object getChild(Object parent, int index) {
        Object result = null;
        if (parent == getRoot()) {
            try {
                List<Account> allAccounts =
                    accountService.getChildrenAccounts(null);
                result = allAccounts.get(index);
            } catch (LimaException eee) {
                log.debug("Can't get child", eee);
            }
        }
        else {
            Account parentAccount = (Account) parent;

            // FIXME sub account is a collection ?
            try {
                List<Account> subaccounts =
                    accountService.getChildrenAccounts(parentAccount);
                result = subaccounts.get(index);
            } catch (LimaException eee) {
                log.debug("Can't get child", eee);
            }
        }
        return result;
    }

    @Override
    public int getIndexOfChild(Object parent, Object child) {
        int result = 0;
        Account parentAccount = (Account) parent;
        Account childAccount = (Account) child;
        
        if (parent == getRoot()) {
            try {
                List<Account> allAccounts =
                    accountService.getChildrenAccounts(null);
                result = allAccounts.indexOf(child);
            } catch (LimaException eee) {
                log.debug("Can't get index child", eee);
            }
        }
        else {
            // FIXME sub account is a collection ?
            try {
                List<Account> subaccounts =
                    accountService.getChildrenAccounts(parentAccount);
                result = subaccounts.indexOf(childAccount);
            } catch (LimaException eee) {
                log.debug("Can't get index child", eee);
            }
        }
        return result;
    }
    
    @Override
    public Object getValueAt(Object node, int column) {
        Object result = "n/a";
        if (node instanceof Account) {
            Account account = (Account) node;
            switch (column) {
                case 0:
                    result = account.getAccountNumber();
                    break;
                case 1:
                    result = account.getLabel();
                    break;
            }
        }
        return result;
    }

    @Override
    public boolean isCellEditable(Object node, int column) {
        return false;
    }

    @Override
    public boolean isLeaf(Object node) {
        return getChildCount(node) == 0;
    }

    /**
     * Add account (path can be null).
     * 
     * @param path
     * @param account
     * @throws LimaException
     */
    public void addAccount(TreePath path, Account account) throws LimaException {
        // Calling account service
        Account parentAccount = (Account)path.getLastPathComponent();
        if (parentAccount == getRoot()) {
            parentAccount = null;
        }
        accountService.createAccount(parentAccount, account);
        modelSupport.fireTreeStructureChanged(path);
    }

    /**
     * Add account (path can be null).
     * 
     * @param path
     * @param account
     * @throws LimaException
     */
    public void addSubLedger(TreePath path, Account account) throws LimaException {
        // Calling account service
        Account parentAccount = (Account)path.getLastPathComponent();
        if (parentAccount == getRoot()) {
            parentAccount = null;
        }
        accountService.createSubLedger(parentAccount, account);
        modelSupport.fireTreeStructureChanged(path);
    }
    
    /**
     * Update account.
     * 
     * @param path
     * @param account
     * @throws LimaException
     */
    public void updateAccount(TreePath path, Account account) throws LimaException {
        // Calling account service
        //FIXME Force fire for no update if error. Where is the cache ?
        try{
            accountService.updateAccount(account);
        }
        catch (LimaBusinessException eee) {
            modelSupport.fireTreeStructureChanged(path);
            throw eee;
        }
        modelSupport.fireTreeStructureChanged(path);
    }
    
    /**
     * Refresh accountschart.
     *
     */
    public void refreshTree() throws LimaException {

        modelSupport.fireNewRoot();
    }

    /**
     * Remove account.
     * 
     * @param path
     * @param account
     * @throws LimaException
     */
    public void removeAccount(TreePath path, Account account) throws LimaException {
        // Calling account service
        int index = getIndexOfChild(
                path.getParentPath().getLastPathComponent(), account);
        accountService.removeAccount(account);
        modelSupport.fireChildRemoved(path.getParentPath(), index, account);
    }
}
