/*
 * #%L
 * Refcomp :: services
 * 
 * $Id: StorageServiceImpl.java 79 2011-01-27 17:45:56Z jcouteau $
 * $HeadURL: svn+ssh://jcouteau@scm.mulcyber.toulouse.inra.fr/svnroot/refcomp/tags/refcomp-0.1/refcomp-services/src/main/java/fr/inra/refcomp/services/StorageServiceImpl.java $
 * %%
 * Copyright (C) 2010 Inra
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.inra.refcomp.services;

import fr.inra.refcomp.entities.Agent;
import fr.inra.refcomp.entities.AgentSkill;
import fr.inra.refcomp.entities.AgentSkillImpl;
import fr.inra.refcomp.entities.EntitiesList;
import fr.inra.refcomp.entities.Cati;
import fr.inra.refcomp.entities.CatiImpl;
import fr.inra.refcomp.entities.Department;
import fr.inra.refcomp.entities.DepartmentImpl;
import fr.inra.refcomp.entities.Frequency;
import fr.inra.refcomp.entities.FrequencyImpl;
import fr.inra.refcomp.entities.Nature;
import fr.inra.refcomp.entities.Skill;
import fr.inra.refcomp.entities.SkillImpl;
import fr.inra.refcomp.entities.Unit;
import fr.inra.refcomp.entities.UnitDeptLink;
import fr.inra.refcomp.entities.UnitDeptLinkImpl;
import fr.inra.refcomp.entities.UnitImpl;
import fr.inra.refcomp.entities.User;
import fr.inra.refcomp.entities.UserImpl;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.validator.NuitonValidator;
import org.nuiton.validator.NuitonValidatorFactory;
import org.nuiton.validator.NuitonValidatorResult;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.addons.WikittyImportExportService;
import org.nuiton.wikitty.entities.BusinessEntity;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.entities.WikittyAuthorisation;
import org.nuiton.wikitty.entities.WikittyAuthorisationImpl;
import org.nuiton.wikitty.entities.WikittyGroup;
import org.nuiton.wikitty.entities.WikittyGroupImpl;
import org.nuiton.wikitty.entities.WikittyTreeNode;
import org.nuiton.wikitty.entities.WikittyUser;
import org.nuiton.wikitty.entities.WikittyUserImpl;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.operators.Element;
import org.nuiton.wikitty.dto.DTOHelper;
import org.nuiton.wikitty.services.WikittySecurityUtil;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.l_;

/**
 * User: couteau
 * Date: 30 nov. 2010
 */
public class StorageServiceImpl implements StorageService{

    /**
     * log.
     */
    private static final Log log = LogFactory.getLog(StorageServiceImpl.class);

    public static final String PRIVATE_GROUP = "private";

    WikittyProxy proxy;

    ApplicationConfig config;

    StorageServiceImpl(WikittyProxy proxy, ApplicationConfig config) {
        this.proxy = proxy;
        this.config = config;
    }

    @Override
    public WikittyProxy getProxy() {
        return proxy;
    }

    @Override
    public void setProxy(WikittyProxy proxy) {
        this.proxy = proxy;
    }

    /**
     * Get all the catis.
     *
     * @return a list containing all the catis.
     */
    public List<Cati> getCatis() {

        //Create criteria : everything that got the cati extension
        Search search = Search.query().eq(Element.ELT_EXTENSION, Cati.EXT_CATI);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(Cati.FIELD_CATI_NAME);

        //get results
        PagedResult<Cati> catiPagedResult = proxy.findAllByCriteria(Cati.class, criteria);

        List<Cati> catis = new ArrayList<Cati>();

        if (catiPagedResult != null) {
            for (Cati cati:catiPagedResult.getAll()){
                catis.add(DTOHelper.toDto(cati));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Catis list : " + catis);
        }
        return catis;
    }

    /**
     * Save a cati in database.
     *
     * @param cati the cati to save
     * @return the saved cati
     * @throws AlreadyExistingException if a cati with the same name already
     *                                  exist in database
     */
    public Cati saveCati(Locale locale, Cati cati)
            throws AlreadyExistingException, InvalidEntityException {

        /*if (exist(cati)) {
            throw new AlreadyExistingException("Cati with name " +
                    cati.getName() + " already exist in database");
        }*/

        NuitonValidator<Cati> validator =
                NuitonValidatorFactory.newValidator(Cati.class);

        NuitonValidatorResult result = validator.validate(cati);

        if (!result.isValid()) {
            String errorMessage = "";
            List<String> fields = result.getFieldsForError();
            for (String field : fields) {
                List<String> messages = result.getErrorMessages(field);
                for (String message : messages) {
                    errorMessage += l_(locale, message) + "\n";
                }
            }
            throw new InvalidEntityException(errorMessage);
        }

        Cati entity = DTOHelper.fromDto(proxy, Cati.class, cati);
        Cati dto = DTOHelper.toDto(proxy.store(entity));

        return dto;
    }

    /**
     * Return a new cati created in database.
     *
     * @return the new cati
     */
    public Cati createCati() {
        Cati cati = new CatiImpl();
        Cati dto = DTOHelper.toDto(cati);

        if (log.isDebugEnabled()) {
            log.debug("Cati " + dto + " created");
        }

        return dto;
    }

    /**
     * Delete a cati
     */
    public void deleteCati(Cati cati) {
        proxy.delete(cati.getWikittyId());
    }

    /**
     * Get a cati from its id
     *
     * @param id the cati id
     * @return the cati
     */
    public Cati getCati(String id) {

        Cati cati = proxy.restore(Cati.class, id);

        return DTOHelper.toDto(cati);
    }

    /**
     * Get all the frequencies.
     *
     * @return a list containing all the frequencies.
     */
    public List<Frequency> getFrequencies() {

        //Create criteria : everything that got frequency extension
        Search search = Search.query().eq(Element.ELT_EXTENSION, Frequency.EXT_FREQUENCY);
        Criteria criteria = search.criteria();

        //get results
        PagedResult<Frequency> frequenciesResult = proxy.findAllByCriteria(Frequency.class, criteria);

        List<Frequency> frequencies = new ArrayList<Frequency>();

        //Fill in frequencies with dtos
        if (frequenciesResult != null) {
            log.debug("Frequencies size : " + frequenciesResult.size());
            for (Frequency frequency:frequenciesResult.getAll()){
                log.debug("adding a frequency to the list");
                frequencies.add(DTOHelper.toDto(frequency));
            }
        }

        log.debug("Frequencies size : " + frequencies.size());

        if (log.isDebugEnabled()) {
            log.debug("Frequencies list : " + frequencies);
        }
        return frequencies;
    }

    /**
     * Save a frequency in database.
     *
     * @param frequency the frequency to save
     * @return the saved frequency
     * @throws AlreadyExistingException if a frequency with the same name already
     *                                  exist in database
     */
    public Frequency saveFrequency(Locale locale, Frequency frequency)
            throws AlreadyExistingException, InvalidEntityException {

        if (!"".equals(frequency.getName()) && exist(frequency)) {
            if (log.isDebugEnabled()) {
                log.debug("Frequency " + frequency.getName() + " already exist");
            }

            throw new AlreadyExistingException("Frequency with name " +
                    frequency.getName() + " already exist in database");
        }

        NuitonValidator<Frequency> validator =
                NuitonValidatorFactory.newValidator(Frequency.class);

        NuitonValidatorResult result = validator.validate(frequency);

        checkResults(locale, result);

        Frequency returnFrequency = null;

        if (frequency != null ){
            Frequency toSaveFrequency = DTOHelper.fromDto(proxy, Frequency.class,frequency);

            Frequency storedFrequency = proxy.store(toSaveFrequency);

            if(log.isDebugEnabled()){
                log.debug("Saved frequency : " +
                        storedFrequency.getWikittyId()+ "-" +
                        storedFrequency.getName());
            }

            returnFrequency = DTOHelper.toDto(storedFrequency);
        }

        return returnFrequency;
    }

    /**
     * Return a new frequency created in database.
     *
     * @return the new frequency
     */
    public Frequency createFrequency() {
        Frequency frequency = new FrequencyImpl();
        Frequency frequency1 = DTOHelper.toDto(frequency);

        if (log.isDebugEnabled()){
            log.debug("Frequency " + frequency1 +" created");
        }

        return frequency1;
    }

    /**
     * Delete a frequency
     */
    public void deleteFrequency(Frequency frequency, boolean force)
            throws UsedException {

        int timeUsed = getTimeUsed(frequency);

        if ((force) || (timeUsed == 0)) {
            proxy.delete(frequency.getWikittyId());
        } else {
            throw new UsedException("Frequency used " + timeUsed + " times");
        }
    }

    protected int getTimeUsed(Frequency frequency) {

        //Create criteria : everything that got frequency extension
        Search search = Search.query().eq(Element.ELT_EXTENSION, AgentSkill.EXT_AGENTSKILL)
                .eq(AgentSkill.FQ_FIELD_AGENTSKILL_FREQUENCY, frequency.getWikittyId());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<AgentSkill> skillPagedResult = proxy.findAllByCriteria(AgentSkill.class, criteria);

        if (log.isDebugEnabled()){
            log.debug("Frequency " + frequency.getWikittyId() + " used " +
                    skillPagedResult.size() + " times");
        }

        return skillPagedResult.size();
    }

    /**
     * Get a frequency from its id
     *
     * @param id the frequency id
     * @return the frequency
     */
    public Frequency getFrequency(String id) {
        Frequency frequencyResult = proxy.restore(Frequency.class, id);
        return DTOHelper.toDto(frequencyResult);
    }

    /**
     * Get all the users.
     *
     * @return a list containing all the users.
     */
    public List<User> getUsers() {

        //Create criteria : everything that got frequency extension
        Search search = Search.query().eq(Element.ELT_EXTENSION, User.EXT_USER);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(User.FIELD_USER_FIRSTNAME);

        //get results
        PagedResult<User> userPagedResult = proxy.findAllByCriteria(User.class, criteria);

        List<User> users = new ArrayList<User>();

        if (userPagedResult != null) {
            for (User user:userPagedResult.getAll()){
                users.add(DTOHelper.toDto(user));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Users list : " + users);
        }
        return users;
    }

    /**
     * Save a user in database.
     *
     * @param user the user to save
     * @return the saved user
     * @throws AlreadyExistingException if a user with the same name already
     *                                  exist in database
     */
    public User saveUser(Locale locale, String context, User user)
            throws AlreadyExistingException, InvalidEntityException {

        /*if (exist(user)) {
            throw new AlreadyExistingException("User with login " +
                    user.getLogin() + " already exist in database");
        }*/

        NuitonValidator<User> validator =
                NuitonValidatorFactory.newValidator(User.class, context);

        NuitonValidatorResult result = validator.validate(user);

        checkResults(locale, result);

        User entity = DTOHelper.fromDto(proxy, User.class, user);
        User dto = DTOHelper.toDto(proxy.store(entity));

        if (log.isDebugEnabled()){
            log.debug("User : " + dto + " saved");
        }

        return dto;
    }

    /**
     * Return a new user created in database.
     *
     * @return the new user
     */
    public User createUser() {
        User user = new UserImpl();
        if(log.isDebugEnabled()){
            log.debug("User created : " + user);
        }

        return DTOHelper.toDto(user);
    }

    /**
     * Delete a user
     */
    public void deleteUser(User user) {
        proxy.delete(user.getWikittyId());
    }

    /**
     * Get a user from its id
     *
     * @param id the user id
     * @return the user
     */
    public User getUser(String id) {
        User user = proxy.restore(User.class, id);
        User dto = DTOHelper.toDto(user);

        if (log.isDebugEnabled()){
            log.debug("Restore User : " + dto);
        }

        return dto;
    }

    @Override
    public Agent getAgent(String id) {

        User user = proxy.restore(User.class, id);
        Agent agent = null;
        if (user != null){
            Collection<String> extensions = user.getExtensionNames();
            for (String extension: extensions){
                if (extension.equals(Agent.EXT_AGENT)){
                    Agent testAgent = proxy.restore(Agent.class, id);
                    if (testAgent.getAgent()){
                        agent = testAgent;
                    }
                }
            }
        }
        Agent dto = null;

        if (agent != null){
            dto = DTOHelper.toDto(agent);
        }

        if (log.isDebugEnabled()){
            log.debug("Restore agent : " + dto);
        }

        return dto;
    }

    @Override
    public EntitiesList getAgentAndDependencies(String id) {
        Agent agent = getAgent(id);

        List<Agent> agents = new ArrayList<Agent>();
        agents.add(agent);

        EntitiesList result = createAgentList(agents);

        return result;
    }

    @Override
    public Agent setAgent(User user) {

        User userEntity = DTOHelper.fromDto(proxy, User.class, user);

        Agent agent = proxy.cast(userEntity, Agent.class);
        agent.setAgent(true);
        if (log.isDebugEnabled()) {
            log.debug("Agent :" + agent);
        }
        return DTOHelper.toDto(agent);
    }

    @Override
    public void unsetAgent(String id) {

        Agent agent = proxy.restore(Agent.class, id);
        if (agent != null) {
            agent.setAgent(false);
        }

    }

    @Override
    public Agent saveAgent(Locale locale, String context, Agent agent)
            throws AlreadyExistingException, InvalidEntityException {

        /*if (exist(agent)) {
            log.debug("Agent already exist");
            throw new AlreadyExistingException("User with login " +
                    agent.getLogin() + " already exist in database");
        }*/

        NuitonValidator<Agent> validator =
                NuitonValidatorFactory.newValidator(Agent.class, context);

        NuitonValidatorResult result = validator.validate(agent);

        checkResults(locale, result);

        Agent entity = DTOHelper.fromDto(proxy, Agent.class, agent);

        log.debug("Agent to save : " + entity);

        Agent savedAgent = proxy.store(entity);

        WikittyGroup privateGroup = getPrivateGroup();

        if(savedAgent.getAgent()) {
            privateGroup.addMembers(savedAgent.getWikittyId());
        } else {
            privateGroup.removeMembers(savedAgent.getWikittyId());
        }

        proxy.store(privateGroup);

        Agent dto = DTOHelper.toDto(savedAgent);

        if (log.isDebugEnabled()){
            log.debug("Agent : " + dto + "saved");
        }

        return dto;
    }

    @Override
    public Skill saveSkill(Locale locale, Skill skill)
            throws AlreadyExistingException, InvalidEntityException {

        if (!"".equals(skill.getName()) && exist(skill)) {
            if(log.isDebugEnabled()){
                log.debug("Skill " + skill.getName() + " already exist");
            }

            throw new AlreadyExistingException("Skill named " +
                    skill.getName() + " already exist in database");
        }

        NuitonValidator<Skill> validator =
                NuitonValidatorFactory.newValidator(Skill.class);

        NuitonValidatorResult result = validator.validate(skill);

        checkResults(locale, result);

        Skill entity = DTOHelper.fromDto(proxy, Skill.class, skill);

        if (log.isDebugEnabled()){
            log.debug("Try to save skill : " + entity);
        }

        entity = proxy.store(entity);

        entity = proxy.restore(Skill.class, entity.getWikittyId());
        if (log.isDebugEnabled()){
            log.debug("Saved skill : " + entity);
        }

        Skill dto = DTOHelper.toDto(entity);

        return dto;
    }

    @Override
    public Skill createSkill(String parentId) {
        Skill skill = new SkillImpl();
        if ((parentId != null)&&(!"".equals(parentId))){
            skill.setParent(parentId);
        }
        Skill dto = DTOHelper.toDto(skill);

        if (log.isDebugEnabled()) {
            log.debug("Skill " + dto + " created");
        }

        return dto;
    }

    /**
     * Delete a skill present in database. Check if the skill is used before
     * deleting. If the skill is used, throw an exception.
     * @param skill the skill to delete
     */
    @Override
    public void deleteSkill(Skill skill) throws UsedException {

        //if not used, delete otherwise, throw an exception
        if (!isSkillUsed(skill)){
            System.out.println("Skill not used");
            List<Skill> children = getChildrenSkill(skill.getWikittyId());
            for (Skill child:children) {
                deleteSkill(child);
            }
            proxy.delete(skill.getWikittyId());
        } else {
            throw new UsedException("Skill still used by users");
        }
    }

    /**
     * Check if a skill is used by users in their skills description
     * @param skill the skill to check
     * @return true if it is used
     */
    protected boolean isSkillUsed(Skill skill) {

        boolean b = false;

        List<Skill> children = getChildrenSkill(skill.getWikittyId());


        for(Skill child:children) {
            b = b || isSkillUsed(child);
        }

        Search search = Search.query()
                .eq(Element.ELT_EXTENSION, AgentSkill.EXT_AGENTSKILL)
                .or()
                .eq(AgentSkill.FQ_FIELD_AGENTSKILL_DOMAINSKILL,
                        skill.getWikittyId())
                .eq(AgentSkill.FQ_FIELD_AGENTSKILL_TECHNICALSKILL,
                        skill.getWikittyId());

        Criteria criteria = search.criteria();

        PagedResult<Wikitty> agentSkillPagedResult = proxy.findAllByCriteria(criteria);

        int size = agentSkillPagedResult.size();

        if(log.isDebugEnabled()){
            log.debug("Number of time the Skill " + skill.getName() +
                    " is used : " + size);
        }
         b = b || size != 0;

        return b;
    }

    @Override
    public Skill getSkill(String id) {
        Skill skill = proxy.restore(Skill.class, id);
        log.debug("Restore skill : " +skill);
        Skill dto = null;
        if (skill != null){
            dto = DTOHelper.toDto(skill);
        }
        return dto;
    }

    protected Boolean exist(Skill skill){

        Search search;

        String name = skill.getName();

        String parent = skill.getParent();

        if (parent != null ){
            search = Search.query().eq(Element.ELT_EXTENSION, Skill.EXT_SKILL)
                    .eq(Skill.FQ_FIELD_WIKITTYTREENODE_NAME, name)
                    .eq(Skill.FQ_FIELD_WIKITTYTREENODE_PARENT, parent);
        } else {
            search = Search.query().eq(Element.ELT_EXTENSION, Skill.EXT_SKILL)
                    .eq(Skill.FQ_FIELD_WIKITTYTREENODE_NAME, name)
                    .isNull(Skill.FQ_FIELD_WIKITTYTREENODE_PARENT);
        }

        Criteria criteria = search.criteria();

        //get results
        PagedResult<Skill> skillPagedResult = proxy.findAllByCriteria(Skill.class, criteria);


        //Remove the skill from the results
        List<Skill> skillList = new ArrayList<Skill>(skillPagedResult.getAll());
        skillList.remove(skill);

        return (skillList.size()>0);
    }

    /**
     * Check if a user is already present in database
     *
     * @param user the user to check
     * @return true is exist, false otherwise
     */
    protected boolean exist(User user) {

        Search search = Search.query().eq(Element.ELT_EXTENSION, User.EXT_USER)
                .eq(Element.ELT_EXTENSION, User.EXT_WIKITTYUSER)
                .eq(User.FQ_FIELD_WIKITTYUSER_LOGIN, user.getLogin());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<User> userPagedResult = proxy.findAllByCriteria(User.class, criteria);

        return (userPagedResult.size() > 0);
    }

    protected Boolean exist(Frequency frequency) {
        //Create criteria
        Search search = Search.query().eq(Element.ELT_EXTENSION, Frequency.EXT_FREQUENCY)
                .eq(Frequency.FQ_FIELD_FREQUENCY_NAME, frequency.getName());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<Frequency> frequencyPagedResult = proxy.findAllByCriteria(Frequency.class, criteria);

        //Remove the frequency from the results
        List<Frequency> frequencies = new ArrayList<Frequency>(frequencyPagedResult.getAll());
        frequencies.remove(frequency);

        return (frequencies.size() > 0);
    }

    /**
     * Check if a cati is already present in database
     *
     * @param cati the cati to check
     * @return true is exist, false otherwise
     */
    protected boolean exist(Cati cati) {
        //Create criteria
        Search search = Search.query().eq(Element.ELT_EXTENSION, Cati.EXT_CATI)
                .eq(Cati.FQ_FIELD_CATI_NAME, cati.getName());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<Cati> catiPagedResult = proxy.findAllByCriteria(Cati.class, criteria);

        return (catiPagedResult.size() > 0);
    }

    public Boolean login(String login, String password){
        try {
            proxy.login(login, password);
        } catch (Exception eee) {
            return false;
        }
        return true;
    }

    public void logout(){
        proxy.logout();
        if (log.isDebugEnabled()){
            log.debug("Logout");
        }
    }

    public List<Skill> getChildrenSkill(String id) {

        if (log.isDebugEnabled()){
            log.debug("Get children for skill : " + id);
        }

        //Create criteria : everything that got frequency extension
        Search search;
        if ((id != null)&& !"".equals(id)){
            System.out.println("Search for not null parent");
            search = Search.query()
                .eq(Skill.FQ_FIELD_WIKITTYTREENODE_PARENT,id);
        } else {
            System.out.println("Search for null parent");
            search = Search.query()
                    .isNull(Skill.FQ_FIELD_WIKITTYTREENODE_PARENT);
        }

        Criteria criteria = search.criteria();
        criteria.addSortAscending(Skill.FQ_FIELD_WIKITTYTREENODE_NAME);

        //get results
        PagedResult<Skill> skillPagedResult = proxy.findAllByCriteria(Skill.class, criteria);

        List<Skill> skills = new ArrayList<Skill>();

        if (skillPagedResult != null) {
            for (Skill skill:skillPagedResult.getAll()){
                skills.add(DTOHelper.toDto(skill));
                log.debug("Adding skill to list : " + skill);
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Skills list : " + skills);
        }
        return skills;
    }

    @Override
    public Department saveDepartment(Locale locale, Department department)
            throws AlreadyExistingException, InvalidEntityException {

        /*if (exist(department)) {
            throw new AlreadyExistingException("Department named " +
                    department.getName() + " already exist in database");
        }*/

        NuitonValidator<Department> validator =
                NuitonValidatorFactory.newValidator(Department.class);

        NuitonValidatorResult result = validator.validate(department);

        checkResults(locale, result);

        Department entity = DTOHelper.fromDto(proxy, Department.class, department);

        if (log.isDebugEnabled()) {
            log.debug("Try to save department : " + entity);
        }

        Department dto = DTOHelper.toDto(proxy.store(entity));

        return dto;
    }

    @Override
    public Department saveDepartment(Locale locale,
                                     Department department,
                                     List<String> units)
            throws AlreadyExistingException, InvalidEntityException {

        /*if (exist(department)) {
            throw new AlreadyExistingException("Department named " +
                    department.getName() + " already exist in database");
        }*/

        NuitonValidator<Department> validator =
                NuitonValidatorFactory.newValidator(Department.class);

        NuitonValidatorResult result = validator.validate(department);

        checkResults(locale, result);

        Department entity = DTOHelper.fromDto(proxy, Department.class, department);


        //get the actual links
        List<UnitDeptLink> links = getDepartmentLinks(entity);

        //clear the links
        entity.clearUnitDeptLink();

        //iterate on the departments and the actual links.
        for (String unit : units) {
            Boolean gotLink = false; //if already present link found
            for (Iterator<UnitDeptLink> iterator = links.iterator(); iterator.hasNext();) {
                UnitDeptLink link = iterator.next();
                if (unit.equals(link.getUnit()) && !gotLink) {
                    //unit found on links, remove it from the list of links
                    //stop looping when unit found once, this will remove
                    //useless extra links if found
                    //add the link to the department
                    entity.addUnitDeptLink(link.getWikittyId());
                    gotLink = true;
                    iterator.remove();
                    log.debug("found and add link : " + link);
                }
            }

            //if unit not already present, create a new link, store it,
            //add it to the department
            if (!gotLink) {
                UnitDeptLink newLink = new UnitDeptLinkImpl();
                newLink.setDepartment(department.getWikittyId());
                newLink.setUnit(unit);
                if (log.isDebugEnabled()) {
                    log.debug("Store link : " + newLink);
                }
                newLink = proxy.store(newLink);
                log.debug("add link : " + newLink);
                entity.addUnitDeptLink(newLink.getWikittyId());
            }
        }

        //Remove the links that did not get updated, they are not present anymore
        //This will remove double links and old links
        for (UnitDeptLink link : links) {
            log.debug("Remove link : " + link);
            proxy.delete(link.getWikittyId());
        }

        //restore department with all its links and return it
        Department savedDepartment = proxy.store(entity);

        if (log.isDebugEnabled()) {
            log.debug("Stored department : " + savedDepartment);
        }

        return DTOHelper.toDto(savedDepartment);
    }

    public List<UnitDeptLink> getDepartmentLinks(Department entity) {

        //Create criteria : everything that got the unitDeptLink extension with
        //the department
        Search search = Search.query().eq(Element.ELT_EXTENSION, UnitDeptLink.EXT_UNITDEPTLINK)
                .eq(UnitDeptLink.FQ_FIELD_UNITDEPTLINK_DEPARTMENT, entity.getWikittyId());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<UnitDeptLink> unitDeptLinkPagedResult = proxy.findAllByCriteria(UnitDeptLink.class, criteria);

        List<UnitDeptLink> unitDeptLinks = new ArrayList<UnitDeptLink>();

        if (unitDeptLinkPagedResult != null) {
            for (UnitDeptLink link : unitDeptLinkPagedResult.getAll()) {
                unitDeptLinks.add(DTOHelper.toDto(link));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Links list : " + unitDeptLinks);
        }
        return unitDeptLinks;
    }

    @Override
    public Department createDepartment() {
        Department department = new DepartmentImpl();
        Department dto = DTOHelper.toDto(department);

        if (log.isDebugEnabled()) {
            log.debug("Skill " + dto + " created");
        }

        return dto;
    }

    @Override
    public void deleteDepartment(Department department) {
        proxy.delete(department.getWikittyId());
    }

    @Override
    public Department getDepartment(String id) {
        Department department = proxy.restore(Department.class, id);
        return DTOHelper.toDto(department);
    }

    /**
     * Get all the departments.
     *
     * @return a list containing all the departments.
     */
    @Override
    public List<Department> getDepartments() {

        //Create criteria : everything that got the department extension
        Search search = Search.query().eq(Element.ELT_EXTENSION, Department.EXT_DEPARTMENT);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(Department.FQ_FIELD_DEPARTMENT_NAME);

        //get results
        PagedResult<Department> departmentPagedResult = proxy.findAllByCriteria(Department.class, criteria);

        List<Department> departments = new ArrayList<Department>();

        if (departmentPagedResult != null) {
            for (Department department : departmentPagedResult.getAll()) {
                departments.add(DTOHelper.toDto(department));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Departments list : " + departments);
        }
        return departments;
    }

    @Override
    public List<Department> getDepartments(String id) {

        //Create criteria : everything that got the unitdptlink extension and the unit id
        Search search = Search.query().eq(Element.ELT_EXTENSION, UnitDeptLink.EXT_UNITDEPTLINK)
                .eq(UnitDeptLink.FQ_FIELD_UNITDEPTLINK_UNIT, id);
        Criteria criteria = search.criteria();

        //get results
        PagedResult<UnitDeptLink> unitDeptLinkPagedResult =
                proxy.findAllByCriteria(UnitDeptLink.class, criteria);

        List<Department> departments = new ArrayList<Department>();

        //for each link get the department and add it to the list
        if (unitDeptLinkPagedResult != null) {
            for (UnitDeptLink unitDeptLink : unitDeptLinkPagedResult.getAll()) {

                Department department = proxy.restore(Department.class,
                        unitDeptLink.getDepartment());

                departments.add(DTOHelper.toDto(department));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Departments list : " + departments);
        }
        
        return departments;
    }

    protected Boolean exist(Department department) {
        //Create criteria
        Search search = Search.query().eq(Element.ELT_EXTENSION, Department.EXT_DEPARTMENT)
                .eq(Department.FQ_FIELD_DEPARTMENT_NAME, department.getName());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<Department> departmentPagedResult = proxy.findAllByCriteria(Department.class, criteria);

        return (departmentPagedResult.size() > 0);
    }

    @Override
    public Unit saveUnit(Locale locale, Unit unit)
            throws AlreadyExistingException, InvalidEntityException {

        /*if (exist(unit)) {
            throw new AlreadyExistingException("Unit named " +
                    unit.getName() + " already exist in database");
        }*/

        NuitonValidator<Unit> validator =
                NuitonValidatorFactory.newValidator(Unit.class);

        NuitonValidatorResult result = validator.validate(unit);

        checkResults(locale, result);

        Unit entity = DTOHelper.fromDto(proxy, Unit.class, unit);

        if (log.isDebugEnabled()) {
            log.debug("Try to save unit : " + entity);
        }

        Unit dto = DTOHelper.toDto(proxy.store(entity));

        return dto;
    }

    @Override
    public Unit saveUnit(Locale locale, Unit unit, List<String> departments)
            throws AlreadyExistingException, InvalidEntityException {
        /*if (exist(unit)) {
            throw new AlreadyExistingException("Unit named " +
                    unit.getName() + " already exist in database");
        }*/


        NuitonValidator<Unit> validator =
                NuitonValidatorFactory.newValidator(Unit.class);

        NuitonValidatorResult result = validator.validate(unit);

        checkResults(locale, result);

        Unit entity = DTOHelper.fromDto(proxy, Unit.class, unit);

        //get the actual links
        List<UnitDeptLink> links = getUnitLinks(entity);

        //clear the links
        entity.clearUnitDeptLink();

        //iterate on the departments and the actual links.
        for(String department:departments){
            Boolean gotLink = false; //if already present link found
            for (Iterator<UnitDeptLink> iterator = links.iterator(); iterator.hasNext();){
                UnitDeptLink link = iterator.next();
                if (department.equals(link.getDepartment()) && !gotLink){
                    //department found on links, remove it from the list of links
                    //stop looping when department found once, this will remove
                    //useless extra links if found
                    //add the link to the unit
                    entity.addUnitDeptLink(link.getWikittyId());
                    gotLink = true;
                    iterator.remove();
                    log.debug("found and add link : " + link);
                }
            }

            //if department not already present, create a new link, store it,
            //add it to the unit
            if (!gotLink){
                UnitDeptLink newLink = new UnitDeptLinkImpl();
                newLink.setDepartment(department);
                newLink.setUnit(unit.getWikittyId());
                if (log.isDebugEnabled()){
                    log.debug("Store link : " + newLink);
                }
                newLink = proxy.store(newLink);
                log.debug("add link : " + newLink);
                entity.addUnitDeptLink(newLink.getWikittyId());
            }
        }

        //Remove the links that did not get updated, they are not present anymore
        //This will remove double links and old links
        for(UnitDeptLink link:links){
            log.debug("Remove link : " + link);
            proxy.delete(link.getWikittyId());
        }

        //restore unit with all its links and return it
        Unit savedUnit = proxy.store(entity);

        if (log.isDebugEnabled()){
            log.debug("Stored unit : " + savedUnit);
        }

        return DTOHelper.toDto(savedUnit);
    }

    public List<UnitDeptLink> getUnitLinks(Unit entity) {

        //Create criteria : everything that got the unitDeptLink extension with
        //the unit
        Search search = Search.query().eq(Element.ELT_EXTENSION, UnitDeptLink.EXT_UNITDEPTLINK)
                .eq(UnitDeptLink.FQ_FIELD_UNITDEPTLINK_UNIT, entity.getWikittyId());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<UnitDeptLink> unitDeptLinkPagedResult = proxy.findAllByCriteria(UnitDeptLink.class, criteria);

        List<UnitDeptLink> unitDeptLinks = new ArrayList<UnitDeptLink>();

        if (unitDeptLinkPagedResult != null) {
            for (UnitDeptLink link : unitDeptLinkPagedResult.getAll()) {
                unitDeptLinks.add(DTOHelper.toDto(link));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Links list : " + unitDeptLinks);
        }
        return unitDeptLinks;
    }

    @Override
    public Unit createUnit() {
        Unit unit = new UnitImpl();
        Unit dto = DTOHelper.toDto(unit);

        if (log.isDebugEnabled()) {
            log.debug("Unit " + dto + " created");
        }

        return dto;
    }

    @Override
    public void deleteUnit(Unit unit) {
        proxy.delete(unit.getWikittyId());
    }

    @Override
    public Unit getUnit(String id) {
        Unit unit = proxy.restore(Unit.class, id);
        return DTOHelper.toDto(unit);
    }

    @Override
    public List<Unit> getUnits(String id) {

        //Create criteria : everything that got the unitdptlink extension and the department id
        Search search = Search.query().eq(Element.ELT_EXTENSION, UnitDeptLink.EXT_UNITDEPTLINK)
                .eq(UnitDeptLink.FQ_FIELD_UNITDEPTLINK_DEPARTMENT, id);
        Criteria criteria = search.criteria();

        //get results
        PagedResult<UnitDeptLink> unitDeptLinkPagedResult =
                proxy.findAllByCriteria(UnitDeptLink.class, criteria);

        List<Unit> units = new ArrayList<Unit>();

        //for each link get the department and add it to the list
        if (unitDeptLinkPagedResult != null) {
            for (UnitDeptLink unitDeptLink : unitDeptLinkPagedResult.getAll()) {

                Unit department = proxy.restore(Unit.class,
                        unitDeptLink.getUnit());

                units.add(DTOHelper.toDto(department));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Units list : " + units);
        }

        return units;
    }

    /**
     * Get all the departments.
     *
     * @return a list containing all the departments.
     */
    @Override
    public List<Unit> getUnits() {

        //Create criteria : everything that got the cati extension
        Search search = Search.query().eq(Element.ELT_EXTENSION, Unit.EXT_UNIT);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(Unit.FQ_FIELD_UNIT_NAME);

        //get results
        PagedResult<Unit> unitPagedResult = proxy.findAllByCriteria(Unit.class, criteria);

        List<Unit> units = new ArrayList<Unit>();

        if (unitPagedResult != null) {
            for (Unit unit : unitPagedResult.getAll()) {
                units.add(DTOHelper.toDto(unit));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Units list : " + units);
        }
        return units;
    }

    @Override
    public void saveEntity(BusinessEntity entity) {
        proxy.store(entity);
    }

    @Override
    public void deleteEntity(BusinessEntity entity) {
        proxy.delete(entity.getWikittyId());
    }

    @Override
    public String getAdminEmail() {
        RefcompConfiguration config = RefcompConfiguration.getInstance();
        return RefcompConfigurationHelper.getAdminEmail(config);
    }

    @Override
    public AgentSkill createAgentSkill(String agentId) {
        //force the impl for the authorisation.
        AgentSkillImpl skill = new AgentSkillImpl();
        WikittyAuthorisation authorisation = new WikittyAuthorisationImpl(skill);
        authorisation.setOwner(agentId);

        skill = WikittyUtil.cast(authorisation, AgentSkillImpl.class);

        AgentSkill dto = DTOHelper.toDto(skill);

        if (log.isDebugEnabled()) {
            log.debug("AgentSkill " + dto + " created");
        }

        return dto;
    }

    @Override
    public void saveAgentSkill(Locale locale, AgentSkill skill)
            throws InvalidEntityException{


        NuitonValidator<AgentSkill> validator =
                NuitonValidatorFactory.newValidator(AgentSkill.class);

        NuitonValidatorResult result = validator.validate(skill);

        checkResults(locale, result);

        AgentSkill entity = DTOHelper.fromDto(proxy, AgentSkill.class, skill);
        proxy.store(entity);
    }

    protected WikittyGroup getPrivateGroup() {

        //Create criteria
        Search search = Search.query().eq(
                WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME, PRIVATE_GROUP);
        Criteria criteria = search.criteria();

        //get results
        WikittyGroup group = proxy.findByCriteria(WikittyGroup.class, criteria);

        return group;
    }

    protected Boolean exist(Unit unit) {
        //Create criteria
        Search search = Search.query().eq(Element.ELT_EXTENSION, Unit.EXT_UNIT)
                .eq(Unit.FQ_FIELD_UNIT_NAME, unit.getName());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<Unit> unitPagedResult = proxy.findAllByCriteria(Unit.class, criteria);

        return (unitPagedResult.size() > 0);
    }

    @Override
    public boolean isAdmin(){
        try {
            boolean isMember = proxy.isMember(WikittySecurityUtil.WIKITTY_APPADMIN_GROUP_NAME);

            if (log.isDebugEnabled()){
                log.info("User admin : " + isMember);
            }

            return isMember;
        } catch (SecurityException eee) {
            return false;
        }
    }

    @Override
    public boolean isAdmin(String id) {
        try {
            boolean result = false;

            //Find the group from its name
            Search search = Search.query()
                    .eq(WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME,
                            WikittySecurityUtil.WIKITTY_APPADMIN_GROUP_NAME)
                    .contains(WikittyGroup.FQ_FIELD_WIKITTYGROUP_MEMBERS,
                            id);
            Criteria criteria = search.criteria();

            Wikitty group = proxy.findByCriteria(criteria);

            if (group != null) {
                log.debug("Is Admin yeah !");
                result = true;
            }

            log.debug ("is admin ? : " + result);

            return result;
        } catch (SecurityException eee) {
            log.debug("error");
            return false;
        }
    }

    @Override
    public boolean isAgent() {
        try {
            WikittyUser user = proxy.getLoggedInUser();
            if (user != null) {
                Agent agent = proxy.restore(Agent.class, user.getWikittyId());
                if (agent == null) {
                    return false;
                } else {
                    return agent.getAgent();
                }
            } else {
                return false;
            }
        }  catch (SecurityException eee) {
            return false;
        }
    }

    @Override
    public Boolean isLoggedIn(){
        try {
            WikittyUser user = proxy.getLoggedInUser();
            if (user == null) {
                return false;
            }
        } catch (SecurityException eee) {
            return false;
        }
        log.info("User logged in");
        return true;
    }

    @Override
    public void adminCreationIfDoNotExist(){
        Search search = Search.query().
                eq(WikittyUser.FQ_FIELD_WIKITTYUSER_LOGIN, "admin");

        Criteria criteria = search.criteria();

        //get results
        PagedResult<WikittyUser> userPagedResult = proxy.findAllByCriteria(WikittyUser.class, criteria);

        if (userPagedResult.size() == 0){

            //Admin does not exist, create it
            WikittyUser admin = new WikittyUserImpl();
            admin.setLogin("admin");
            admin.setPassword("admin");
            admin = proxy.store(admin);

            WikittyGroup adminGroup = WikittySecurityUtil.createAppAdminGroup(admin);
            if (log.isDebugEnabled()){
                log.debug("Admin group : " + adminGroup);
            }
            proxy.store(adminGroup);
            log.info("Admin account created");
        }
    }

    @Override
    public void createPrivateGroupIfDoNotExist() {
        Search search = Search.query().
                eq(WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME, PRIVATE_GROUP);

        Criteria criteria = search.criteria();

        //get results
        PagedResult<WikittyGroup> groupPagedResult = proxy.findAllByCriteria(WikittyGroup.class, criteria);

        if (groupPagedResult.size() == 0) {

            WikittyGroup privateGroup = new WikittyGroupImpl();
            privateGroup.setName(PRIVATE_GROUP);
            proxy.store(privateGroup);
            log.info("Private group created");
        }
    }

    @Override
    public void setPrivateSkills(List<String> ids) {

        List<WikittyAuthorisation> skills =
                proxy.restore(WikittyAuthorisation.class, ids);
        WikittyGroup privateGroup = getPrivateGroup();
        Set<String> privateSet = new HashSet<String>();
        privateSet.add(privateGroup.getWikittyId());

        for(WikittyAuthorisation skill:skills) {
            log.debug("Set skill private");
            skill.setReader(privateSet);
        }
        proxy.store(skills);
    }

    @Override
    public void setPublicSkills(List<String> ids) {

        List<WikittyAuthorisation> skills =
                proxy.restore(WikittyAuthorisation.class, ids);

        //set reader to null so that everybody can read
        for (WikittyAuthorisation skill : skills) {
            log.debug("Set skill public");
            skill.setReader(null);
        }
        proxy.store(skills);
    }

    @Override
    public void clear() {
        proxy.clear();
    }

    @Override
    public EntitiesList getSkills(String agentId){

        WikittyGroup privateGroup = getPrivateGroup();
        WikittyUser user = getLoggedInUser();

        Search search = Search.query()
                .eq(AgentSkill.FQ_FIELD_AGENTSKILL_AGENT, agentId);

        Set<String> members = privateGroup.getMembers();

        if (members != null && members.contains(user.getWikittyId())){
            //do nothing, criteria is ok
        } else {
            search.neq(WikittyAuthorisation.FQ_FIELD_WIKITTYAUTHORISATION_READER, privateGroup.getWikittyId());
        }

        Criteria criteria = search.criteria();

        PagedResult<AgentSkill> skills = proxy.findAllByCriteria(AgentSkill.class, criteria);

        EntitiesList result = createAgentSkillList(skills.getAll());

        return result;
    }

    @Override
    public AgentSkill addSkill(String agentId) {

        //create the new Skill and store it
        AgentSkill newSkill = new AgentSkillImpl();

        newSkill.setAgent(agentId);

        AgentSkill dto = DTOHelper.toDto(newSkill);

        return dto;
    }

    @Override
    public void removeSkill(String skillId) {
        //delete skill from database
        proxy.delete(skillId);
    }

    @Override
    public WikittyUser getLoggedInUser() {
        return proxy.getLoggedInUser();
    }

    @Override
    public EntitiesList saveMySkills(Locale locale, List<AgentSkill> skills)
            throws InvalidEntityException{

        NuitonValidator<AgentSkill> validator =
                NuitonValidatorFactory.newValidator(AgentSkill.class);

        Map<String, String> messages = new HashMap<String, String>();

        //Validate each skill one by one and add messages to the map
        for (AgentSkill skill : skills) {
            NuitonValidatorResult result = validator.validate(skill);
            if (!result.isValid()) {
                if (log.isDebugEnabled()) {
                    log.debug("Entity not valid");
                }
                List<String> fields = result.getFieldsForError();
                for (String field : fields) {
                    List<String> errorMessages = result.getErrorMessages(field);
                    for (String message : errorMessages) {
                        messages.put(_(message), l_(locale, message));
                    }
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Entity valid, will be saved");
                }
            }
        }

        //Throw an exception if one skill is not valid
        if (!messages.isEmpty()) {
            String errorMessage = "";
            for (String message:messages.values()) {
                errorMessage += message + "\n";
            }
            throw new InvalidEntityException(errorMessage);
        }

        List<AgentSkill> skillsEntities = new ArrayList<AgentSkill>();

        String agentId = getLoggedInUser().getWikittyId();

        for (AgentSkill skill : skills) {
            AgentSkill entity = DTOHelper.fromDto(proxy, AgentSkill.class, skill);
            entity.setAgent(agentId);

            //check if got authorisation extension, if not add it
            if(!entity.getExtensionNames().contains(WikittyAuthorisation.EXT_WIKITTYAUTHORISATION)) {
                System.out.println("noAuthor, add it");
                WikittyAuthorisation authorisation = new WikittyAuthorisationImpl((AgentSkillImpl)entity);
                authorisation.setOwner(agentId);
            }

            skillsEntities.add(entity);
        }

        proxy.store(skillsEntities);

        return getSkills(agentId);
    }

    @Override
    public AgentSkill getAgentSkill(String skillId) {
        AgentSkill restoredSkill = proxy.restore(AgentSkill.class, skillId);
        AgentSkill dto = DTOHelper.toDto(restoredSkill);
        return dto;
    }

    @Override
    public EntitiesList searchUser(String search, String cati,
                                      String department, String unit,
                                      String frequency) {

        String[] queryTerms = search.split(" ");

        Search userQuery = Search.query();

        for (String term:queryTerms){
            userQuery.keyword(term);
        }

        if (cati != null) {
            userQuery.eq(Agent.FQ_FIELD_AGENT_CATI, cati);
        }

        if (unit != null) {
            userQuery.contains(Agent.FQ_FIELD_AGENT_UNIT, unit);
        }

        if (department != null) {
            userQuery.eq(Agent.FQ_FIELD_AGENT_DEPARTMENT, department);
        }

        Criteria userCriteria = userQuery.criteria();

        PagedResult<Agent> agentsResult = proxy.findAllByCriteria(Agent.class, userCriteria);

        List<Agent> agents = agentsResult.getAll();

        EntitiesList result = createAgentList(agents);

        return result;
    }

    @Override
    public EntitiesList searchSkill(String search, String cati,
                                      String department, String unit,
                                      String frequency) {

        // First get all the users that can be in the result set (filter on
        // cati/unit/department)
        Search userQuery = Search.query().eq(Element.ELT_EXTENSION,
                AgentSkill.EXT_AGENTSKILL);

        //Cati filter
        if (cati != null) {
            userQuery.eq(Agent.FQ_FIELD_AGENT_CATI, cati);
        }

        //Unit filter
        if (unit != null) {
            userQuery.contains(Agent.FQ_FIELD_AGENT_UNIT, unit);
        }

        //Department filter
        if (department != null) {
            userQuery.eq(Agent.FQ_FIELD_AGENT_DEPARTMENT, department);
        }

        //Get users
        Criteria userCriteria = userQuery.criteria();
        PagedResult<Wikitty> agents = proxy.findAllByCriteria(userCriteria);
        List<String> agentIds = new ArrayList<String>();
        for (Wikitty agent : agents.getAll()) {
            agentIds.add(agent.getId());
        }

        if(log.isDebugEnabled()) {
            log.debug("Agents found : " + agentIds);
        }

        //Second, get all the skills that match the query
        Search skillQuery = Search.query().eq(Element.ELT_EXTENSION,
                Skill.EXT_SKILL);

        if (search != null){
            String[] queryTerms = search.split(" ");

            for (String term : queryTerms) {
                skillQuery.keyword(term);
                log.debug("Term : " + term);
            }
        }


        Criteria skillCriteria = skillQuery.criteria();
        log.debug(skillCriteria);
        PagedResult<Wikitty> skills = proxy.findAllByCriteria(skillCriteria);
        List<String> skillIds = new ArrayList<String>();
        for (Wikitty skill : skills.getAll()) {
            skillIds.add(skill.getId());
        }

        if (log.isDebugEnabled()) {
            log.debug("Skills found : " + skillIds);
        }


        //Then get all the AgentSkills that match the request and filter
        Search agentSkillQuery = Search.query();

        if (frequency != null) {
            agentSkillQuery.eq(AgentSkill.FQ_FIELD_AGENTSKILL_FREQUENCY, frequency);
        }
        if (!agentIds.isEmpty()) {
            agentSkillQuery.contains(AgentSkill.FQ_FIELD_AGENTSKILL_AGENT, agentIds);
        }

        //Or between Domain skill and technical skill
        agentSkillQuery = agentSkillQuery.or();

        if (!skillIds.isEmpty()) {
            agentSkillQuery.contains(AgentSkill.FQ_FIELD_AGENTSKILL_DOMAINSKILL, skillIds);
            agentSkillQuery.contains(AgentSkill.FQ_FIELD_AGENTSKILL_TECHNICALSKILL, skillIds);
        }

        Criteria skillsCriteria = agentSkillQuery.criteria();
        log.debug(skillsCriteria);

        PagedResult<AgentSkill> skillsResult = proxy.findAllByCriteria(
                AgentSkill.class, skillsCriteria);

        List<AgentSkill> skillsList = skillsResult.getAll();

        if (log.isDebugEnabled()) {
            log.debug("AgentSkills found : " + skillsList);
        }

        EntitiesList result = createAgentSkillList(skillsList);

        return result;
    }

    @Override
    public void setAdmin(String agentId, boolean admin) {
        Search searchGroup = Search.query().eq(WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME,
                WikittySecurityUtil.WIKITTY_APPADMIN_GROUP_NAME);
        Criteria groupCriteria = searchGroup.criteria();
        WikittyGroup adminGroup = proxy.findByCriteria(WikittyGroup.class,
                groupCriteria);
        if (admin) {
            if (log.isInfoEnabled()) {
                log.info("Add agent " + agentId + " to admin group");
            }
            adminGroup.addMembers(agentId);
            proxy.store(adminGroup);
        } else {
            if (log.isInfoEnabled()) {
                log.info("Remove agent " + agentId + " from admin group");
            }
            adminGroup.removeMembers(agentId);
            proxy.store(adminGroup);
        }
    }

    protected EntitiesList createAgentSkillList(List<AgentSkill> skills) {
        EntitiesList result = new EntitiesList();

        //Run through the skills only once to transform to DTO and get dependencies
        for (AgentSkill skill : skills) {
            //transform skill to DTO
            result.addEntity(DTOHelper.toDto(skill));

            //Agent
            Agent agent = proxy.restore(Agent.class,
                    skill.getAgent());
            if (agent != null) {
                log.debug("Agent : " + agent);
                result.addDependency(agent.getWikittyId(),
                        DTOHelper.toDto(agent));
            }

            //Domain skill
            Skill domainSkill = proxy.restore(Skill.class,
                    skill.getDomainSkill());
            if (domainSkill != null) {
                log.debug("Domain skill : " + domainSkill);
                result.addDependency(domainSkill.getWikittyId(),
                        DTOHelper.toDto(domainSkill));
            }

            //Technical skill
            Skill technicalSkill = proxy.restore(Skill.class,
                    skill.getTechnicalSkill());
            if (technicalSkill != null) {
                log.debug("Technical skill : " + technicalSkill);
                result.addDependency(technicalSkill.getWikittyId(),
                        DTOHelper.toDto(technicalSkill));
            }

            //Frequency
            Frequency frequency = proxy.restore(Frequency.class,
                    skill.getFrequency());
            if (frequency != null) {
                result.addDependency(frequency.getWikittyId(),
                        DTOHelper.toDto(frequency));
            }

            //Nature
            Nature nature = proxy.restore(Nature.class,
                    skill.getNature());
            if (nature != null) {
                result.addDependency(nature.getWikittyId(),
                        DTOHelper.toDto(nature));
            }

            //Rights
            WikittyAuthorisation authorisation = proxy.restore(
                    WikittyAuthorisation.class,
                    skill.getWikittyId());
            if (authorisation != null) {
                result.addDependency(authorisation.getWikittyId(),
                        DTOHelper.toDto(authorisation));
            }
        }

        return result;
    }

    protected EntitiesList createAgentList(List<Agent> agents) {
        EntitiesList result = new EntitiesList();

        //Run through the agents only once to transform to DTO and get dependencies
        for (Agent agent : agents) {
            //transform agent to DTO
            result.addEntity(DTOHelper.toDto(agent));

            //Department
            Department department = proxy.restore(Department.class,
                    agent.getDepartment());
            if (department != null) {
                result.addDependency(department.getWikittyId(),
                        DTOHelper.toDto(department));
            }

            //Cati
            Cati cati= proxy.restore(Cati.class,
                    agent.getCati());
            if (cati != null) {
                result.addDependency(cati.getWikittyId(),
                        DTOHelper.toDto(cati));
            }

            //Units
            List<String> unitsIds = new ArrayList<String>(agent.getUnit());

            List<Unit> units = proxy.restore(Unit.class, unitsIds);
            for (Unit unit:units) {
                result.addDependency(unit.getWikittyId(),
                        DTOHelper.toDto(unit));
            }
        }

        return result;
    }

    @Override
    public void importCSV(Reader reader) {

        //import
        WikittyImportExportService ieService =
                ServiceFactory.getImportExportService(proxy);
        ieService.syncImport(WikittyImportExportService.FORMAT.CSV, reader);
    }

    @Override
    public void exportSkillsReferentiel(Writer writer) {
        try {
            //Get the entities to export
            Search search = Search.query()
                    .eq(Element.ELT_EXTENSION, Skill.EXT_SKILL);
            Criteria criteria = search.criteria();

            //export
            WikittyImportExportService ieService =
                    ServiceFactory.getImportExportService(proxy);
            ieService.syncExportAllByCriteria(
                    WikittyImportExportService.FORMAT.CSV, criteria, writer);

        } catch (Exception eee) {
            log.debug("Cannot export : " + eee);
        }
    }

    @Override
    public void exportEntitiesReferentiel(Writer writer) {
        try {
            //Get the entities to export
            Search search = Search.query().or()
                    .eq(Element.ELT_EXTENSION, Cati.EXT_CATI)
                    .eq(Element.ELT_EXTENSION, Department.EXT_DEPARTMENT)
                    .eq(Element.ELT_EXTENSION, Unit.EXT_UNIT)
                    .eq(Element.ELT_EXTENSION, UnitDeptLink.EXT_UNITDEPTLINK);
            Criteria criteria = search.criteria();

            //export
            WikittyImportExportService ieService =
                    ServiceFactory.getImportExportService(proxy);
            ieService.syncExportAllByCriteria(
                    WikittyImportExportService.FORMAT.CSV, criteria, writer);

        } catch (Exception eee) {
            log.debug("Cannot export : " + eee);
        }
    }

    @Override
    public void exportUserBase(Writer writer) {
        try {
            //Get the entities to export
            Search searchUsers = Search.query().or()
                    .eq(Element.ELT_EXTENSION, WikittyUser.EXT_WIKITTYUSER)
                    .eq(Element.ELT_EXTENSION, AgentSkill.EXT_AGENTSKILL);
            Criteria usersCriteria = searchUsers.criteria();

            //export
            WikittyImportExportService ieService = ServiceFactory.getImportExportService(proxy);
            ieService.syncExportAllByCriteria(WikittyImportExportService.FORMAT.CSV, usersCriteria, writer);

        } catch (Exception eee) {
            log.debug("Cannot export : " + eee);
        }
    }

    /**
     * Get the path for a Skill
     * @param id the skill id
     */
    @Override
    public String getPath(String id) {
        Skill skill = proxy.restore(Skill.class, id);

        System.out.println("Get path for : " + skill.getName());

        String parentID = skill.getParent();

        String path = "";

        if (parentID != null && !parentID.isEmpty()) {
            path = getPath(parentID);
        }

        path += "/" + skill.getName();

        return path;
    }

    /**
     * check the validation results and throw exception with appropriate
     * error messages.
     * @param locale the user locale for error messages
     * @param result the validation result
     * @throws InvalidEntityException if the validation gets errors
     */
    protected void checkResults(Locale locale, NuitonValidatorResult result)
            throws InvalidEntityException{
        if (!result.isValid()) {
            String errorMessage = "";
            List<String> fields = result.getFieldsForError();
            for (String field : fields) {
                List<String> messages = result.getErrorMessages(field);
                for (String message : messages) {
                    errorMessage += l_(locale, message) + "\n";
                }
            }
            throw new InvalidEntityException(errorMessage);
        } else {
            if (log.isDebugEnabled()){
                log.debug("Entity valid");
            }
        }
    }

    /**
     * Move a skill in the tree skill
     * @return true if move is ok, false if cannot move (moving inside itself)
     */
    @Override
    public boolean moveSkill(String destination, String node) {
        if (isChildren(node, destination, true)){
            return false;
        } else {
            Skill skill = proxy.restore (Skill.class, node);
            skill.setParent(destination);
            proxy.store(skill);
            return true;
        }
    }

    public boolean sendPassword(Locale locale, String userId) {

        User user = proxy.restore(User.class, userId);

        if (user != null) {

            Properties properties = config.getFlatOptions();

            Session session = Session.getDefaultInstance(properties, null);
            MimeMessage message = new MimeMessage(session);
            try {
                message.addRecipient(
                        Message.RecipientType.TO,
                        new InternetAddress(user.getEmail()));

                message.setSubject(l_(locale, "refcomp.admin.email.password.subject"));

                //Create message here
                String body = String.format(
                        l_(locale, "refcomp.admin.email.password.body"),
                        user.getFirstName(),
                        user.getLastName(),
                        user.getLogin(),
                        user.getPassword());

                message.setText(body);

                Transport.send(message);
            } catch (MessagingException ex) {
                log.debug("Error sending email", ex);
                return false;
            }
            return true;
        } else {
            log.debug("Cannot send email to a non existing user");
            return false;
        }
    }

    protected boolean isChildren(String parent, String children, boolean recursive) {
        WikittyTreeNode childrenNode = proxy.restore(WikittyTreeNode.class, children);
        String parentId = childrenNode.getParent();

        if (parentId == null) {
            return false;
        }

        if (parentId.equals(parent)) {
            return true;
        } else {
            if (!recursive) {
                return false;
            } else {
                return isChildren(parent, parentId, true);
            }
        }
    }
}
