
package fr.ifremer.tutti.ui.swing.action;

/*
 * #%L
 * Tutti :: UI
 * $Id: ComputeBatchWeightsAction.java 1566 2014-02-04 08:31:02Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.4.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/action/ComputeBatchWeightsAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.BatchContainer;
import fr.ifremer.tutti.persistence.entities.data.BenthosBatch;
import fr.ifremer.tutti.persistence.entities.data.CatchBatch;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.MarineLitterBatch;
import fr.ifremer.tutti.persistence.entities.data.SampleCategory;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.service.catches.TuttiWeightComputingException;
import fr.ifremer.tutti.service.catches.WeightCleaningService;
import fr.ifremer.tutti.service.catches.WeightComputingService;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import java.util.List;

import static org.nuiton.i18n.I18n.t;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.0
 */
public class ComputeBatchWeightsAction extends AbstractTuttiAction<EditCatchesUIModel, EditCatchesUI, EditCatchesUIHandler> {

    protected final WeightCleaningService cleaningService;

    public ComputeBatchWeightsAction(EditCatchesUIHandler handler) {
        super(handler, true);
        cleaningService = getContext().getWeightCleaningService();
    }

    @Override
    public boolean prepareAction() throws Exception {
        boolean doAction = super.prepareAction();
        if (doAction) {

            // do a check of double weights
            FishingOperation fishingOperation = getModel().getFishingOperation();
            Multimap<String, String> errors = cleaningService.checkFishingOperation(fishingOperation.getId());

            if (errors.isEmpty()) {

                // no errors

                sendMessage(t("tutti.editCatchBatch.action.computeWeights.no.double.weight.detected"));
            } else {

                // show errors to user as warning

                String errorsStr = cleaningService.errorsToString(errors);
                String htmlMessage = t("tutti.editCatchBatch.action.computeWeights.double.weight.detected", errorsStr);

                JOptionPane.showMessageDialog(getContext().getActionUI(),
                                              htmlMessage,
                                              t("tutti.editCatchBatch.action.computeWeights.double.weight.detected.title"),
                                              JOptionPane.WARNING_MESSAGE);


            }
        }
        return doAction;
    }

    @Override
    public void doAction() throws Exception {
        PersistenceService persistenceService = getContext().getPersistenceService();

        WeightComputingService weightComputingService = getContext().getWeightComputingService();

        EditCatchesUIModel model = getModel();
        model.setLoadingData(true);

        // ---------
        // Compute species batches
        // ---------

        BatchContainer<SpeciesBatch> computedSpeciesBatches;
        Float totalSpeciesSortedWeight;
        String operationId = model.getFishingOperation().getId();
        try {
            computedSpeciesBatches =
                    weightComputingService.getComputedSpeciesBatches(operationId);
            totalSpeciesSortedWeight = computeSpeciesBatches(persistenceService, computedSpeciesBatches);

        } catch (TuttiWeightComputingException e) {
            getUI().getTabPane().setSelectedIndex(1);

            int index = e.getIndex();
            SpeciesBatchRowModel row = getUI().getSpeciesTabContent().getModel().getRows().get(index);
            int column;
            if (SpeciesBatch.PROPERTY_SAMPLE_CATEGORY_WEIGHT.equals(e.getProperty())) {
                column = getFinestCategoryColumn(row.getFinestCategory().getCategoryId());
            } else {
                column = 6;
            }
            TuttiUIUtil.doSelectCell(getUI().getSpeciesTabContent().getTable(), index, column);

            throw e;
        }

        // ---------
        // Compute benthos batches
        // ---------

        BatchContainer<BenthosBatch> computedBenthosBatches;
        Float totalBenthosSortedWeight;
        try {
            computedBenthosBatches =
                    weightComputingService.getComputedBenthosBatches(operationId);
            totalBenthosSortedWeight = computeBenthosBatches(persistenceService, computedBenthosBatches);

        } catch (TuttiWeightComputingException e) {
            getUI().getTabPane().setSelectedIndex(2);

            int index = e.getIndex();
            BenthosBatchRowModel row = getUI().getBenthosTabContent().getModel().getRows().get(index);
            int column;
            if (BenthosBatch.PROPERTY_SAMPLE_CATEGORY_WEIGHT.equals(e.getProperty())) {
                column = getFinestCategoryColumn(row.getFinestCategory().getCategoryId());
            } else {
                column = 6;
            }
            TuttiUIUtil.doSelectCell(getUI().getBenthosTabContent().getTable(), index, column);

            throw e;
        }

        // ---------
        // Compute marine litter batches
        // ---------

        BatchContainer<MarineLitterBatch> computedMarineLitterBatches;
        try {
            computedMarineLitterBatches =
                    weightComputingService.getComputedMarineLitterBatches(operationId,
                                                                          model.getMarineLitterTotalWeight());

        } catch (TuttiWeightComputingException e) {
            getUI().getTabPane().setSelectedIndex(3);
            TuttiUIUtil.doSelectCell(getUI().getMarineLitterTabContent().getTable(), e.getIndex(), 3);
            throw e;
        }

        // ---------
        // Check species rates
        // ---------

        Float rate = getConfig().getDifferenceRateBetweenSortedAndTotalWeights();
        if (model.getSpeciesTotalSortedWeight() != null
            && model.getSpeciesTotalSortedWeight() >= totalSpeciesSortedWeight
            && model.getSpeciesTotalSortedWeight() < (1 + rate / 100) * totalSpeciesSortedWeight) {

            // Si  le "Poids total VRAC" est saisi est que sa valeur
            // est supérieure de moins de x% (x en configuration)
            // du "Poids total Vrac trié", demander confirmation que
            // le "Poids total VRAC" est bien une valeur observée
            // sinon la remplacer par le "Poids total Vrac trié"

            getUI().getTabPane().setSelectedIndex(1);

            String htmlMessage = String.format(
                    AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                    t("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.message.species", rate),
                    t("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.help"));

            int answer = JOptionPane.showConfirmDialog(getContext().getActionUI(),
                                                       htmlMessage,
                                                       t("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.title"),
                                                       JOptionPane.YES_NO_OPTION,
                                                       JOptionPane.QUESTION_MESSAGE);

            if (answer == JOptionPane.NO_OPTION) {
                model.setSpeciesTotalSortedWeight(null);
                model.setSpeciesTotalSortedComputedWeight(totalSpeciesSortedWeight);
            }
        }

        // ---------
        // Check benthos rates
        // ---------

        if (model.getBenthosTotalSortedWeight() != null
            && model.getBenthosTotalSortedWeight() >= totalBenthosSortedWeight
            && model.getBenthosTotalSortedWeight() < (1 + rate / 100) * totalBenthosSortedWeight) {

            // Si  le "Poids total VRAC" est saisi est que sa valeur
            // est supérieure de moins de x% (x en configuration)
            // du "Poids total Vrac trié", demander confirmation que
            // le "Poids total VRAC" est bien une valeur observée
            // sinon la remplacer par le "Poids total Vrac trié"

            getUI().getTabPane().setSelectedIndex(2);

            String htmlMessage = String.format(
                    AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                    t("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.message.benthos", rate),
                    t("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.help"));

            int answer = JOptionPane.showConfirmDialog(getContext().getActionUI(),
                                                       htmlMessage,
                                                       t("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.title"),
                                                       JOptionPane.YES_NO_OPTION,
                                                       JOptionPane.QUESTION_MESSAGE);

            if (answer == JOptionPane.NO_OPTION) {
                model.setBenthosTotalSortedWeight(null);
                model.setBenthosTotalSortedComputedWeight(totalBenthosSortedWeight);
            }
        }

        boolean modified = model.isModify();
        CatchBatch catchBatch = model.toEntity();
        weightComputingService.computeCatchBatchWeights(catchBatch,
                                                        computedSpeciesBatches,
                                                        computedBenthosBatches,
                                                        computedMarineLitterBatches);
        model.fromEntity(catchBatch);
        model.setLoadingData(false);

        model.setModify(modified);
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        getUI().repaint();
        // see http://forge.codelutin.com/issues/3853
        SwingUtilities.invokeLater(
                new Runnable() {
                    @Override
                    public void run() {
                        getUI().getComputeSpeciesBatchButton().requestFocus();
                    }
                }
        );
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected Float computeSpeciesBatches(PersistenceService persistenceService, BatchContainer<SpeciesBatch> computedSpeciesBatches) {
        Float totalSortedWeight = 0f;

        if (computedSpeciesBatches != null) {
            List<SpeciesBatchRowModel> rows = Lists.newArrayList();
            List<SpeciesBatch> children = computedSpeciesBatches.getChildren();
            for (SpeciesBatch batch : children) {
                SpeciesBatchRowModel row = getUI().getSpeciesTabContent().getHandler().loadBatch(batch, null, rows);
                if (persistenceService.isVracBatch(row)) {
                    SampleCategory<?> sampleCategory = row.getFirstSampleCategory();
                    Float weight = TuttiEntities.getValueOrComputedValue(
                            sampleCategory.getCategoryWeight(),
                            sampleCategory.getComputedWeight());
                    totalSortedWeight += weight;
                }
            }
            getUI().getSpeciesTabContent().getModel().setRows(rows);
        }

        return totalSortedWeight;
    }

    protected Float computeBenthosBatches(PersistenceService persistenceService, BatchContainer<BenthosBatch> computedBenthosBatches) {
        Float totalSortedWeight = 0f;

        if (computedBenthosBatches != null) {
            List<BenthosBatchRowModel> rows = Lists.newArrayList();
            List<BenthosBatch> children = computedBenthosBatches.getChildren();
            for (BenthosBatch batch : children) {
                BenthosBatchRowModel row = getUI().getBenthosTabContent().getHandler().loadBatch(batch, null, rows);

                if (persistenceService.isVracBatch(row)) {
                    SampleCategory<?> sampleCategory = row.getFirstSampleCategory();
                    Float weight = TuttiEntities.getValueOrComputedValue(
                            sampleCategory.getCategoryWeight(),
                            sampleCategory.getComputedWeight());
                    totalSortedWeight += weight;
                }
            }
            getUI().getBenthosTabContent().getModel().setRows(rows);

        }
        return totalSortedWeight;
    }

    protected int getFinestCategoryColumn(Integer categoryId) {
        int column = 1;

        List<Integer> samplingOrderIds = getConfig().getSamplingOrderIds();

        int index = samplingOrderIds.indexOf(categoryId);
        if (index > -1) {
            column += index + 1;
        }

        return column;
    }
}
