package fr.ifremer.tutti.ui.swing.content.operation.catches.marinelitter;

/*
 * #%L
 * Tutti :: UI
 * $Id: MarineLitterBatchUIHandler.java 1247 2013-09-28 12:15:25Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.6/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/marinelitter/MarineLitterBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import fr.ifremer.tutti.PropagatePropertyChangeListener;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.BatchContainer;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.MarineLitterBatch;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.service.ValidationService;
import fr.ifremer.tutti.ui.swing.content.operation.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.marinelitter.create.CreateMarineLitterBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.marinelitter.create.CreateMarineLitterBatchUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.service.WeightUnit;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellEditor;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellEditor;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.validator.NuitonValidatorResult;

import javax.swing.JComponent;
import java.awt.Color;
import java.awt.Component;
import java.util.Collection;
import java.util.List;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class MarineLitterBatchUIHandler extends AbstractTuttiBatchTableUIHandler<MarineLitterBatchRowModel, MarineLitterBatchUIModel, MarineLitterBatchUI> {

    private static final Log log =
            LogFactory.getLog(MarineLitterBatchUIHandler.class);

    protected ValidationService validationService = getContext().getValidationService();

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected final WeightUnit weightUnit;

    public MarineLitterBatchUIHandler(TuttiUI<?, ?> parentUi, MarineLitterBatchUI ui) {
        super(parentUi, ui,
              MarineLitterBatchRowModel.PROPERTY_MARINE_LITTER_CATEGORY,
              MarineLitterBatchRowModel.PROPERTY_MARINE_LITTER_SIZE_CATEGORY,
              MarineLitterBatchRowModel.PROPERTY_WEIGHT,
              MarineLitterBatchRowModel.PROPERTY_NUMBER,
              MarineLitterBatchRowModel.PROPERTY_COMMENT);

        weightUnit = getConfig().getMarineLitterWeightUnit();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiBatchTableUIHandler methods                           --//
    //------------------------------------------------------------------------//

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        boolean empty = bean == null;

        MarineLitterBatchUIModel model = getModel();

        List<MarineLitterBatchRowModel> rows;

        if (empty) {
            rows = null;
        } else {

            if (log.isInfoEnabled()) {
                log.info("Get marineLitter batch for fishingOperation: " +
                         bean.getId());
            }
            rows = Lists.newArrayList();

            if (!TuttiEntities.isNew(bean)) {

                // get all marine litter root
                BatchContainer<MarineLitterBatch> batchContainer =
                        persistenceService.getRootMarineLitterBatch(bean.getId());

                for (MarineLitterBatch aBatch : batchContainer.getChildren()) {
                    MarineLitterBatchRowModel entry = loadBatch(aBatch);
                    rows.add(entry);
                }
            }
        }
        model.setRows(rows);
        recomputeBatchActionEnable();
    }

    protected MarineLitterBatchRowModel loadBatch(MarineLitterBatch aBatch) {

        MarineLitterBatchRowModel newRow =
                new MarineLitterBatchRowModel(weightUnit, aBatch);

        List<Attachment> attachments =
                persistenceService.getAllAttachments(newRow.getObjectType(),
                                                     newRow.getObjectId());

        newRow.addAllAttachment(attachments);
        return newRow;
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public MarineLitterBatchTableModel getTableModel() {
        return (MarineLitterBatchTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(MarineLitterBatchRowModel row) {
        MarineLitterBatch batch = row.toBean();
        NuitonValidatorResult validator = validationService.validateMarineLitterBatch(batch);
        boolean result = !validator.hasErrorMessagess();
        return result;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 MarineLitterBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {
        recomputeRowValidState(row);

        saveSelectedRowIfNeeded();

        // when row valid state has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<MarineLitterBatchRowModel> rowMonitor,
                                             MarineLitterBatchRowModel row) {

        if (row != null && row.isValid() && rowMonitor.wasModified()) {

            // monitored bean was modified, save it
            if (log.isInfoEnabled()) {
                log.info("Row " + row + " was modified, will save it");
            }

            rowMonitor.setBean(null);
            saveRow(row);
            rowMonitor.setBean(row);

            // clear modified flag on the monitor
            rowMonitor.clearModified();
        }
    }

    @Override
    protected void onModelRowsChanged(List<MarineLitterBatchRowModel> rows) {
        super.onModelRowsChanged(rows);

        for (MarineLitterBatchRowModel row : rows) {
            // update categoriesUsed
            addToMarineLitterCategoriesUsed(row);
        }
    }

    @Override
    protected void onRowValidStateChanged(int rowIndex,
                                          MarineLitterBatchRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);

        // when row valid state has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             MarineLitterBatchRowModel oldRow,
                                             int newRowIndex,
                                             MarineLitterBatchRowModel newRow) {
        super.onAfterSelectedRowChanged(oldRowIndex, oldRow, newRowIndex, newRow);

        // when selected row has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void addHighlighters(JXTable table) {
        super.addHighlighters(table);

        // paint in a special color for comment cell (with not null value)
        Color cellWithValueColor = getConfig().getColorCellWithValue();

        Highlighter commentHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        new HighlightPredicate.IdentifierHighlightPredicate(MarineLitterBatchTableModel.COMMENT),
                        // for not null value
                        new HighlightPredicate() {
                            @Override
                            public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                                String value = (String) adapter.getValue();
                                return StringUtils.isNotBlank(value);
                            }
                        }), cellWithValueColor);
        table.addHighlighter(commentHighlighter);

        // paint in a special color for attachment cell (when some attachments)

        Highlighter attachmentHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        new HighlightPredicate.IdentifierHighlightPredicate(MarineLitterBatchTableModel.ATTACHMENT),
                        // for not null value
                        new HighlightPredicate() {
                            @Override
                            public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                                Collection attachments = (Collection) adapter.getValue();
                                return CollectionUtils.isNotEmpty(attachments);
                            }
                        }
                ), cellWithValueColor);
        table.addHighlighter(attachmentHighlighter);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<MarineLitterBatchUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        if (log.isDebugEnabled()) {
            log.debug("beforeInit: " + ui);
        }

        EditCatchesUIModel catchesUIModel =
                ui.getContextValue(EditCatchesUIModel.class);

        MarineLitterBatchUIModel model = new MarineLitterBatchUIModel(catchesUIModel);
        ui.setContextValue(model);

        // propagate when value is changing
        PropagatePropertyChangeListener.listenAndPropagate(
                catchesUIModel,
                model,
                EditCatchesUIModel.PROPERTY_MARINE_LITTER_TOTAL_WEIGHT,
                EditCatchesUIModel.PROPERTY_MARINE_LITTER_TOTAL_WEIGHT);
    }

    @Override
    public void afterInitUI() {

        if (log.isDebugEnabled()) {
            log.debug("afterInit: " + ui);
        }

        initUI(ui);

        JXTable table = getTable();

        // create table column model
        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        {
            // MarineLitter Category column

            addColumnToModel(columnModel,
                             null,
                             newTableCellRender(CaracteristicQualitativeValue.class, null),
                             MarineLitterBatchTableModel.MACRO_WASTE_CATEGORY);
        }

        {
            // MarineLitter Size Category column

            addColumnToModel(columnModel,
                             null,
                             newTableCellRender(CaracteristicQualitativeValue.class, null),
                             MarineLitterBatchTableModel.MACRO_WASTE_SIZE_CATEGORY);
        }

        { // Number column

            addIntegerColumnToModel(columnModel,
                                    MarineLitterBatchTableModel.NUMBER,
                                    TuttiUI.INT_3_DIGITS_PATTERN);
        }

        { // Weight column

            addFloatColumnToModel(columnModel,
                                  MarineLitterBatchTableModel.WEIGHT,
                                  weightUnit);
        }

        { // Comment column

            addColumnToModel(columnModel,
                             CommentCellEditor.newEditor(ui),
                             CommentCellRenderer.newRender(),
                             MarineLitterBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellEditor.newEditor(ui),
                             AttachmentCellRenderer.newRender(getDecorator(Attachment.class, null)),
                             MarineLitterBatchTableModel.ATTACHMENT);
        }

        // create table model
        MarineLitterBatchTableModel tableModel =
                new MarineLitterBatchTableModel(weightUnit, columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initBatchTable(table, columnModel, tableModel);
        recomputeBatchActionEnable();
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getTable();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void createBatch() {

        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        CreateMarineLitterBatchUI createBatchEditor = parent.getMarineLitterTabCreateBatch();

        createBatchEditor.getHandler().openUI(getModel());
        parent.getHandler().setMarineLitterSelectedCard(EditCatchesUIHandler.CREATE_BATCH_CARD);
    }

    public void addBatch(CreateMarineLitterBatchUIModel model) {
        if (model.isValid()) {

            MarineLitterBatchTableModel tableModel = getTableModel();

            MarineLitterBatchRowModel newRow = tableModel.createNewRow();
            newRow.setMarineLitterCategory(model.getMarineLitterCategory());
            newRow.setMarineLitterSizeCategory(model.getMarineLitterSizeCategory());
            newRow.setNumber(model.getNumber());
            newRow.setWeight(model.getWeight());

            recomputeRowValidState(newRow);

            saveRow(newRow);

            tableModel.addNewRow(newRow);
            AbstractSelectTableAction.doSelectCell(getTable(), tableModel.getRowCount() - 1, 0);

            //update categories used
            addToMarineLitterCategoriesUsed(newRow);
        }

        recomputeBatchActionEnable();
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void recomputeBatchActionEnable() {

        int rowIndex = getTable().getSelectedRow();

        boolean enableRemove = false;

        if (rowIndex != -1) {

            // there is a selected row
            enableRemove = true;
        }
        MarineLitterBatchUIModel model = getModel();
        model.setRemoveBatchEnabled(enableRemove);
    }

    protected void saveRow(MarineLitterBatchRowModel row) {

        MarineLitterBatch entityToSave = row.toEntity();

        FishingOperation fishingOperation = getModel().getFishingOperation();
        entityToSave.setFishingOperation(fishingOperation);
        if (log.isInfoEnabled()) {
            log.info("Selected fishingOperation: " + fishingOperation.getId());
        }

        if (TuttiEntities.isNew(entityToSave)) {

            entityToSave = persistenceService.createMarineLitterBatch(entityToSave);
            row.setId(entityToSave.getId());
        } else {
            persistenceService.saveMarineLitterBatch(entityToSave);
        }

        getModel().fireBatchUpdated(row);
    }

    public void removeFromMarineLitterCategoriesUsed(MarineLitterBatchRowModel row) {
        Preconditions.checkNotNull(row);
        Preconditions.checkNotNull(row.getMarineLitterCategory());
        Preconditions.checkNotNull(row.getMarineLitterSizeCategory());
        if (log.isInfoEnabled()) {
            log.info("Remove from speciesUsed: " +
                     decorate(row.getMarineLitterSizeCategory()) +
                     " - " + decorate(row.getMarineLitterCategory()));
        }
        MarineLitterBatchUIModel model = getModel();
        model.getMarineLitterCategoriesUsed().remove(row.getMarineLitterSizeCategory(),
                                                     row.getMarineLitterCategory());

    }

    protected void addToMarineLitterCategoriesUsed(MarineLitterBatchRowModel row) {
        Preconditions.checkNotNull(row);
        Preconditions.checkNotNull(row.getMarineLitterCategory());
        Preconditions.checkNotNull(row.getMarineLitterSizeCategory());
        if (log.isDebugEnabled()) {
            log.debug("Add to marineLitterCategoriesUsed: " +
                      decorate(row.getMarineLitterSizeCategory()) +
                      " - " + decorate(row.getMarineLitterCategory()));
        }
        MarineLitterBatchUIModel model = getModel();
        model.getMarineLitterCategoriesUsed().put(row.getMarineLitterSizeCategory(),
                                                  row.getMarineLitterCategory());

    }
}