package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos;

/*
 * #%L
 * Tutti :: UI
 * $Id: BenthosBatchTableModel.java 1203 2013-09-23 09:39:50Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.5.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/benthos/BenthosBatchTableModel.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.data.SampleCategory;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SampleCategoryColumnIdentifier;
import fr.ifremer.tutti.ui.swing.util.WeightUnit;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableModel;
import fr.ifremer.tutti.ui.swing.util.table.ColumnIdentifier;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.table.TableColumnModelExt;

import javax.swing.table.TableColumn;
import java.util.Enumeration;
import java.util.Set;

import static org.nuiton.i18n.I18n.n_;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class BenthosBatchTableModel extends AbstractTuttiTableModel<BenthosBatchRowModel> {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(BenthosBatchTableModel.class);

    public static final ColumnIdentifier<BenthosBatchRowModel> SPECIES = ColumnIdentifier.newId(
            BenthosBatchRowModel.PROPERTY_SPECIES,
            n_("tutti.editBenthosBatch.table.header.species"),
            n_("tutti.editBenthosBatch.table.header.species.tip"));

    public static final ColumnIdentifier<BenthosBatchRowModel> WEIGHT = ColumnIdentifier.newId(
            BenthosBatchRowModel.PROPERTY_COMPUTED_WEIGHT,
            n_("tutti.editBenthosBatch.table.header.weight"),
            n_("tutti.editBenthosBatch.table.header.weight.tip"));

    public static final ColumnIdentifier<BenthosBatchRowModel> COMPUTED_NUMBER = ColumnIdentifier.newId(
            BenthosBatchRowModel.PROPERTY_COMPUTED_NUMBER,
            n_("tutti.editBenthosBatch.table.header.computedNumber"),
            n_("tutti.editBenthosBatch.table.header.computedNumber.tip"));

    public static final ColumnIdentifier<BenthosBatchRowModel> COMMENT = ColumnIdentifier.newId(
            BenthosBatchRowModel.PROPERTY_COMMENT,
            n_("tutti.editBenthosBatch.table.header.comment"),
            n_("tutti.editBenthosBatch.table.header.comment.tip"));

    public static final ColumnIdentifier<BenthosBatchRowModel> ATTACHMENT = ColumnIdentifier.newReadOnlyId(
            BenthosBatchRowModel.PROPERTY_ATTACHMENT,
            n_("tutti.editBenthosBatch.table.header.file"),
            n_("tutti.editBenthosBatch.table.header.file.tip"));

    public static final ColumnIdentifier<BenthosBatchRowModel> SPECIES_TO_CONFIRM = ColumnIdentifier.newId(
            BenthosBatchRowModel.PROPERTY_SPECIES_TO_CONFIRM,
            n_("tutti.editBenthosBatch.table.header.toConfirm"),
            n_("tutti.editBenthosBatch.table.header.toConfirm.tip"));

    /**
     * Columns for the frequency edition.
     *
     * @since 0.2
     */
    protected final Set<ColumnIdentifier<BenthosBatchRowModel>> frequencyCols;

    /**
     * Columns implies in the sample category definition.
     *
     * @since 0.2
     */
    protected final Set<SampleCategoryColumnIdentifier<BenthosBatchRowModel>> sampleCols;

    /**
     * Sample categories model.
     *
     * @since 2.4
     */
    protected final SampleCategoryModel sampleCategoryModel;

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected final WeightUnit weightUnit;

    public BenthosBatchTableModel(WeightUnit weightUnit,
                                  SampleCategoryModel sampleCategoryModel,
                                  TableColumnModelExt columnModel) {
        super(columnModel, false, false);
        this.weightUnit = weightUnit;
        this.sampleCategoryModel = sampleCategoryModel;
        setNoneEditableCols(SPECIES);

        frequencyCols = Sets.newHashSet();
        frequencyCols.add(COMPUTED_NUMBER);
        frequencyCols.add(WEIGHT);

        sampleCols = Sets.newHashSet();
        Enumeration<TableColumn> columns = columnModel.getColumns();
        while (columns.hasMoreElements()) {
            TableColumn tableColumn = columns.nextElement();
            Object identifier = tableColumn.getIdentifier();
            if (identifier instanceof SampleCategoryColumnIdentifier<?>) {
                SampleCategoryColumnIdentifier<BenthosBatchRowModel> categoryColumnIdentifier =
                        (SampleCategoryColumnIdentifier<BenthosBatchRowModel>) identifier;
                sampleCols.add(categoryColumnIdentifier);
            }
        }
    }

    @Override
    public BenthosBatchRowModel createNewRow() {
        BenthosBatchRowModel result =
                new BenthosBatchRowModel(weightUnit, sampleCategoryModel);

        // by default empty row is not valid
        result.setValid(false);
        return result;
    }

    @Override
    public void setValueAt(Object aValue,
                           int rowIndex,
                           int columnIndex,
                           ColumnIdentifier<BenthosBatchRowModel> propertyName,
                           BenthosBatchRowModel entry) {
        if (sampleCols.contains(propertyName)) {

            SampleCategoryColumnIdentifier<BenthosBatchRowModel> sampleCategoryColumnIdentifier = (SampleCategoryColumnIdentifier<BenthosBatchRowModel>) propertyName;
            sampleCategoryColumnIdentifier.setWeightValue(entry, aValue);

            // must find out first ancestor with this category
            Integer sampleCategoryId = sampleCategoryColumnIdentifier.getSampleCategoryId();
            if (log.isDebugEnabled()) {
                log.debug("Sample category: " + sampleCategoryId + " modified at row: " + rowIndex);
            }
            SampleCategory<?> sampleCategory = entry.getSampleCategoryById(sampleCategoryId);
            BenthosBatchRowModel firstAncestor = entry.getFirstAncestor(sampleCategory);

            int firstRowIndex = getRowIndex(firstAncestor);
            if (log.isDebugEnabled()) {
                log.debug("First ancestor row: " + firstRowIndex);
            }

            // must save this row now

            // get shell of the ancestor
            Set<BenthosBatchRowModel> shell = Sets.newHashSet();
            firstAncestor.collectShell(shell);

            for (BenthosBatchRowModel batchRowModel : shell) {
                int currentRowIndex = getRowIndex(batchRowModel);
                if (log.isDebugEnabled()) {
                    log.debug("Update shell row: " + currentRowIndex);
                }
                fireTableCellUpdated(currentRowIndex, columnIndex);
            }
        } else {
            super.setValueAt(aValue, rowIndex, columnIndex, propertyName, entry);
        }
    }

    @Override
    protected boolean isCellEditable(int rowIndex,
                                     int columnIndex,
                                     ColumnIdentifier<BenthosBatchRowModel> propertyName) {

        boolean result = super.isCellEditable(rowIndex,
                                              columnIndex,
                                              propertyName);
        if (result) {

            if (frequencyCols.contains(propertyName)) {

                // must have filled a species to edit this column
                BenthosBatchRowModel entry = getEntry(rowIndex);
                result = entry.isBatchLeaf();

            } else if (sampleCols.contains(propertyName)) {

                // can only edit if a category value is setted
                BenthosBatchRowModel entry = getEntry(rowIndex);

                // check from protocol what is possible ?
                Species species = entry.getSpecies();

                if (species == null) {

                    // no species, can not edit
                    result = false;
                } else {

                    // protocol authorize it

                    // final test: can edit only if sample category is setted
                    SampleCategory<?> value =
                            (SampleCategory<?>) propertyName.getValue(entry);

                    result = value.getCategoryValue() != null;

                }
            }
        }
        return result;
    }

    /**
     * Return the next editable row index for frequency from the given
     * {@code rowIndex}, or {@code null} if none found.
     *
     * @param rowIndex the starting index where to look
     * @return the next editable row index for frequency from the given {@code rowIndex}, or {@code null} if none found.
     * @since 2.5
     */
    public Integer getNextEditableFrequencyRow(Integer rowIndex) {
        Integer result = null;

        for (int i = rowIndex, max = getRowCount(); i < max; i++) {
            BenthosBatchRowModel entry = getEntry(i);
            if (entry.isBatchLeaf()) {
                result = i;
                break;
            }
        }
        return result;
    }
}
