
package fr.ifremer.tutti.ui.swing.content.operation.catches;

/*
 * #%L
 * Tutti :: UI
 * $Id: ComputeWeightsAction.java 739 2013-04-05 13:36:16Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/ComputeWeightsAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.TuttiBusinessException;
import fr.ifremer.tutti.persistence.entities.data.BatchContainer;
import fr.ifremer.tutti.persistence.entities.data.CatchBatch;
import fr.ifremer.tutti.persistence.entities.data.SortedUnsortedEnum;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.service.catches.TuttiWeightComputingService;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency.BenthosFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.action.AbstractTuttiAction;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import org.apache.commons.collections.CollectionUtils;

import javax.swing.JOptionPane;
import java.util.List;

import static org.nuiton.i18n.I18n._;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.0
 */
public class ComputeWeightsAction extends AbstractTuttiAction<EditCatchesUIModel, EditCatchesUI, EditCatchesUIHandler> {

    public ComputeWeightsAction(EditCatchesUIHandler handler) {
        super(handler, false);
    }

    @Override
    protected void doAction() throws Exception {
        TuttiWeightComputingService tuttiWeightComputingService = getContext().getTuttiWeightComputingService();

        EditCatchesUIModel model = getModel();

        BatchContainer<SpeciesBatch> computedSpeciesBatches =
                tuttiWeightComputingService.getComputedSpeciesBatches(model.getFishingOperation());
        Float totalSpeciesSortedWeight = computeSpeciesBatches(computedSpeciesBatches);

        BatchContainer<SpeciesBatch> computedBenthosBatches =
                tuttiWeightComputingService.getComputedBenthosBatches(model.getFishingOperation());
        Float totalBenthosSortedWeight = computeBenthosBatches(computedBenthosBatches);

        Float rate = getConfig().getDifferenceRateBetweenSortedAndTotalWeights();
        if (model.getSpeciesTotalSortedWeight() != null
                && model.getSpeciesTotalSortedWeight() >= totalSpeciesSortedWeight
                && model.getSpeciesTotalSortedWeight() < (1 + rate / 100) * totalSpeciesSortedWeight) {

            // Si  le "Poids total VRAC" est saisi est que sa valeur
            // est supérieure de moins de x% (x en configuration)
            // du "Poids total Vrac trié", demander confirmation que
            // le "Poids total VRAC" est bien une valeur observée
            // sinon la remplacer par le "Poids total Vrac trié"
            String htmlMessage = String.format(
                    AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                    _("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.message", rate),
                    _("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.help"));

            int answer = JOptionPane.showConfirmDialog(getContext().getActionUI(),
                    htmlMessage,
                    _("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.title"),
                    JOptionPane.YES_NO_OPTION,
                    JOptionPane.QUESTION_MESSAGE);

            if (answer == JOptionPane.NO_OPTION) {
                model.setSpeciesTotalSortedWeight(null);
                model.setSpeciesTotalSortedComputedWeight(totalSpeciesSortedWeight);
            }
        }
        if (model.getBenthosTotalSortedWeight() != null
                && model.getBenthosTotalSortedWeight() >= totalBenthosSortedWeight
                && model.getBenthosTotalSortedWeight() < (1 + rate / 100) * totalSpeciesSortedWeight) {

            // Si  le "Poids total VRAC" est saisi est que sa valeur
            // est supérieure de moins de x% (x en configuration)
            // du "Poids total Vrac trié", demander confirmation que
            // le "Poids total VRAC" est bien une valeur observée
            // sinon la remplacer par le "Poids total Vrac trié"
            String htmlMessage = String.format(
                    AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                    _("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.message", rate),
                    _("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.help"));

            int answer = JOptionPane.showConfirmDialog(getContext().getActionUI(),
                    htmlMessage,
                    _("tutti.editCatchBatch.action.computeWeights.replaceTotalSortedWeight.title"),
                    JOptionPane.YES_NO_OPTION,
                    JOptionPane.QUESTION_MESSAGE);

            if (answer == JOptionPane.NO_OPTION) {
                model.setBenthosTotalSortedWeight(null);
                model.setBenthosTotalSortedComputedWeight(totalBenthosSortedWeight);
            }
        }

        boolean modified = model.isModify();
        CatchBatch catchBatch = model.toBean();
        tuttiWeightComputingService.computeCatchBatchWeights(catchBatch,
                                                             computedSpeciesBatches,
                                                             computedBenthosBatches);
        model.setCatchBatch(catchBatch);
        model.fromBean(catchBatch);
        model.setModify(modified);
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        getUI().repaint();
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected Float computeSpeciesBatches(BatchContainer<SpeciesBatch> computedSpeciesBatches) {
        Float totalSortedWeight = 0f;

        List<SpeciesBatchRowModel> rows = Lists.newArrayList();
        List<SpeciesBatch> children = computedSpeciesBatches.getChildren();
        int rowIndex = 0;
        for (SpeciesBatch batch : children) {
            SpeciesBatchRowModel row = getUI().getSpeciesTabContent().getHandler().loadBatch(batch, null, rows);
            Float weight = row.getSortedUnsortedCategoryWeight();
            if (weight == null) {
                weight = row.getSortedUnsortedCategoryComputedWeight();
            }
            if (weight == null) {
                AbstractSelectTableAction.doSelectCell(getUI().getSpeciesTabContent().getTable(), rowIndex, 1);
                throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.noWeight"));

            } else if (SortedUnsortedEnum.SORTED.matchValue(
                    row.getSortedUnsortedCategoryValue())) {
                totalSortedWeight += weight;
            }
            rowIndex++;
            if (row.getChildBatch() != null) {
                rowIndex += row.getChildBatch().size();
            }
        }
        getUI().getSpeciesTabContent().getModel().setRows(rows);

        return totalSortedWeight;
    }

    protected Float computeBenthosBatches(BatchContainer<SpeciesBatch> computedBenthosBatches) {
        Float totalSortedWeight = 0f;

        List<BenthosBatchRowModel> rows = Lists.newArrayList();
        List<SpeciesBatch> children = computedBenthosBatches.getChildren();
        int rowIndex = 0;
        for (SpeciesBatch batch : children) {
            BenthosBatchRowModel row = getUI().getBenthosTabContent().getHandler().loadBatch(batch, null, rows);
            Float weight = row.getSortedUnsortedCategoryWeight();
            if (weight == null) {
                weight = row.getSortedUnsortedCategoryComputedWeight();
            }
            if (weight == null) {
                AbstractSelectTableAction.doSelectCell(getUI().getBenthosTabContent().getTable(), rowIndex, 1);
                throw new TuttiBusinessException(_("tutti.editCatchBatch.action.computeWeights.error.noWeight"));

            } else if (SortedUnsortedEnum.SORTED.matchValue(
                    row.getSortedUnsortedCategoryValue())) {
                totalSortedWeight += weight;
            }
            rowIndex++;
            if (row.getChildBatch() != null) {
                rowIndex += row.getChildBatch().size();
            }
        }
        getUI().getBenthosTabContent().getModel().setRows(rows);

        return totalSortedWeight;
    }
}
