/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.admin.helpers;

import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.ui.helpers.ThesaurusDataHelper;
import com.jurismarches.vradi.ui.admin.loadors.RootThesaurusTreeTableNodeLoadors;
import com.jurismarches.vradi.ui.admin.loadors.ThesaurusTreeTableNodeLoadors;
import com.jurismarches.vradi.ui.admin.models.ThesaurusTreeTableModel;
import com.jurismarches.vradi.ui.tree.VradiTreeTableNode;

import jaxx.runtime.swing.nav.treetable.NavTreeTableHelper;
import jaxx.runtime.swing.nav.treetable.NavTreeTableModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyServiceEvent;
import org.nuiton.wikitty.WikittyServiceListener;
import org.nuiton.wikitty.WikittyService.ServiceListenerType;

import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.tree.VradiDataProvider;

import java.util.*;

/**
 * Navigation tree helpers.
 *
 * @author sletellier
 * @see jaxx.runtime.swing.nav.tree.NavTreeHelper
 */
public class ThesaurusTreeTableHelper extends NavTreeTableHelper<VradiTreeTableNode> implements WikittyServiceListener {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(ThesaurusTreeTableHelper.class);

    protected RootThesaurus rootThesaurus;

    /**
     * Create helper with first nodes are the root thesaurus
     *
     */
    public ThesaurusTreeTableHelper() {
        this(new VradiDataProvider());
    }

    /**
     * Create helper with first nodes are the root thesaurus in parm (for move)
     *
     * @param rootThesaurus root thesaurus to display
     */
    public ThesaurusTreeTableHelper(RootThesaurus rootThesaurus) {
        this(new VradiDataProvider());
        this.rootThesaurus = rootThesaurus;
    }

    /**
     * Create helper with first nodes are the root thesaurus
     *
     * @param dataProvider vradi data provider
     */
    public ThesaurusTreeTableHelper(VradiDataProvider dataProvider) {
        super();
        init(dataProvider);
    }

    protected void init(VradiDataProvider dataProvider) {
        setDataProvider(dataProvider);

        // register each tree on wikitty service
        VradiService.getWikittyService().addWikittyServiceListener(this, ServiceListenerType.ALL);
    }

    /**
     * Try to select thesaurus pass in param
     *
     * @param thesaurus list of thesaurus to select
     */
    public void tryToSelect(List<Thesaurus> thesaurus) {
        tryToSelect(extractIds(thesaurus));
    }

    /**
     * Try to select thesaurus ids pass in param
     *
     * @param thesaurusIds list of thesaurus ids to select
     */
    public void tryToSelect(Collection<String> thesaurusIds) {
        if (thesaurusIds == null) {
            if (log.isDebugEnabled()) {
                log.debug("Try to select, ids are null");
            }
            return;
        }
        List<VradiTreeTableNode> nodes = new ArrayList<VradiTreeTableNode>();

        // Find recursivly
        for (String id : thesaurusIds) {
            VradiTreeTableNode rootNode = getRootNode();
            VradiTreeTableNode nodeFound = findNode(rootNode, id);
            if (nodeFound != null) {
                nodes.add(nodeFound);
            }
        }

        selectNodes(nodes);
    }

    /**
     * Try to unselect thesaurus pass in param
     *
     * @param thesaurus list of thesaurus to unselect
     */
    public void tryToUnselect(List<Thesaurus> thesaurus) {
        tryToUnselect(extractIds(thesaurus));
    }

    /**
     * Try to unselect thesaurus ids pass in param
     *
     * @param thesaurusIds list of thesaurus ids to select
     */
    public void tryToUnselect(Collection<String> thesaurusIds) {
        if (thesaurusIds == null) {
            return;
        }
        List<VradiTreeTableNode> nodes = new ArrayList<VradiTreeTableNode>();

        // Find recursivly
        for (String id : thesaurusIds) {
            VradiTreeTableNode nodeFound = findNode(getRootNode(), id);
            if (nodeFound != null) {
                nodes.add(nodeFound);
            }
        }

        unSelectNodes(nodes);
    }

    protected List<String> extractIds(List<Thesaurus> beans){
        List<String> ids = new ArrayList<String>();
        if (beans != null){
            for (Thesaurus bean : beans){
                ids.add(bean.getWikittyId());
            }
        }
        return ids;
    }

    @Override
    public VradiDataProvider getDataProvider() {
        return (VradiDataProvider)dataProvider;
    }

    @SuppressWarnings({"unchecked"})
    public NavTreeTableModel createTreeModel() {

        NavTreeTableModel model = null;
        if (rootThesaurus != null) {
            VradiTreeTableNode root = new VradiTreeTableNode(
                    String.class,
                    "Root node",
                    null,
                    null
            );
            VradiTreeTableNode rootThesaurusNode = new VradiTreeTableNode(
                    RootThesaurus.class,
                    rootThesaurus.getWikittyId(),
                    null,
                    new ThesaurusTreeTableNodeLoadors()
            );

            root.add(rootThesaurusNode);

            model = createModel(root, new ThesaurusTreeTableModel());

            // Populate childs nodes
            rootThesaurusNode.populateChilds(getBridge(), getDataProvider());
        } else {

            VradiTreeTableNode root = new VradiTreeTableNode(
                    String.class,
                    "Root node",
                    null,
                    new RootThesaurusTreeTableNodeLoadors()
            );

            model = createModel(root, new ThesaurusTreeTableModel());

            // Populate childs nodes
            root.populateChilds(getBridge(), getDataProvider());
        }

        return model;
    }
    
    public static RootThesaurus getRootThesaurus(VradiTreeTableNode node) {
        if (!node.getInternalClass().equals(RootThesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreRootThesaurus(node.getId());
    }

    public static Thesaurus getThesaurus(VradiTreeTableNode node) {
        if (!node.getInternalClass().equals(Thesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreThesaurus(node.getId());
    }

    protected void refresh(VradiTreeTableNode parentNode) {
        if (parentNode == null) {
            return;
        }
        List<VradiTreeTableNode> selectedNodes = getSelectedNodes();
        parentNode.removeAllChildren();
        parentNode.populateChilds(getBridge(), getDataProvider());
        selectNodes(selectedNodes);
    }

    @SuppressWarnings({"ConstantConditions"})
    @Override
    public void putWikitty(WikittyServiceEvent event) {
        
        // map between id and extensions "name" (not extension ids)
        Map<String, Set<String>> idAndExtensions = event.getIdExtensions();
        for (String wikittyId : event.getIds()) {
            Set<String> wikittyExtensions = idAndExtensions.get(wikittyId);

            // Thesaurus
            if (wikittyExtensions.contains(Thesaurus.EXT_THESAURUS)) {

                // Find existing
                VradiTreeTableNode existingNode = findNode(getRootNode(), wikittyId);

                // cas modification
                if (existingNode != null) {

                    // Parent id
                    VradiTreeTableNode parent = existingNode.getParent();

                    Thesaurus thesaurusConcerned = ThesaurusDataHelper.restoreThesaurus(wikittyId);

                    // Move case
                    String newParentId = thesaurusConcerned.getParent();
                    if (parent != null && !parent.getId().equals(newParentId)) {
                        if (log.isDebugEnabled()) {

                            Thesaurus parentFrom = ThesaurusDataHelper.restoreThesaurus(parent.getId());
                            Thesaurus newParent = ThesaurusDataHelper.restoreThesaurus(newParentId);
                            log.debug("Move case : " + thesaurusConcerned.getName()
                                    + " from parent" + parentFrom.getName()
                                    + " to parent" + newParent.getName());
                        }

                        VradiTreeTableNode newParentNode = findNode(getRootNode(), newParentId);

                        // Delete old
                        VradiTreeTableNode oldParent = existingNode.getParent();
                        refresh(oldParent);

                        // Destination sort
                        refresh(newParentNode);

                    } else {
                        if (log.isDebugEnabled()) {
                            log.debug("Modification case :  " + thesaurusConcerned.getName());
                        }

                        // Modification case
                        VradiTreeTableNode parentNode = existingNode.getParent();
                        refresh(parentNode);
                    }
                    return;
                }

                // Creation case
                else {
                    Thesaurus thesaurusConcerned = ThesaurusDataHelper.restoreThesaurus(wikittyId);

                    if (log.isDebugEnabled()) {
                        log.debug("Creation case :  " + thesaurusConcerned.getName());
                    }
                    
                    // Find parent node
                    VradiTreeTableNode parentNode = findNode(getRootNode(), thesaurusConcerned.getParent());

                    // Tri
                    refresh(parentNode);
                }
            }

            // Root thesaurus
            if (wikittyExtensions.contains(RootThesaurus.EXT_ROOTTHESAURUS)) {
                // Tri
                refresh(getRootNode());
            }
        }
    }

    @Override
    public void removeWikitty(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service remove event : " + event);
        }

        for (String wikittyId : event.getIds()) {
            VradiTreeTableNode node = findNode(getRootNode(), wikittyId);

            // if null : not in tree
            if (node != null) {
                if (log.isDebugEnabled()) {
                    log.debug("Removing node " + wikittyId + " from parent");
                }

                VradiTreeTableNode parent = node.getParent();
                refresh(parent);
            }
        }
    }

    @Override
    public void clearWikitty(WikittyServiceEvent event) {
        // should not happen in vradi
    }

    @Override
    public void putExtension(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service put extension event : " + event);
        }
        // TODO ???
    }

    @Override
    public void removeExtension(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service remove extension event : " + event);
        }
    }

    @Override
    public void clearExtension(WikittyServiceEvent event) {
        // should not happen in vradi
    }
}
