/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.admin.helpers;

import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.ui.admin.loadors.ClientUsersNodeLoadors;
import jaxx.runtime.swing.nav.tree.NavTreeHelper;
import jaxx.runtime.swing.nav.tree.NavTreeNodeChildLoador;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyServiceEvent;
import org.nuiton.wikitty.WikittyServiceListener;
import org.nuiton.wikitty.WikittyService.ServiceListenerType;

import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.tree.VradiDataProvider;
import com.jurismarches.vradi.ui.tree.VradiTreeNode;

/**
 * Navigation tree helpers.
 *
 * @author sletellier
 * @see jaxx.runtime.swing.nav.tree.NavTreeHelper
 */
public abstract class AdminNavigationTreeHelper extends NavTreeHelper<VradiTreeNode> implements WikittyServiceListener {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(AdminNavigationTreeHelper.class);

    public AdminNavigationTreeHelper(VradiDataProvider dataProvider) {
        super();
        setDataProvider(dataProvider);

        // register each tree on wikitty service
        VradiService.getWikittyService().addWikittyServiceListener(this, ServiceListenerType.ALL);
    }

    @Override
    public VradiDataProvider getDataProvider() {
        return (VradiDataProvider)dataProvider;
    }

    public TreeModel createTreeModel() {

        // Create root static node
        VradiTreeNode root = new VradiTreeNode(
                String.class,
                "Root node",
                null,
                null
        );

        VradiTreeNode child = createNode();

        // Add to root
        root.add(child);

        // Create model
        TreeModel model = createModel(root);

        // Populate childs nodes
        root.populateChilds(getBridge(), getDataProvider());

        return model;
    }

    public abstract VradiTreeNode createNode();
    
    public VradiTreeNode getParentSelectedCategoryNode(String categoryName) {
        return getParentCategoryNode(getSelectedNode(), categoryName);
    }
    
    public static VradiTreeNode getParentCategoryNode(VradiTreeNode node, String categoryName) {
        if (node == null) {
            return null;
        }

        if (categoryName.equals(node.getId())) {
            return node;
        }
        return getParentCategoryNode(node.getParent(), categoryName);
    }

    public VradiTreeNode getParentSelectedNode(Class type) {
        return getParentNode(getSelectedNode(), type);
    }

    public static VradiTreeNode getParentNode(VradiTreeNode node, Class type) {
        if (node == null) {
            return null;
        }

        if (type.equals(node.getInternalClass())) {
            return node;
        }
        return getParentNode(node.getParent(), type);
    }

    public void refresh(VradiTreeNode parentNode) {
        if (parentNode == null) {
            return;
        }
        parentNode.removeAllChildren();
        parentNode.populateChilds(getBridge(), getDataProvider());
        getBridge().nodeStructureChanged(parentNode);
    }

    public void createNode(Class<? extends NavTreeNodeChildLoador<String, ?, VradiTreeNode>> nodeLoadorsClass, String categoryName) {
        createNode(nodeLoadorsClass, categoryName, null);
    }

    public void createNode(Class<? extends NavTreeNodeChildLoador<String, ?, VradiTreeNode>> nodeLoadorsClass, String categoryName, String id) {

        VradiTreeNode parentNode = findNode(getRootNode(), categoryName);
        VradiTreeNode node = getChildLoador(nodeLoadorsClass).createNode(id, getDataProvider());

        insertNodeAndSelect(parentNode, node);
    }

    public void createUserNode(VradiTreeNode clientNode) {

        VradiTreeNode parentNode = findNode(clientNode, AdminBeanConstant.USER.getCategoryName());
        VradiTreeNode node = getChildLoador(ClientUsersNodeLoadors.class).createNode(null, getDataProvider());

        insertNodeAndSelect(parentNode, node);
    }

    protected void insertNodeAndSelect(VradiTreeNode parentNode, VradiTreeNode node) {

        parentNode.add(node);

        getBridge().notifyNodeInserted(node);
        TreePath path = new TreePath(getBridge().getPathToRoot(node));
        getUI().expandPath(path);
        selectNode(node);
    }

    /**
     * Method to reselect a category node (during some tree updates, while
     * saving, node id change so, it not possible to reselect using oldPath.
     *
     * @param node The node to reselect
     */
    public void reSelectCategory(VradiTreeNode node){
        String id = node.getParent().getId();
        log.debug ("Try to load subcategory for node : " + id);
        selectNode(id ,node.getId());
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#putWikitty(org.nuiton.wikitty.WikittyServiceEvent)
     */
    @Override
    public void putWikitty(WikittyServiceEvent event) {

    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#removeWikitty(org.nuiton.wikitty.WikittyServiceEvent)
     */
    @Override
    public void removeWikitty(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service remove event : " + event);
        }

        for (String wikittyId : event.getIds()) {
            VradiTreeNode node = findNode(getRootNode(), wikittyId);

            // if null : not in tree
            if (node != null) {
                if (log.isDebugEnabled()) {
                    log.debug("Removing node " + wikittyId + " from parent");
                }

                VradiTreeNode parent = node.getParent();
                int index = parent.getIndex(node);
                node.removeFromParent();
                getBridge().nodesWereRemoved(parent, new int[] {index}, new VradiTreeNode[]{node});
            }
        }
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#clearWikitty(org.nuiton.wikitty.WikittyServiceEvent)
     */
    @Override
    public void clearWikitty(WikittyServiceEvent event) {
        // should not happen in vradi
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#putExtension(org.nuiton.wikitty.WikittyServiceEvent)
     */
    @Override
    public void putExtension(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service put extension event : " + event);
        }
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#removeExtension(org.nuiton.wikitty.WikittyServiceEvent)
     */
    @Override
    public void removeExtension(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service remove extension event : " + event);
        }
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#clearExtension(org.nuiton.wikitty.WikittyServiceEvent)
     */

    @Override
    public void clearExtension(WikittyServiceEvent event) {
        // should not happen in vradi
    }

}
