/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: ThesaurusManager.java 1512 2010-09-28 09:22:36Z chatellier $
 * $HeadURL: svn+ssh://chatellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.2.0/vradi-services/src/main/java/com/jurismarches/vradi/services/managers/ThesaurusManager.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services.managers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.FacetTopic;
import org.nuiton.wikitty.PagedResult;
import org.nuiton.wikitty.Wikitty;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Like;
import org.nuiton.wikitty.search.Search;

import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.RootThesaurusImpl;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.search.CompareFilter;
import com.jurismarches.vradi.services.search.Filter;
import com.jurismarches.vradi.services.search.FilterList;
import com.jurismarches.vradi.services.search.VradiQueryParser;

/**
 * Class containing the methods to manage the thesaurus :
 * - node creation, update, retrieving, deletion
 * - get child node number
 * - get children
 *
 * @author schorlet
 * @version $Revision: 1512 $ $Date: 2010-09-28 11:22:36 +0200 (mar., 28 sept. 2010) $
 * @date 2010-01-22 20:18:29
 */
public class ThesaurusManager {

    /** log. */
    private static final Log log = LogFactory.getLog(ThesaurusManager.class);

    protected WikittyProxy wikittyProxy;

    public ThesaurusManager(WikittyProxy proxy) {
        this.wikittyProxy = proxy;
    }

    /**
     * Return root thesaurus ids (ie thesaurus nodes without parent) sorted
     * by root thesaurus name.
     * 
     * @return a {@link RootThesaurus} collection
     * @throws VradiException
     */
    public List<RootThesaurus> getRootThesaurus() throws VradiException {

        Search search = Search.query().eq(Element.ELT_EXTENSION, RootThesaurus.EXT_ROOTTHESAURUS);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(RootThesaurus.FQ_FIELD_TREENODE_NAME);

        PagedResult<RootThesaurus> rootThesaurusResult = wikittyProxy.findAllByCriteria(RootThesaurus.class, criteria);
        List<RootThesaurus> rootThesaurus = rootThesaurusResult.getAll();

        if (log.isDebugEnabled()) {
            log.debug("Root thesaurus list : " + rootThesaurus);
        }
        return rootThesaurus;
    }

    /**
     * Retourne l'ensemble des thesaurus qui sont attachés a un formulaire
     * sorted by thesaurus name.
     * 
     * @param form form dont on veut les thesaurus
     * @return thesaurus attachés au formulaire
     */
    public List<Thesaurus> getThesaurusAttachedToForm(Form form) {
        Search searchThesaurus = Search.query();
        searchThesaurus.eq(Thesaurus.FQ_FIELD_TREENODE_ATTACHMENT, form.getWikittyId());
        Criteria criteria = searchThesaurus.criteria();
        criteria.addSortAscending(Thesaurus.FQ_FIELD_TREENODE_NAME);

        PagedResult<Thesaurus> findAllByCriteria = wikittyProxy.findAllByCriteria(
                Thesaurus.class, criteria);
        List<Thesaurus> thesaurus = findAllByCriteria.getAll();
        return thesaurus;
    }

    public Thesaurus getThesaurus(String thesaurusId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getThesaurus(" + thesaurusId + ")");
        }
        Thesaurus node = wikittyProxy.restore(Thesaurus.class, thesaurusId);
        return node;
    }

    public List<Thesaurus> getThesaurus(List<String> thesaurusIds)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getThesaurus(" + thesaurusIds + ")");
        }
        List<Thesaurus> nodes = wikittyProxy.restore(Thesaurus.class, thesaurusIds);

        return nodes;
    }

    /**
     * Returns all children of the specified {@code thesaurusId} sorted by 
     * ordre and name.
     * 
     * @param thesaurusId thesaurus wikitty id
     * @return all list of <code>TreeNodeImpl</code>
     * @throws VradiException
     */
    public List<Thesaurus> getChildrenThesaurus(String thesaurusId)
            throws VradiException {

        if (log.isTraceEnabled()) {
            log.trace("getChildrenThesaurus(" + thesaurusId + ")");
        }

        Search query = Search.query();
        query.eq(Element.ELT_EXTENSION, Thesaurus.EXT_THESAURUS);
        query.eq(Thesaurus.FQ_FIELD_TREENODE_PARENT, thesaurusId);

        Criteria criteria = query.criteria();
        criteria.addSortAscending(Thesaurus.FQ_FIELD_THESAURUS_ORDER);
        criteria.addSortAscending(Thesaurus.FQ_FIELD_TREENODE_NAME);

        PagedResult<Thesaurus> nodes = wikittyProxy
                .findAllByCriteria(Thesaurus.class, criteria);
        List<Thesaurus> all = nodes.getAll();

        return all;
    }

    /**
     * Delete all thesaurus and sub thesaurus.
     * 
     * @param thesaurusId thesaurusId to delete
     * @return delete thesaurus ids
     * @throws VradiException
     */
    public List<String> deleteThesaurus(String thesaurusId) throws VradiException {
        List<String> result = wikittyProxy.deleteTree(thesaurusId);
        return result;
    }
    
    /**
     * Delete all thesaurus recursively.
     * 
     * @throws VradiException
     */
    public void deleteAllThesaurus() throws VradiException {
        
        List<RootThesaurus> rootThesauruses = getRootThesaurus();
        for (RootThesaurus rootThesaurus : rootThesauruses) {
            wikittyProxy.deleteTree(rootThesaurus.getWikittyId());
        }
    }

    public int getNbFormsForThesaurus(String thesaurusId)
            throws VradiException {
        if (log.isTraceEnabled()) {
            log.trace("getNbFormsForThesaurus(" + thesaurusId + ")");
        }

        Map.Entry<Thesaurus, Integer> entry =
                wikittyProxy.restoreNode(Thesaurus.class, thesaurusId, null);

        if (entry == null) {
            return 0;
        }

        return entry.getValue();
    }
    
    /**
     * Propose thesaurus nodes that might be in relation with a specified form.
     * Does'nt return thesaurus already associated with form.
     *
     * @param form      the <code>Form</code> containing the information needed
     *                  to search the thesaurus nodes
     * @return a list of <code>Thesaurus</code>
     * @throws VradiException
     */
    public List<Thesaurus> proposeThesaurus(Form form) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("proposeThesaurus(form)");
        }

        // recherche les thesaurus qui ont des tags et qui ne sont pas
        // deja associés au formulaire courant
        Search canditateSearch = Search.query();
        canditateSearch.bw(Thesaurus.FQ_FIELD_THESAURUS_TAGS, "*", "*");
        canditateSearch.neq(Thesaurus.FQ_FIELD_TREENODE_ATTACHMENT, form.getWikittyId());
        Criteria candidateCriteria = canditateSearch.criteria();
        PagedResult<Thesaurus> pagedResult = wikittyProxy.findAllByCriteria(Thesaurus.class, candidateCriteria);
        List<Thesaurus> canditateThesauruses = pagedResult.getAll();
        
        // search for the specified form
        Criteria criteria = Search.query()
            .eq(Element.ELT_EXTENSION, Form.EXT_FORM)
            .eq(Element.ELT_ID, form.getWikittyId())
            .criteria();
        
        // add a facet criteria for each thesaurus node id,
        // searching for tags of each thesaurus nodes
        for (Thesaurus canditateThesaurus : canditateThesauruses) {

            Set<String> tags = canditateThesaurus.getTags();
            Search tagSearch = Search.query(Search.KIND.OR);
            for (String tag : tags) {
                if (StringUtils.isNotBlank(tag)) {
                    tagSearch = tagSearch.keyword(tag);
                }
            }

            Criteria facetCriteria = tagSearch.criteria(canditateThesaurus.getWikittyId());
            criteria.addFacetCriteria(facetCriteria);
        }

        // execute the search
        criteria.setEndIndex(0);
        PagedResult<Form> forms = wikittyProxy.findAllByCriteria(Form.class, criteria);
        List<Thesaurus> result = new ArrayList<Thesaurus>();

        // collects topic names (which are thesaurus node ids)
        if (forms != null && forms.getNumFound() > 0) {
            List<String> thesaurusIds = new ArrayList<String>();

            Map<String, List<FacetTopic>> facetsMap = forms.getFacets();
            for (Map.Entry<String, List<FacetTopic>> entry : facetsMap.entrySet()) {
                List<FacetTopic> facetTopics = entry.getValue();
                for (FacetTopic facetTopic : facetTopics) {
                    if (facetTopic.getCount() > 0) {
                        thesaurusIds.add(facetTopic.getTopicName());
                        break;
                    }
                }
            }

            if (!thesaurusIds.isEmpty()) {
                result = getThesaurus(thesaurusIds);
            }
        }
        
        return result;
    }

    /**
     * Gets the query makers whose queries are potentially to modify
     * after a thesaurus node modification
     *
     * @param rootThesaurusName rootThesaurus name
     * @param thesaurusName the modified thesaurus node
     * @return a map containing the query makers and their queries which contains
     *         the thesaurus node name
     */
    public Map<Group, List<QueryBean>> getQueriesToModifyAfterThesaurusModification(
            String rootThesaurusName, String thesaurusName) {
        if (log.isDebugEnabled()) {
            log.debug("getQueriesToModifyAfterThesaurusModification(" + rootThesaurusName + ", " + thesaurusName + ")");
        }
        Map<Group, List<QueryBean>> results = new HashMap<Group, List<QueryBean>>();
        
        if (rootThesaurusName == null) {
            return results;
        }

        String requestPart = rootThesaurusName + ":" + (thesaurusName == null ? "" : thesaurusName);
        Criteria criteria = Search.query()
            .eq(Element.ELT_EXTENSION, Group.EXT_GROUP)
            .like(Group.FQ_FIELD_QUERYMAKER_QUERIES, requestPart, Like.SearchAs.AsText)
            .criteria();
        
        PagedResult<Group> pagedResult = wikittyProxy.findAllByCriteria(Group.class, criteria);
        List<Group> groups = pagedResult.getAll();

        if (log.isDebugEnabled()) {
            log.debug("[getQueriesToModifyAfterThesaurusModification]  " + groups.size() +
                    "Groups found for request part requestPart : " + requestPart);
        }

        for (Group group : groups) {
            
            Set<String> queries = group.getQueries();
            List<QueryBean> queriesToModify = new ArrayList<QueryBean>();
            
            for (String query : queries) {
                try {
                    QueryBean queryBean = new QueryBean(query, group.getWikittyId());
                    FilterList filter = VradiQueryParser.parse(queryBean.getQuery());
                    
                    if (isThesaurusInQuery(filter, rootThesaurusName, thesaurusName)) {
                        queriesToModify.add(queryBean);
                    }
                    
                } catch (Exception e) {
                    // ignored exception
                    log.warn(e.getMessage(), e);
                }
            }
            
            if (!queriesToModify.isEmpty()) {
                results.put(group, queriesToModify);
            }
        }

        return results;
    }

    protected boolean isThesaurusInQuery(FilterList list, String rootThesaurusName, String thesaurusName) {
        boolean insideQuery = false;
        List<Filter> filters = list.getFilters();
        
        for (Filter filter : filters) {
            if (filter instanceof FilterList) {
                insideQuery = isThesaurusInQuery((FilterList) filter, rootThesaurusName, thesaurusName);

            } else if (filter instanceof CompareFilter) {
                insideQuery = isThesaurusInQuery((CompareFilter) filter, rootThesaurusName, thesaurusName);
            }
            
            if (insideQuery) {
                break;
            }
        }
        return insideQuery;
    }

    protected boolean isThesaurusInQuery(CompareFilter compareFilter,
                               String rootThesaurusName, String thesaurusName) {
        String name = compareFilter.getName();
        String value = compareFilter.getValue();
        boolean result = rootThesaurusName.equals(name)
                && (thesaurusName == null || thesaurusName.equals(value));

        if (log.isDebugEnabled()) {
            log.debug("[isThesaurusInQuery] Root Thesaurus name : " + rootThesaurusName +
                    " name : " + name + " ThesaurusName : " +
                    thesaurusName + " value : " + value + " result : " + result);
        }
        return result;
    }

    protected void replaceThesaurusInQuery(FilterList list,
            String oldRootThesaurusName, String newRootThesaurusName,
            String oldThesaurusName, String newThesaurusName) {

        List<Filter> filters = list.getFilters();
        
        for (Filter filter : filters) {
            if (filter instanceof FilterList) {
                replaceThesaurusInQuery((FilterList) filter,
                        oldRootThesaurusName, newRootThesaurusName,
                        oldThesaurusName, newThesaurusName);

            } else if (filter instanceof CompareFilter) {
                replaceThesaurusInQuery((CompareFilter) filter,
                        oldRootThesaurusName, newRootThesaurusName,
                        oldThesaurusName, newThesaurusName);
            }
        }
    }

    protected void replaceThesaurusInQuery(CompareFilter compareFilter,
            String oldRootThesaurusName, String newRootThesaurusName,
            String oldThesaurusName, String newThesaurusName) {
        String value = compareFilter.getValue();
        
        if (value.equals(oldThesaurusName)) {
            String name = compareFilter.getName();

            if (oldRootThesaurusName.equals(name)) {

                compareFilter.setName(newRootThesaurusName);
                compareFilter.setValue(newThesaurusName);
            }
        }
    }

    /**
     * Create new thesaurus.
     * 
     * @param rootThesaurusName root thesaurus name (must contains only alphnum characters)
     * @return new created thesaurus
     * @throws VradiException if name is not valid
     */
    public RootThesaurus createRootThesaurus(String rootThesaurusName) throws VradiException {

        // check null name
        if (rootThesaurusName == null) {
            throw new VradiException("Null root thesaurus name");
        }

        // check name characters
        if (!rootThesaurusName.matches("\\w+")) {
            throw new VradiException("Root thesaurus name contains invalid characters : " + rootThesaurusName);
        }

        // check unicity
        if (isRootThesaurusNameExists(rootThesaurusName, null)) {
            throw new VradiException("Root thesaurus \"" + rootThesaurusName + "\" already exists");
        }

        // create thesaurus
        RootThesaurus newRootThesaurus = new RootThesaurusImpl();
        newRootThesaurus.setName(rootThesaurusName);
        newRootThesaurus = wikittyProxy.store(newRootThesaurus);
        
        return newRootThesaurus;
    }

    /**
     * Return true if rootThesaurusName already exists.
     * 
     * @param rootThesaurusName rootThesaurus name to test
     * @return true if rootThesaurusName already exists
     */
    public boolean isRootThesaurusNameExists(String rootThesaurusName) {
        return isRootThesaurusNameExists(rootThesaurusName, null);
    }

    /**
     * Return true if rootThesaurusName already exists.
     * 
     * @param rootThesaurusName rootThesaurus name to test
     * @param exceptedRootThesaurusId in case of rename, exclude exceptedRootThesaurusId from check
     * @return true if rootThesaurusName already exists
     */
    public boolean isRootThesaurusNameExists(String rootThesaurusName, String exceptedRootThesaurusId) {
        Search query = Search.query();
        query = query.eq(Element.ELT_EXTENSION, RootThesaurus.EXT_ROOTTHESAURUS);
        query = query.eq(Thesaurus.FQ_FIELD_TREENODE_NAME, rootThesaurusName);
        if (exceptedRootThesaurusId != null) {
            query = query.neq(Element.ELT_ID, exceptedRootThesaurusId);
        }

        Criteria criteria = query.criteria();
        criteria.setFirstIndex(0);
        criteria.setEndIndex(0);

        PagedResult<Wikitty> pagedResults = wikittyProxy.findAllByCriteria(criteria);

        boolean result = false;
        if (pagedResults.getNumFound() > 0) {
            result = true;
        }
        return result;
    }

    /**
     * Check if name of thesaurus is existing in rootThesaurus.
     *
     * @param rootThesaurus to check
     * @param thesaurusName name to check
     * @return true if thesaurus already exist
     */
    public boolean isThesaurusNameExistsInRootThesaurus(RootThesaurus rootThesaurus, String thesaurusName) {
        return isThesaurusNameExistsInRootThesaurus(rootThesaurus, thesaurusName, null);
    }

    /**
     * Check if name of thesaurus is existing in rootThesaurus.
     *
     * @param rootThesaurus to check
     * @param thesaurusName name to check
     * @param exceptedThesaurusId in case of rename, exclude exceptedThesaurusId from check
     * @return true if thesaurus already exist
     */
    public boolean isThesaurusNameExistsInRootThesaurus(RootThesaurus rootThesaurus, String thesaurusName, String exceptedThesaurusId) {

        Search query = Search.query();
        query = query.eq(Element.ELT_EXTENSION, Thesaurus.EXT_THESAURUS);
        query = query.eq(Thesaurus.FQ_FIELD_THESAURUS_ROOTTHESAURUS, rootThesaurus.getWikittyId());
        query = query.eq(Thesaurus.FQ_FIELD_TREENODE_NAME, thesaurusName);
        if (exceptedThesaurusId != null) {
            query = query.neq(Element.ELT_ID, exceptedThesaurusId);
        }

        Criteria criteria = query.criteria();
        criteria.setFirstIndex(0);
        criteria.setEndIndex(0);

        PagedResult<Wikitty> pagedResults = wikittyProxy.findAllByCriteria(criteria);

        boolean result = false;
        if (pagedResults.getNumFound() > 0) {
            result = true;
        }
        return result;
    }
}
