/*
 * #%L
 * Vradi :: Services
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services;

import static org.nuiton.i18n.I18n._;

import java.util.Arrays;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.TreeNodeImpl;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyExtensionStorage;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.WikittyServiceCached;
import org.nuiton.wikitty.WikittyServiceImpl;
import org.nuiton.wikitty.WikittyServiceNotifier;
import org.nuiton.wikitty.WikittyStorage;
import org.nuiton.wikitty.WikittyUserImpl;
import org.nuiton.wikitty.jdbc.WikittyExtensionStorageJDBC;
import org.nuiton.wikitty.jdbc.WikittyStorageJDBC;
import org.nuiton.wikitty.solr.WikittySearchEnginSolr;

import com.jurismarches.vradi.VradiServiceConfiguration;
import com.jurismarches.vradi.entities.ClientImpl;
import com.jurismarches.vradi.entities.FormImpl;
import com.jurismarches.vradi.entities.GroupImpl;
import com.jurismarches.vradi.entities.InfogeneImpl;
import com.jurismarches.vradi.entities.ModificationTagImpl;
import com.jurismarches.vradi.entities.QueryMakerImpl;
import com.jurismarches.vradi.entities.RootThesaurusImpl;
import com.jurismarches.vradi.entities.SendingImpl;
import com.jurismarches.vradi.entities.SessionImpl;
import com.jurismarches.vradi.entities.StatusImpl;
import com.jurismarches.vradi.entities.ThesaurusImpl;
import com.jurismarches.vradi.entities.UserImpl;
import com.jurismarches.vradi.entities.VradiUserImpl;
import com.jurismarches.vradi.entities.WebHarvestStreamImpl;
import com.jurismarches.vradi.entities.XmlFieldBindingImpl;
import com.jurismarches.vradi.entities.XmlStreamImpl;

/**
 * ServiceFactory.
 *
 * @author $Author$
 * @version $Revision$ $Date$
 * @since 24 févr. 2010 22:18:17
 */
public class ServiceFactory {

    /** Log. */
    private static final Log log = LogFactory.getLog(ServiceFactory.class);

    /** Wikitty service (cached version) for remote access. */
    protected static WikittyService vradiWikittyService;

    /** Single instance of storage service. */
    protected static VradiStorageService vradiStorageService;

    /** Single instance of wikitty proxy. */
    protected static WikittyProxy wikittyProxy = null;

    
    static {
        // Dans un block static pour definition des variables 
        // d'environnement solr et jbossts
        // A voir pourquoi, les converters s'initialise mal durant les tests
        
        // FIXME XXX TODO WARNING this seams to cause a huge communication BUG with jgroups !

        //ConverterUtil.initConverters();
        //ConvertUtils.register(new LocaleConverter(), Locale.class);
        VradiServiceConfiguration config = VradiServiceConfiguration.getInstance();
        //I18n.init(config.getLocale());
    }

    public static synchronized VradiStorageService getVradiStorageService() {
        if (vradiStorageService == null) {
            vradiStorageService = new VradiStorageServiceImpl();
        }
        return vradiStorageService;
    }

    /**
     * Creates wikitty service (cached).
     * 
     * Getter for service, with default service configuration.
     *
     * @return a unique instance of WikittyService
     */
    public static synchronized WikittyService getWikittyService() {
        if (vradiWikittyService == null) {
            VradiServiceConfiguration configuration = VradiServiceConfiguration.getInstance();

            // Contains :
            //  - WikittyServiceNotifier.WIKITTY_EVENT_JGROUP_OPTION
            //  - WikittyServiceNotifier.WIKITTY_EVENT_PROPAGATE_OPTION
            //  - WikittyServiceNotifier.WIKITTY_EVENT_JGROUP_OPTION
            Properties flatProperties = configuration.getFlatOptions();

            getWikittyService(flatProperties);
        }

        return vradiWikittyService;
    }

    /**
     * Creates wikitty service (cached).
     *
     * Constructor added for UI embedded mode, with custom configuration.
     * 
     * @param properties custom properties
     * @return a unique instance of WikittyService
     */
    public static synchronized WikittyService getWikittyService(Properties properties) {
        if (vradiWikittyService == null) {
            
            WikittyExtensionStorage extensionStorage = new WikittyExtensionStorageJDBC(properties);
            WikittyStorage wikittyStorage = new WikittyStorageJDBC(extensionStorage, properties);
            WikittySearchEnginSolr searchEngin = new WikittySearchEnginSolr(extensionStorage);

            // declare service
            WikittyService service = new WikittyServiceImpl(extensionStorage, wikittyStorage, searchEngin);

            // add notification (on server, propagate option set to true)
            WikittyService serviceNotifier = new WikittyServiceNotifier(service, properties);

            // add cache
            // par defaut le cache ne se synchronise pas
            vradiWikittyService = new WikittyServiceCached(serviceNotifier, properties);
        }

        return vradiWikittyService;
    }

    /**
     * Creates wikitty proxy.
     *
     * @return a unique instance of WikittyProxy.
     */
    public static synchronized WikittyProxy getWikittyProxy() {
        if (wikittyProxy == null) {
            WikittyService serviceCache = getWikittyService();

            // init proxy on cached service
            wikittyProxy = new WikittyProxy(serviceCache);

            // post operation : register extensions
            updateExtensions(wikittyProxy);
            // post operation : register migration classes
            ServiceMigration.configureMigration();

            // fait pour ne pas empecher le lancement de l'application
            // si l'indexation dure tres longtemps
            Thread indexThread = new Thread() {
                public void run() {
                 // post operation : reindex data if necessary
                    reindexData(wikittyProxy);
                }
            };
            indexThread.start();
            
        }

        return wikittyProxy;
    }

    /**
     * Register all extensions.
     * 
     * Already existing extensions with same version are skipped,
     * new extension are created and stored.
     * 
     * @param localWikittyProxy wikitty proxy
     */
    static void updateExtensions(WikittyProxy localWikittyProxy) {

        // easier if ordered
        // TODO EC20100616 don't work with required extensions :(
        List<WikittyExtension> extensions = Arrays.asList(
            ClientImpl.extensionClient,
            FormImpl.extensionForm,
            GroupImpl.extensionGroup,
            InfogeneImpl.extensionInfogene,
            ModificationTagImpl.extensionModificationTag,
            QueryMakerImpl.extensionQueryMaker,
            RootThesaurusImpl.extensionRootThesaurus,
            SendingImpl.extensionSending,
            SessionImpl.extensionSession,
            StatusImpl.extensionStatus,
            ThesaurusImpl.extensionThesaurus,
            TreeNodeImpl.extensionTreeNode,
            UserImpl.extensionUser,
            VradiUserImpl.extensionVradiUser,
            WebHarvestStreamImpl.extensionWebHarvestStream,
            WikittyUserImpl.extensionWikittyUser,
            XmlFieldBindingImpl.extensionXmlFieldBinding,
            XmlStreamImpl.extensionXmlStream);

        localWikittyProxy.storeExtension(extensions);
    }

    /**
     * Check if version has changed and launch wikitty service reindex.
     * 
     * @param configuration configuration (containing version)
     * @param wikittyService wikitty service
     */
    protected static void reindexData(WikittyProxy localWikittyProxy) {
        
        VradiServiceConfiguration configuration = VradiServiceConfiguration.getInstance();

        String currentVersion = configuration.getApplicationVersion();
        String lastVersion = configuration.getServiceVersion();
        
        // different is enougth to reindex
        // currentVersion is never null, lastVersion could be
        if (!currentVersion.equals(lastVersion)) {
            
            // do version change migration
            ServiceMigration.versionChangeMigration(lastVersion, currentVersion, localWikittyProxy);
            
            if (log.isInfoEnabled()) {
                log.info(_("Version change detected : %s. Reindexing data...", lastVersion));
            }
            long timeBefore = System.currentTimeMillis();
            
            // call syncEngin
            localWikittyProxy.syncEngin();

            long timeAfter = System.currentTimeMillis();
            
            if (log.isInfoEnabled()) {
                log.info("Reindexing completed in " + (timeAfter - timeBefore) + " ms");
            }
            
            // save new version
            configuration.setServiceVersion(currentVersion);
            configuration.saveForUser();
        }
        else {
            if (log.isInfoEnabled()) {
                log.info("No version change, skipping reindexing.");
            }
        }
    }
}
