/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import static org.nuiton.i18n.I18n._;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.DateUtils;
import org.nuiton.wikitty.BusinessEntityWikitty;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.PagedResult;
import org.nuiton.wikitty.Wikitty;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.VradiConstants.SessionStatus;
import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.entities.EntityHelper;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.ModificationTagImpl;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.SendingImpl;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.SessionImpl;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.StatusImpl;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.Configuration;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;

/**
 * Class containing the methods to manage the forms :
 * - creation, update, deletion, retrieving
 * - binding forms with clients whose queries are returning the forms
 *
 * @author schorlet
 * @version $Revision: 1332 $ $Date: 2010-09-10 15:20:13 +0200 (ven., 10 sept. 2010) $
 */
public class FormManager {
    
    private static final Log log = LogFactory.getLog(FormManager.class);

    private final WikittyProxy proxy;
    private final ThesaurusManager thesaurusManager;
    private final SearchManager searchManager;

    public FormManager(WikittyProxy proxy,
            ThesaurusManager thesaurusManager, SearchManager searchManager) {
        this.proxy = proxy;
        this.thesaurusManager = thesaurusManager;
        this.searchManager = searchManager;
    }

    public List<Form> updateForms(List<Form> forms) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateForms(forms)");
        }

        if (forms == null || forms.isEmpty()) {
            return forms;
        }

        List<Thesaurus> thesaurusList = new ArrayList<Thesaurus>();
        for (Form form : forms) {
            if (log.isDebugEnabled()) {
                log.debug("updating form: " + form.getId());
            }

            Date now = new Date();
            if (form.getCreationDate() == null) {
                form.setCreationDate(now);

            } else {
                
                // FIXME EC20100910 remove direct access to wikitty
                if(!form.getExtensionNames().contains(ModificationTag.EXT_MODIFICATIONTAG)) {
                    BusinessEntityWikitty businessEntityWikitty = (BusinessEntityWikitty)form;
                    Wikitty wikitty = businessEntityWikitty.getWikitty();
                    wikitty.addExtension(ModificationTagImpl.extensionModificationTag);
                }
                
                // FIXME EC20100910 remove direct access to wikitty
                form.setField(ModificationTag.EXT_MODIFICATIONTAG,
                        ModificationTag.FIELD_MODIFICATIONTAG_LASTMODIFIED, now);
            }

            Set<String> thesaurus = form.getThesaurus();
            if (thesaurus != null) {
                for (String thesaurusId : thesaurus) {
                    Thesaurus node = thesaurusManager.getThesaurus(thesaurusId);

                    if (node.getChildren() == null
                            || !node.getChildren().contains(form.getWikittyId())) {

                        node.addChildren(form.getWikittyId());

                        if (!thesaurusList.contains(node)) {
                            thesaurusList.add(node);
                        }
                    }
                }
            }
        }

        List<Form> result = proxy.store(forms);

        if (!thesaurusList.isEmpty()) {
            proxy.store(thesaurusList);
        }

        return result;
    }

    public Session getLastCloseSession() {

        // Get last close session to search after this one
        Search search = Search.query().eq(Element.ELT_EXTENSION, Session.EXT_SESSION);
        search.or().eq(Session.FQ_FIELD_SESSION_STATUS,
                String.valueOf(VradiConstants.SessionStatus.CANCELED.getValue()))
                .eq(Session.FQ_FIELD_SESSION_STATUS,
                String.valueOf(VradiConstants.SessionStatus.SENT.getValue()))
                .criteria();

        Criteria criteria = search.criteria();
        criteria.addSortDescending(Session.FQ_FIELD_SESSION_SESSIONDATE);

        PagedResult<Session> result = proxy.findAllByCriteria(Session.class, criteria);
        if (result == null || result.size() == 0) {
            return null;
        }
        return result.getFirst();
    }

    public Session getLastOpenSession() throws VradiException {

        Session lastClose = getLastCloseSession();

        String formatedLastCloseDate = null;
        if (lastClose != null) {

            if (log.isDebugEnabled()) {
                log.debug("Last closed session : " + lastClose.getSessionDate());
            }
            try {
                formatedLastCloseDate = WikittyUtil.formatDate(lastClose.getSessionDate());
            } catch (ParseException e) {
                log.error("Cant parse date");
            }
        }
        Search search = Search.query().eq(Element.ELT_EXTENSION, Session.EXT_SESSION);

        if (formatedLastCloseDate != null) {
            search.ge(Session.FQ_FIELD_SESSION_SESSIONDATE,
                        formatedLastCloseDate);
        }
        
        search.or().eq(Session.FQ_FIELD_SESSION_STATUS,
                String.valueOf(SessionStatus.ACTIVE.getValue()))
                .eq(Session.FQ_FIELD_SESSION_STATUS,
                String.valueOf(SessionStatus.STOPPED.getValue()))
                .eq(Session.FQ_FIELD_SESSION_STATUS,
                String.valueOf(SessionStatus.ERROR.getValue()));

        Criteria criteria = search.criteria();

        criteria.addSortDescending(Session.FQ_FIELD_SESSION_SESSIONDATE);

        PagedResult<Session> result = proxy.findAllByCriteria(Session.class, criteria);
        Session session = null;
        if (result != null && result.size() > 0) {
            session = result.getFirst();
        }

        if (log.isDebugEnabled() && session != null){
            log.debug("Last sessions found : " + session.getWikittyId() + " - "
                    + session.getSessionDate() +" with status : "
                    + session.getStatus());
        }

        return session;
    }

    /**
     * Find all session for specific date.
     * 
     * @param sessionDate session date
     * @throws VradiException
     * @return all session for date
     */
    public List<Session> getSessions(Date sessionDate) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getSessions for date " + sessionDate.toString());
        }
        Search search = Search.query().eq(Element.ELT_EXTENSION, Session.EXT_SESSION);

        search.bw(Session.FQ_FIELD_SESSION_SESSIONDATE,
                getBeginOfDaySolrDate(sessionDate),
                getEndOfDaySolrDate(sessionDate));

        // Get result
        Criteria criteria = search.criteria();
        List<Session> allSession = proxy.findAllByCriteria(Session.class, criteria).getAll();

        if (log.isDebugEnabled()) {
            log.debug(allSession.size() + " sessions found");
        }

        return allSession;
    }

    /**
     * Retourne la date a minuit au format solr.
     * 
     * @param date
     * @return
     * @throws VradiException
     */
    protected static String getBeginOfDaySolrDate(Date date) throws VradiException {
        Date localDate = DateUtils.setMinTimeOfDay(date);
        try {
            return WikittyUtil.formatDate(localDate);
        } catch (ParseException eee) {
            throw new VradiException("Cant format date " + date, eee);
        }
    }

    /**
     * Retourne la date a 23h59:59.999 au format solr.
     * 
     * @param date
     * @return
     * @throws VradiException
     */
    protected static String getEndOfDaySolrDate(Date date) throws VradiException {
        Date localDate = DateUtils.setMaxTimeOfDay(date);
        try {
            return WikittyUtil.formatDate(localDate);
        } catch (ParseException eee) {
            throw new VradiException("Cant format date " + date, eee);
        }
    }

    /**
     * Crée une nouvelle session initialisée.
     * 
     * @return la session créée
     * @throws VradiException
     */
    public Session createNewSession() throws VradiException {

        // TODO EC20100909 autorisé la création de plusieurs session
        Session lastSession = getLastOpenSession();
        if (lastSession != null) {
            throw new VradiException(_("Already opened session"));
        }

        // Get all session for this date
        Date now = new Date();

        // TODO EC201009089 ca sert a rien de toutes les récuperer pour avoir le dernier numéro
        List<Session> sessionList = getSessions(now);

        // Calculate number af session for day
        int num = sessionList.size() + 1;

        // create new session
        if (log.isDebugEnabled()) {
            log.debug("No session found, creating a new one with num : " + num);
        }

        Session currentSession = new SessionImpl();
        currentSession.setSessionDate(now);
        currentSession.setNum(num);
        currentSession.setStatus(SessionStatus.ACTIVE.getValue());
        currentSession.setParagraph(Configuration.getInstance().getSessionDefaultParagraph());

        currentSession = proxy.store(currentSession);

        return currentSession;
    }

    /**
     * Recherche tous les formulaires non lié et les lie à la session
     * demandée.
     * 
     * @param currentSession session a remplir
     * @return la session en argument
     * @throws VradiException
     */
    public Session bindForms(Session currentSession) throws VradiException {

        if (log.isDebugEnabled()) {
            log.debug("bindForms " + currentSession.getSessionDate());
        }

        try {

            // un sending = un utilsateur et ses formulaires
            List<Sending> toSend = findAllUnboundForms(currentSession);

            //Reload session to avoid getting wikitty obsolete error
            Session reloadedSession = proxy.restore(Session.class,
                        currentSession.getWikittyId());

            if (reloadedSession != null) {
                currentSession = reloadedSession;
            }

            // store the data
            // les sending sont deja sauvé par createAllSending()
            //proxy.store(toSend);

            for (Sending s : toSend) {
                currentSession.addSending(s.getWikittyId());
            }

            // Store the session
            currentSession = proxy.store(currentSession);

            if (log.isDebugEnabled()) {
                int size = (currentSession.getSending() == null ||
                        currentSession.getSending().isEmpty()) ?
                        0 : currentSession.getSending().size();

                log.debug("Saving session '" + currentSession.getSessionDate()
                        + "' number '" + currentSession.getNum()
                        + "' status '" + SessionStatus.getStatus(currentSession.getStatus()).getDescription()
                        + "' with '" + size
                        + "' sending");
            }

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't bind form : ", e);
            }
            throw new VradiException("Can't bind form : ", e);
        }
        return currentSession;
    }

    /**
     * Construit la liste des sending (un utilisateur et ses formulaires, soit
     * un mail).
     * 
     * @param currentSession session
     * @return les sending a traiter
     * @throws VradiException
     */
    protected List<Sending> findAllUnboundForms(Session currentSession) throws VradiException {

        // Only selected forms
        Status status = getSelectionneStatus();

        // List of the forms to bind ( = not yet bound)
        List<Sending> toSend = new ArrayList<Sending>();

        // recherche de tous les executeurs de requette (client, groupes...)
        // was all query make before
        //List<QueryMaker> queryMakers = searchManager.findQueryMakersWithQueries();
        List<Group> groups = searchManager.findGroupsWithQueries();

        // iterate on the group
        for (Group group : groups) {

            // List of the forms to bind ( = not yet bound)
            List<Form> formsToBind = new ArrayList<Form>();

            // normalement c'est pas null, sinon la requette findGroupsWithQueries
            // marche pas
            Set<String> queries = group.getQueries();

            for (String queryEntry : queries) {
                
                try {
                    QueryBean queryBean = new QueryBean(queryEntry);
                    String query = queryBean.getQuery();
    
                    // create a new VradiFormPage to find all the forms corresponding to the query
                    FormPagedResult vradiFormPage = new FormPagedResult();
    
                    // Limited for older inscription date
                    Date olderInscriptionDate = getOlderInscriptionDate(group);
                    
                    // Execute request
                    vradiFormPage = searchManager.findForms(query,
                            vradiFormPage,
                            Form.FQ_FIELD_FORM_DATEPUB,
                            olderInscriptionDate,
                            status.getWikittyId());

                    // Extract result
                    List<Form> resultForms = vradiFormPage.getFormsToShow();

                    if (log.isDebugEnabled()) {
                        log.debug("Query " + query + " return " + resultForms.size() + " forms");
                    }

                    formsToBind.addAll(resultForms);
                }
                catch (IOException ex) {
                    // pas grave on zape ?
                    if (log.isErrorEnabled()) {
                        log.error("Can't parse group query", ex);
                    }
                }
                catch (UnsupportedQueryException ex) {
                    // pas grave on zape ?
                    if (log.isErrorEnabled()) {
                        log.error("Can't execute group query", ex);
                    }
                }
            }

            // if some forms are to be bound, bind them
            if (!formsToBind.isEmpty()) {
                List<Sending> sendings = createAllSending(currentSession, group, formsToBind, true);
                toSend.addAll(sendings);
            }
        }

        return toSend;
    }

    /**
     * Retourne la plus ancienne date du query maker si c'est une date
     * ou des user gu groupe si c'est un groupe.
     * 
     * @param queryMaker queryMaker
     * @return
     */
    protected Date getOlderInscriptionDate(QueryMaker queryMaker) {
        Date result = new Date();

        if (queryMaker instanceof User) {
            User user = (User)queryMaker;
            Date inscriptionDate = user.getInscriptionDate();
            if (inscriptionDate != null && result.after(DateUtils.setMinTimeOfDay(inscriptionDate))){
                result = inscriptionDate;
            }
            if (log.isDebugEnabled()){
                log.debug("The older inscription date for user " + user.getName() + " is " + result) ;
            }
        }

        if (queryMaker instanceof Group) {
            Group group = (Group)queryMaker;
            Set<String> userIds = group.getUser();
            if (userIds != null){
                for (String userId : userIds) {
                    User user = proxy.restore(User.class, userId);
                    Date inscriptionDate = getOlderInscriptionDate(user);
                    if (inscriptionDate != null && result.after(inscriptionDate)) {
                        result = inscriptionDate;
                    }
                }
            }

            if (log.isDebugEnabled()) {
                log.debug("The older inscription date for group " + group.getName() + " is " + result) ;
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("The older inscription date " + result) ;
        }
        return result;
    }

    public List<Sending> createAllSending(Session currentSession, QueryMaker queryMaker, List<? extends Form> formsToBind, boolean removeAlreadyBind) {
        List<Sending> result = new ArrayList<Sending>();

        currentSession = proxy.restore(Session.class, currentSession.getWikittyId());

        // Create or find current sending for queryMaker
        Sending sending = findExistingOrCreateSending(currentSession, queryMaker);

        if (queryMaker instanceof User) {
            sending.setUser(queryMaker.getWikittyId());
            if (log.isDebugEnabled()) {
                log.debug("Add user " + queryMaker.getWikittyId());
            }

        } else if (queryMaker instanceof Group) {
            sending.setGroup(queryMaker.getWikittyId());
            if (log.isDebugEnabled()) {
                log.debug("Add group " + queryMaker.getWikittyId());
            }

            // Create sending for user of the group
            Set<String> usersIds = ((Group) queryMaker).getUser();
            if (usersIds != null){
                for (String userId : usersIds){
                    User user = proxy.restore(User.class, userId);
                    if (user != null) {
                        List<Sending> userSending = createAllSending(currentSession, user, formsToBind, removeAlreadyBind);
                        result.addAll(userSending);
                    }
                }
            }
        }

        if (removeAlreadyBind) {

            // Remove already bound forms with queryMaker
            formsToBind = removeAlreadyBound(currentSession, formsToBind, queryMaker);

            // Remove all form created after user inscription
            if (queryMaker instanceof User) {

                User user = (User) queryMaker;

                // Get user inscription date
                Date userInscriptionDate = user.getInscriptionDate();

                // Check date
                for (Form f : new ArrayList<Form>(formsToBind)) {

                    // Using last modification date
                    Date formCreationDate = f.getCreationDate();

                    // This test never happens, new date is set by default
                    // If not enable, not attach forms
                    if (!user.getEnable() || formCreationDate.before(userInscriptionDate)) {
                        if (log.isDebugEnabled()) {
                            log.debug("Removing form '" +
                                    f.getObjet() +
                                    "' because date of creation (" +
                                    formCreationDate +
                                    ") is before user inscription date (" +
                                    userInscriptionDate + ") for user : " +
                                    user.getName());
                        }

                        formsToBind.remove(f);
                    }
                }
            }

            if (log.isDebugEnabled()) {
                log.debug("Adding " + formsToBind.size() + " forms to sending");
            }

        }
        
        // Attach forms to sending.
        for (Form f : formsToBind) {
            String fid = f.getWikittyId();
            sending.removeForm(fid);
            sending.addForm(fid);
        }

        Set<String> formsAttached = sending.getForm();

        // If not empty
        if (formsAttached != null && !formsAttached.isEmpty()) {
            result.add(sending);
        }

        // If not empty
        if (!result.isEmpty()) {

            //restore session before adding sendings and storing to avoid
            //obsolete wikitty exception
            currentSession = proxy.restore(Session.class,currentSession.getWikittyId());

            // Attach sending to session
            for (Sending s : result) {
                currentSession.addSending(s.getWikittyId());
            }

            // store datas
            result = proxy.store(result);
            proxy.store(currentSession);
        }

        return result;
    }

    public List<Sending> removeAllSending(Session session, Form form, QueryMaker queryMaker) {

        Set<String> sendingList = session.getSending();
        List<String> sendingArray = new ArrayList<String>();
        if (sendingList != null) {
            sendingArray = new ArrayList<String>(sendingList);
        }

        List<Sending> sendings = proxy.restore(Sending.class, sendingArray);

        // To have the real instance
        //queryMaker = EntityHelper.castAsRealQueryMaker(queryMaker);

        // List of impacted queryMakers
        List<String> impacted = new ArrayList<String>();
        impacted.add(queryMaker.getWikittyId());

        if (queryMaker instanceof Group) {

            // Find sending for user of the group
            Set<String> userIds = ((Group) queryMaker).getUser();
            if (userIds != null) {
                for (String userId : userIds) {
                    impacted.add(userId);
                }
            }
        }

        // Remove
        List<Sending> sendingImpacted = new ArrayList<Sending>();
        for (Sending sending : sendings) {
            if (sending.getForm().contains(form.getWikittyId())) {
                if (impacted.contains(sending.getUser()) ||
                    impacted.contains(sending.getGroup())) {

                    // Remove
                    sending.removeForm(form.getWikittyId());

                    // Add deleted
                    sending.addDeletedForms(form.getWikittyId());
                    sendingImpacted.add(sending);
                }
            }

            // If is empty, set deleted
            if (sending.getForm().isEmpty()) {
                sending.setStatus(VradiConstants.SendingStatus.DELETED.getValue());
            }
        }
        proxy.store(sendingImpacted);

        List<Sending> result = proxy.restore(Sending.class, sendingArray);

        return result;
    }

    /**
     * Recherche dans la session l'object sending qui correspond au query maker
     * donné ou en crée une nouvelle s'il n'a pas été trouvé.
     * 
     * Le sending retourné n'est pas affecter au query maker.
     * 
     * @param currentSession
     * @param queryMaker
     * @return
     */
    protected Sending findExistingOrCreateSending(Session currentSession, QueryMaker queryMaker) {

        // Restore existing sending in session
        Set<String> sendingsId = currentSession.getSending();

        // 
        List<Sending> sendings = null;
        if (sendingsId != null) {
            sendings = proxy.restore(Sending.class, new ArrayList<String>(sendingsId));
        }
        else {
            sendings = new ArrayList<Sending>();
        }

        // chargement, si il existe, du sending concernant
        // ce client / user / group
        for (Sending sending : sendings) {
            String qmId = queryMaker.getWikittyId();
            if (qmId.equals(sending.getUser()) ||
                    qmId.equals(sending.getGroup()) &&
                    !(sending.getStatus() == VradiConstants.SendingStatus.DELETED.getValue())) {

                if (log.isDebugEnabled()) {
                    log.debug("Sending is existing for queryMaker : " + queryMaker);
                }

                return sending;
            }
        }

        // Set properties in default
        SendingImpl newSending = new SendingImpl();
        newSending.setSentDate(null);
        newSending.setReceptionDate(null);
        newSending.setParagraph(VradiConstants.DEFAULT_SENDING_PARAGRAPH);
        newSending.setReceptionProof(false);
        newSending.setStatus(VradiConstants.SendingStatus.TO_SEND.getValue());

        if (log.isDebugEnabled()) {
            log.debug("Sending dont exist for queryMaker : " + queryMaker + ", creating one");
        }
        return newSending;
    }

    /**
     * Retourne la liste {@code forms} d'origine en ayant filtré les formulaires
     * qui ont déja été lié.
     * 
     * @param currentSession
     * @param forms
     * @param queryMaker
     * @return
     */
    protected List<? extends Form> removeAlreadyBound(Session currentSession, List<? extends Form> forms, QueryMaker queryMaker) {
        List<String> formIds = new ArrayList<String>();

        // Extract formIds
        for (Form form : forms){
            formIds.add(form.getWikittyId());
        }

        // Recherche d'un sending existant
        String qmId = queryMaker.getWikittyId();

        Search andSearch = Search.query().and();

        Set<String> sendingIds = currentSession.getSending();

        // Except current
        if (sendingIds != null) {
            for (String currentSendingId : sendingIds) {
                andSearch.neq(Element.ELT_ID, currentSendingId);
            }
        }

        // If its attached or deleted
        andSearch.or().contains(Sending.FQ_FIELD_SENDING_FORM, formIds)
                      .contains(Sending.FQ_FIELD_SENDING_DELETEDFORMS, formIds);

        // And attached to client / user / group
        andSearch.or().eq(Sending.FQ_FIELD_SENDING_CLIENT, qmId)
                      .eq(Sending.FQ_FIELD_SENDING_USER, qmId)
                      .eq(Sending.FQ_FIELD_SENDING_GROUP, qmId);

        // And not selected
        andSearch.not()
                 .eq(Sending.FQ_FIELD_SENDING_STATUS,
                         String.valueOf(VradiConstants.SendingStatus.DELETED.getValue()));

        // Find
        PagedResult<Sending> pagedResult = proxy.findAllByCriteria(Sending.class, andSearch.criteria());

        if (log.isDebugEnabled()){
            log.debug("Remove " + pagedResult.size() + " already bounds forms");
        }
        for (Sending send : pagedResult.getAll()){
            formIds.removeAll(send.getForm());
        }
        if (log.isDebugEnabled()){
            log.debug("Rest " + formIds.size() + " forms");
        }

        List<Form> result = proxy.restore(Form.class, formIds);
        return result;
    }

    /*
     * Status
     */

    public Status getStatus(String statusId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getStatus(" + statusId + ")");
        }
        Status status = proxy.restore(Status.class, statusId);
        return status;
    }

    public List<Status> getStatuses(List<String> statusIds) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getStatuses(statusIds)");
        }
        List<Status> statuses = proxy.restore(Status.class, statusIds);
        return new ArrayList<Status>(statuses);
    }

    public List<Status> createDefaultStatuses() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("createDefaultStatuses()");
        }

        List<Status> all = new ArrayList<Status>();
        all.add(getNonTraiteStatus());
        all.add(getNonSelectionneStatus());
        all.add(getPreselectionneStatus());
        all.add(getSelectionneStatus());

        return new ArrayList<Status>(all);
    }

    public Status getNonTraiteStatus() {
        return getOrCreateStatus(VradiConstants.FormStatus.NON_TRAITE);
    }

    public Status getNonSelectionneStatus() {
        return getOrCreateStatus(VradiConstants.FormStatus.NON_SELECTIONNE);
    }

    public Status getPreselectionneStatus() {
        return getOrCreateStatus(VradiConstants.FormStatus.PRESELECTIONNE);
    }

    public Status getSelectionneStatus() {
        return getOrCreateStatus(VradiConstants.FormStatus.SELECTIONNE);
    }

    protected Status getOrCreateStatus(VradiConstants.FormStatus type){
        if (log.isDebugEnabled()) {
            log.debug("getOrCreateStatus(" + type.getName() + ")");
        }
        Search search = Search.query()
                .eq(Element.ELT_EXTENSION, Status.EXT_STATUS)
                .eq(Status.FQ_FIELD_STATUS_NAME, type.getName());

        Criteria criteria = search.criteria();

        Status status = proxy.findByCriteria(Status.class,
                criteria);

        if (status == null) {
            log.warn("Status not found creating : " + type.getName());

            status = new StatusImpl();
            status.setName(type.getName());
            status.setValue(type.getValue());
            status.setDescription(type.getDescription());
            status.setModifiable(false);

            status = proxy.store(status);
        }
        return status;
    }

    public List<Status> updateStatuses(List<Status> statuses)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateStatuses(statuses)");
        }

        try {
            List<Status> result = proxy.store(statuses);
            return result;

        } catch (Exception e) {
            log.error("Cant update statuses : ", e);
            throw new VradiException("Cant update statuses : ", e);
        }
    }

    public Status updateStatus(Status status) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateStatus(status)");
        }

        try {
            status = proxy.store(status);
            return status;

        } catch (Exception e) {
            log.error("Cant update status : ", e);
            throw new VradiException("Cant update status : ", e);
        }
    }

    public void deleteStatuses(List<String> statusIds) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("deleteStatus(" + statusIds + ")");
        }

        try {
            proxy.delete(statusIds);

        } catch (Exception e) {
            log.error("Cant delete status : ", e);
            throw new VradiException("Cant delete status : ", e);
        }
    }

    protected List<User> getUsersOfClient(String clientId) {

        Search search = Search.query().eq(Element.ELT_EXTENSION, User.EXT_USER)
                .eq(User.FQ_FIELD_USER_CLIENT, clientId);
        Criteria criteria = search.criteria();

        PagedResult<User> usersPageResult = proxy.findAllByCriteria(User.class, criteria);
        return new ArrayList<User>(usersPageResult.getAll());

    }

}
