/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package com.jurismarches.vradi.services;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import com.jurismarches.vradi.services.managers.FormManager;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.BusinessEntity;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.PagedResult;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;

import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.beans.ThesaurusCartography;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.WebHarvestStream;
import com.jurismarches.vradi.entities.XmlFieldBinding;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.managers.SearchManager;
import com.jurismarches.vradi.services.managers.ThesaurusManager;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;

/**
 * Vradi data service implementation (on a local wikiity proxy).
 * 
 * @author chatellier
 * @version $Revision: 1274 $
 * 
 * Last update : $Date: 2010-09-06 14:48:32 +0200 (lun., 06 sept. 2010) $
 * By : $Author: jcouteau $
 */
public class VradiDataServiceImpl implements VradiDataService {

    /** Log.*/
    private static final Log log = LogFactory.getLog(VradiDataServiceImpl.class);

    /** Wikiity proxy. */
    protected WikittyProxy wikittyProxy;

    protected ThesaurusManager thesaurusManager;
    protected SearchManager searchManager;
    protected FormManager formManager;

    /**
     * Constructor with a wikitty proxy.
     * 
     * Make sure that wikitty proxy doesn't access remote service.
     * 
     * @param wikittyProxy wikitty proxy
     */
    public VradiDataServiceImpl(WikittyProxy wikittyProxy) {
        this.wikittyProxy = wikittyProxy;
        initManagers();
    }

    /**
     * Init managers.
     */
    protected void initManagers() {
        thesaurusManager = new ThesaurusManager(wikittyProxy);
        searchManager = new SearchManager(wikittyProxy, thesaurusManager);
        formManager = new FormManager(wikittyProxy, thesaurusManager, searchManager);
    }

    /**
     * Do a find all by applying a criteria to WikittyProxy.
     * 
     * Example :
     * List<Client> client = VradiServiceHelper.findAllByExtension(Client.class, Client.EXT_CLIENT)
     * 
     * @param type type to search
     * @param extension extension to search
     * @param sortFields sort fields (ascending to sort on)
     * @param <T> type of result
     * 
     * @return all of entities with extension
     */
    protected <T extends BusinessEntity> List<T> findAllByExtension(Class<T> type, String extension, String... sortFields) {
        
        Search search = Search.query().eq(Element.ELT_EXTENSION, extension);
        Criteria criteria = search.criteria();

        // set sort order
        for (String sortField : sortFields) {
            criteria.addSortAscending(sortField);
        }

        PagedResult<T> findAllByCriteria = wikittyProxy.findAllByCriteria(type, criteria);
        List<T> results = findAllByCriteria.getAll();
        
        return new ArrayList<T>(results);
    }

    /**
     * Find all clients.
     *
     * @return all clients
     */
    @Override
    public List<Client> findAllClients() {
        return findAllByExtension(Client.class, Client.EXT_CLIENT, Client.FQ_FIELD_CLIENT_NAME);
    }

    /**
     * Find all users.
     *
     * @return all users
     */
    @Override
    public List<User> findAllUsers() {

        //FIXME JC 25/08/2010 Should also not return user that have id of a non existing client

        //Create query to return only User that have clients. Users with no
        //clients should not exist and are corrupted.
        Search search = Search.query().eq(User.FQ_FIELD_USER_CLIENT, "*");
        Criteria criteria = search.criteria();

        PagedResult<User> users = wikittyProxy.findAllByCriteria(User.class,
                criteria);

        return new ArrayList<User>(users.getAll());
    }

    /**
     * Find all groups.
     *
     * @return all groups
     */
    @Override
    public List<Group> findAllGroups() {
        return findAllByExtension(Group.class, Group.EXT_GROUP, Group.FQ_FIELD_GROUP_NAME);
    }

    /**
     * Create all default status.
     *
     * @return all status created
     */
    @Override
    public List<Status> createDefaultStatuses() throws VradiException {
        return formManager.createDefaultStatuses();
    }
    
    /**
     * Find all webHarvest stream.
     *
     * @return all stream webHarvest
     */
    @Override
    public List<WebHarvestStream> findAllWebHarvest() {
        return findAllByExtension(WebHarvestStream.class, WebHarvestStream.EXT_WEBHARVESTSTREAM, WebHarvestStream.FQ_FIELD_XMLSTREAM_NAME);
    }

    /**
     * Find all xml streams.
     *
     * @return all xml streams
     */
    @Override
    public List<XmlStream> findAllXmlStreams() {
        return findAllByExtension(XmlStream.class, XmlStream.EXT_XMLSTREAM, XmlStream.FQ_FIELD_XMLSTREAM_NAME);
    }
    
    /**
     * Find all thesaurus.
     *
     * @return all xml streams
     */
    @Override
    public List<Thesaurus> findAllThesaurus() {
        return findAllByExtension(Thesaurus.class, Thesaurus.EXT_THESAURUS, Thesaurus.FQ_FIELD_TREENODE_NAME);
    }

    /**
     * Find all users for a client.
     *
     * @param clientId client wikitty id
     * @return all users for client
     */
    @Override
    public List<User> findClientUsers(String clientId) {

        // Get user of client
        Search search = Search.query().eq(User.FQ_FIELD_USER_CLIENT, clientId);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(User.FQ_FIELD_USER_NAME);

        PagedResult<User> usersResult = wikittyProxy.findAllByCriteria(User.class, criteria);
        List<User> users = new ArrayList<User>(usersResult.getAll());
        return users;
    }

    /**
     * Retrieves the form type whose name is the parameter 'name'
     *
     * @param name the name of the form type we want to retrieve
     * @return the form type whose name is the parameter
     * @throws VradiException 
     */
    @Override
    public WikittyExtension getFormType(String name) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getFormType(" + name + ")");
        }

        try {
            WikittyService wikittyService = wikittyProxy.getWikittyService();
            WikittyExtension wikittyExtension = wikittyService.restoreExtensionLastVersion(null, name);
            return wikittyExtension;
            
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't get form type", eee);
            }
            throw new VradiException("Can't get form type", eee);
        }
    }

    /**
     * Updates the form type whose name is 'name'.
     *
     * @param name the name of the form type to update
     * @param fields the new fields of the form type
     * @param requires the new requires of the form type
     * @param tagValues the new tag values of the form type
     * @return the form type up to date
     */
    public WikittyExtension updateFormType(String name, Map<String, FieldType> fields,
           String requires, Map<String, String> tagValues) throws VradiException {
        
        if (log.isDebugEnabled()) {
            log.debug("updateFormType(" + name + ", fields, requires, tagValues)");
        }

        WikittyExtension resultExtension;

        try {
            WikittyExtension lastVersion = getFormType(name);
            String newVersion;
            
            if (lastVersion != null) {
                newVersion = WikittyUtil.incrementMajorRevision(lastVersion.getVersion());
            } else {
                newVersion = WikittyUtil.DEFAULT_VERSION;
            }
            
            resultExtension = new WikittyExtension(name, newVersion,
                    requires, new LinkedHashMap<String, FieldType>(fields));
            
            if (tagValues != null) {
                for (Map.Entry<String, String> entry : tagValues.entrySet()) {
                    resultExtension.addTagValue(entry.getKey(), entry.getValue());
                }
            }
            
            List<WikittyExtension> extensions = Arrays.asList(resultExtension);
            wikittyProxy.getWikittyService().storeExtension(null, extensions);
            
            if (log.isDebugEnabled()) {
                log.debug(String.format("FormType named %s saved with id: %s and version: %s",
                    name, resultExtension.getId(), newVersion));
            }
        
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Update form type", eee);
            }
            throw new VradiException("Update form type", eee);
        }
        
        return resultExtension;
    }

    /**
     * Update the form type given in parameter
     *
     * @param extension the form type to update
     * @return the form type up to date
     */
    public WikittyExtension updateFormType(WikittyExtension extension) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateFormType(" + extension.toDefinition() + ")");
        }
        
        try {
            Map<String, FieldType> fields = new LinkedHashMap<String, FieldType>();
            
            for (String fieldName : extension.getFieldNames()) {
                FieldType fieldType = extension.getFieldType(fieldName);
                fields.put(fieldName, fieldType);
            }
    
            WikittyExtension updated = updateFormType(extension.getName(), fields,
                    extension.getRequires(), extension.getTagValues());
            
            return updated;
            
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't update form type", eee);
            }
            throw new VradiException("Can't update form type", eee);
        }
    }
    
    @Override
    public FormPagedResult findForms(String query, WikittyExtension extension,
            String dateType, Date beginDate, Date endDate, String streamId,
            List<String> thesaurus, String[] statusIds,
            final FormPagedResult formPagedResult) throws UnsupportedQueryException, VradiException {

        FormPagedResult result = searchManager.findForms(query, extension, dateType, beginDate, endDate,
                streamId, thesaurus, statusIds, formPagedResult);
        return result;
    }
    
    @Override
    public ThesaurusCartography getThesaurusCartography(String query,
            WikittyExtension extension, String dateType, Date beginDate,
            Date endDate, String streamId, List<String> thesaurus, String[] statusIds)
            throws VradiException, UnsupportedQueryException {
        
        ThesaurusCartography thesaurusCartography = searchManager.getThesaurusCartography(
                query, extension, dateType, beginDate, endDate, streamId, thesaurus, statusIds);
        
        return thesaurusCartography;
    }
    
    /**
     * Updates the xml field bindings given in parameters.
     *
     * @param bindings the list of the xml field bindings to update
     * @return the list of the xml field bindings up to date
     */
    public List<XmlFieldBinding> updateXmlFieldBindings(List<XmlFieldBinding> bindings)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateXmlFieldBindings(bindings)");
        }
        
        try {
            List<XmlFieldBinding> list = new ArrayList<XmlFieldBinding>();
        
            if (bindings != null) {
                bindings = wikittyProxy.store(bindings);
                list.addAll(bindings);
            }
            
            return list;
            
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't update xml field bindings", eee);
            }
            throw new VradiException("Can't update xml field bindings", eee);
        }
    }
    
    /**
     * Retrieves all the form types
     *
     * @return a list containing all the form types
     */
    @Override
    public List<WikittyExtension> getAllFormTypes() throws VradiException {

        List<WikittyExtension> extensions = new ArrayList<WikittyExtension>();

        WikittyService wikittyService = wikittyProxy.getWikittyService();
        List<String> allExtensionIds = wikittyService
                .getAllExtensionsRequires(null, Form.EXT_FORM);
        Map<String, WikittyExtension> lastVersions =
                new HashMap<String, WikittyExtension>();

        for (String extensionId : allExtensionIds) {
            String extensionName  = WikittyExtension.computeName(extensionId);
            if(lastVersions.get(extensionName) == null) {
                WikittyExtension extension = wikittyService
                        .restoreExtensionLastVersion(null, extensionName);
                extension = computeExtension(extension);
                lastVersions.put(extensionName, extension);
            }
        }
        extensions.addAll(lastVersions.values());

        return extensions;
    }

    /**
     * Compute the specified extension and return a new one with ordered fields.
     * Fields order is specified by their tag values.
     */
    protected WikittyExtension computeExtension(WikittyExtension extension) {

        Collection<String> fieldNames = extension.getFieldNames();
        List<String> fieldOrder = new ArrayList<String>();
        List<String> orderlessFields = new ArrayList<String>();

        for (String fieldName : fieldNames) {
            FieldType fieldType = extension.getFieldType(fieldName);

            try {
                Integer.valueOf(fieldType.getTagValue("rank"));
                fieldOrder.add(fieldName);

            } catch(Exception eee) {
                orderlessFields.add(fieldName);
            }
        }

        LinkedHashMap<String, FieldType> fields = new LinkedHashMap<String, FieldType>();

        int size = fieldOrder.size();
        for(int i = 0; i < size; i++) {
            String key = fieldOrder.get(i);
            fields.put(key, extension.getFieldType(key));
        }

        size = orderlessFields.size();
        for(int i = 0; i < size; i++) {
            String key = orderlessFields.get(i);
            fields.put(key, extension.getFieldType(key));
        }

        WikittyExtension fieldOrderedExtension =
                new WikittyExtension(extension.getName(),
                        extension.getVersion(),
                        extension.getRequires(),
                        fields);
        fieldOrderedExtension.setTagValues(extension.getTagValues());

        return fieldOrderedExtension;
    }
    
    @Override
    public List<RootThesaurus> getRootThesaurus() throws VradiException {
        return thesaurusManager.getRootThesaurus();
    }

    @Override
    public int getNbFormsForThesaurus(String thesaurusId)
            throws VradiException {
        return thesaurusManager.getNbFormsForThesaurus(thesaurusId);
    }

    @Override
    public List<Thesaurus> getChildrenThesaurus(String thesaurusId) throws VradiException {
        return thesaurusManager.getChildrenThesaurus(thesaurusId);
    }
}
