/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package com.jurismarches.vradi.services;

import java.util.Date;
import java.util.List;
import java.util.Map;

import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyProxy;

import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.beans.ThesaurusCartography;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.WebHarvestStream;
import com.jurismarches.vradi.entities.XmlFieldBinding;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;

/**
 * This service contains business code to do on data before calling
 * {@link WikittyProxy}.
 * 
 * It's intended to be used on a local {@link WikittyProxy}.
 * 
 * @author chatellier
 * @version $Revision: 1308 $
 * 
 * Last update : $Date: 2010-09-09 17:35:18 +0200 (jeu., 09 sept. 2010) $
 * By : $Author: chatellier $
 */
public interface VradiDataService {

    /**
     * Find all clients ordered by client name.
     *
     * @return all clients
     */
    public List<Client> findAllClients();

    /**
     * Find all users ordered by user name.
     *
     * @return all users
     */
    public List<User> findAllUsers();
    
    /**
     * Find all users for a client ordered by user name.
     *
     * @param clientId client wikitty id
     * @return all users for client
     */
    public List<User> findClientUsers(String clientId);
    
    /**
     * Find all thesaurus orderer by thesaurus name.
     *
     * TODO EC20100610 maybe nerver used ?
     * 
     * @return all thesaurus
     */
    public List<Thesaurus> findAllThesaurus();

    /**
     * Find all groups ordered by group name.
     *
     * @return all groups
     */
    public List<Group> findAllGroups();

    /**
     * Create all default status.
     *
     * @return all status
     * @throws VradiException
     */
    public List<Status> createDefaultStatuses() throws VradiException;
    
    /**
     * Find all webHarvest stream ordered by stream name.
     *
     * @return all stream webHarvest
     */
    public List<WebHarvestStream> findAllWebHarvest();

    /**
     * Find all xml streams ordered by stream name.
     *
     * @return all xml streams
     */
    public List<XmlStream> findAllXmlStreams();

    /**
     * Return root thesaurus nodes.
     * 
     * Since 0.1.0, root thesaurus are not created anymore.
     * 
     * @return a {@link RootThesaurus}
     * @throws VradiException
     */
    List<RootThesaurus> getRootThesaurus() throws VradiException;

    /**
     * Retrieves the form type whose name is the parameter 'name'
     *
     * @param name the name of the form type we want to retrieve
     * @return the form type whose name is the parameter
     * @throws VradiException 
     */
    WikittyExtension getFormType(String name) throws VradiException;

    /**
     * Creates or Update a form type.
     *
     * @param extension the extension designing the form type
     * @return a <code>WikittyExtension</code>
     * @throws VradiException 
     */
    WikittyExtension updateFormType(WikittyExtension extension) throws VradiException;

    /**
     * Creates or Update a form type.
     *
     * @param name      the name of the form type
     * @param fields    the fields of the form type
     * @param requires  the extension required by the extension to update
     * @param tagValues the map containing the tags and their values
     * @return a <code>WikittyExtension</code>
     * @throws VradiException 
     */
    WikittyExtension updateFormType(String name, Map<String, FieldType> fields, String requires, Map<String, String> tagValues) throws VradiException;
    
    /**
     * Searches for forms with the specified filters.
     * 
     * The <code>beginDate</code> and <code>endDate</code> specifies a
     * date range filter which is applied on <code>dateType</code> field.
     * The date range filter is applied only if the three parameters are not null.
     * 
     * @param query     the query filter
     * @param extension the extension filter
     * @param dateType  the date field. Must be fully qualified (extension.field)
     * @param beginDate the begin date filter
     * @param endDate   the end date filter
     * @param streamId  id of stream
     * @param thesaurus the thesaurus filter
     * @param statusIds  the ids of the status filters
     * @param formPagedResult ...
     * @return FormPagedResult
     * 
     * @throws UnsupportedQueryException
     * @throws VradiException
     */
    FormPagedResult findForms(String query, WikittyExtension extension, String dateType,
                   Date beginDate, Date endDate, String streamId, java.util.List<String> thesaurus,
                   String[] statusIds, FormPagedResult formPagedResult)
            throws VradiException, UnsupportedQueryException;
    
    /**
     * Computes the number of forms associated with every thesaurus node.
     * 
     * @param query     the query filter
     * @param extension the extension filter
     * @param dateType  the date field. Must be fully qualified (extension.field)
     * @param beginDate the begin date filter
     * @param endDate   the end date filter
     * @param streamId  id of stream
     * @param thesaurus the thesaurus filter
     * @param statusIds  the ids of the status filters
     * @return a <code>VradiCartographyDTO</code> containing forms result and a map of form count indexed by thesaurus nodes
     * @throws VradiException
     * @throws UnsupportedQueryException
     */
    ThesaurusCartography getThesaurusCartography(String query,
            WikittyExtension extension, String dateType, Date beginDate,
            Date endDate, String streamId, java.util.List<String> thesaurus, String[] statusIds)
            throws VradiException, UnsupportedQueryException;
    
    /**
     * Update a list of <code>XmlFieldBinding</code>.
     *
     * @param bindings the list of XmlFieldBinding to update
     * @return the list of updated XmlFieldBinding
     * @throws VradiException 
     */
    List<XmlFieldBinding> updateXmlFieldBindings(List<XmlFieldBinding> bindings) throws VradiException;
    
    /**
     * Retrieves all the form types
     *
     * @return a list containing all the form types
     * @throws VradiException 
     */
    public List<WikittyExtension> getAllFormTypes() throws VradiException;
    
    /**
     * Returns all children of the specified <code>thesaurusId</code>.
     * @param thesaurusId 
     * 
     * @return all list of <code>TreeNodeImpl</code>
     * @throws VradiException
     */
    List<Thesaurus> getChildrenThesaurus(String thesaurusId) throws VradiException;
    
    /**
     * Computes the number of forms associated with the specified {@code thesaurusId}.
     *
     * @param thesaurusId thesaurus id
     * @return the number of forms associated
     * @throws VradiException
     */
    int getNbFormsForThesaurus(String thesaurusId) throws VradiException;

}
