/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services;

import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

import org.apache.activemq.broker.BrokerFactory;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyExtensionStorage;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.WikittyService;
import org.sharengo.wikitty.WikittyStorage;
import org.sharengo.wikitty.jdbc.WikittyExtensionStorageJDBC;
import org.sharengo.wikitty.jdbc.WikittyStorageJDBC;
import org.sharengo.wikitty.jms.WikittyExtensionStorageJMS;
import org.sharengo.wikitty.jms.WikittyExtensionSubscriberJMS;
import org.sharengo.wikitty.jms.WikittyJMSUtil;
import org.sharengo.wikitty.jms.WikittyStorageJMS;
import org.sharengo.wikitty.jms.WikittyStorageSubscriberJMS;
import org.sharengo.wikitty.multistorage.MultiStorageConfiguration;
import org.sharengo.wikitty.multistorage.WikittyServiceMultiStorage;
import org.sharengo.wikitty.multistorage.MultiStorageConfiguration.BasicConfiguration;

import com.jurismarches.vradi.entities.ClientImpl;
import com.jurismarches.vradi.entities.FormImpl;
import com.jurismarches.vradi.entities.GroupImpl;
import com.jurismarches.vradi.entities.InfogeneImpl;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.QueryMakerImpl;
import com.jurismarches.vradi.entities.SendingImpl;
import com.jurismarches.vradi.entities.SessionImpl;
import com.jurismarches.vradi.entities.StatusImpl;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.UserImpl;
import com.jurismarches.vradi.entities.VradiAdminImpl;
import com.jurismarches.vradi.entities.XmlFieldBindingImpl;
import com.jurismarches.vradi.entities.XmlStreamImpl;
import com.jurismarches.vradi.migration.Migration005;
import com.jurismarches.vradi.migration.MigrationWikittyService;

/**
 * ServiceFactory.
 *
 * @author $Author$
 * @version $Revision$ $Date$
 * @since 24 févr. 2010 22:18:17
 */
public class ServiceFactory {
    static final Log log = LogFactory.getLog(ServiceFactory.class);

    /**
     * single instance of storage service
     */
    private static VradiStorageService vradiStorageService = null;
    /**
     * single instance of xml service
     */
    private static XmlStreamService xmlStreamService = null;
    /**
     * single instance of mailing service
     */
    private static MailingService mailingService = null;

    /**
     * single instance of wikitty proxy
     */
    private static WikittyProxy wikittyProxy = null;

    static {
        Configuration config = Configuration.getInstance();
        config.printConfig();
    }
    
    public static synchronized VradiStorageService getVradiStorageService() {
        if (vradiStorageService == null) {
            vradiStorageService = new VradiStorageServiceImpl();
        }
        return vradiStorageService;
    }

    public static synchronized XmlStreamService getXmlStreamService() {
        if (xmlStreamService == null) {
            xmlStreamService = new XmlStreamServiceImpl();
        }
        return xmlStreamService;
    }

    public static synchronized MailingService getMailingService() {
        if (mailingService == null) {
            mailingService = new MailingServiceImpl();
        }
        return mailingService;
    }

    /**
     * Creates wikitty proxy.
     * creates the vradi data directory and the solr config file if not exists.
     *
     * @return a unique instance of WikittyProxy.
     */
    public static synchronized WikittyProxy getWikittyProxy() {
        if (wikittyProxy == null) {
            boolean performMigration = !Configuration.getInstance().skipMigration();
            Migration005 migration = new Migration005();
            
            if (performMigration) {
                // prepare migration
                migration.prepareMigration();
            }
            
            MultiStorageConfiguration configuration = new BasicConfiguration();

            configureJDBC(configuration);
            configurePublisherJMS(configuration);
            configureSubscriberJMS(configuration);
            
            // FIXME EC-20100426 replace it with WikittyServiceMultiStorage
            // when migration will be removed
            WikittyServiceMultiStorage multiStorage = new MigrationWikittyService(configuration);
            wikittyProxy = new WikittyProxy();
            wikittyProxy.setWikittyService(multiStorage);
            
            updateExtensions(wikittyProxy);
            
            if (performMigration) {
                // perform migration
                migration.performMigration(wikittyProxy);
            }
        }
        
        return wikittyProxy;
    }
    
    
    static void updateExtensions(WikittyProxy wikittyProxy) {
        List<WikittyExtension> extensions = Arrays.asList(
            ClientImpl.extensionClient,
            FormImpl.extensionForm,
            GroupImpl.extensionGroup,
            InfogeneImpl.extensionInfogene,
            QueryMakerImpl.extensionQueryMaker,
            SendingImpl.extensionSending,
            SessionImpl.extensionSession,
            StatusImpl.extensionStatus,
            UserImpl.extensionUser,
            VradiAdminImpl.extensionVradiAdmin,
            XmlFieldBindingImpl.extensionXmlFieldBinding,
            XmlStreamImpl.extensionXmlStream,
            // not generated
            ModificationTag.EXTENSION_MODIFICATION_TAG,
            Thesaurus.EXTENSION_THESAURUS);
        
        WikittyService wikittyService = wikittyProxy.getWikittyService();
        wikittyService.storeExtension(extensions);
    }
    
    static void configureSubscriberJMS(MultiStorageConfiguration configuration) {
        Properties subscriberProperties = Configuration.getInstance().getJmsSubscriberProperties();
        
        String connectorURL = subscriberProperties.getProperty("connectorURL");
        if (connectorURL != null) {
            Properties brokerProperties = Configuration.getInstance().getJmsPublisherProperties();
            String brokerConnectorURL = brokerProperties.getProperty("connectorURL");
            
            /*
             * skip bad configuration:
             * 
             * if (subscriber url == publisher url) then
             *    return
             * end if
             */
            if (connectorURL.equals(brokerConnectorURL)) {
                return;
            }
            
            Properties properties = new Properties();
            properties.put(WikittyJMSUtil.JNDI_PROVIDER_URL, connectorURL);
            
            try {
                WikittyStorage wikittyStorage = configuration.getCurrentMasterWikittyStorage();
                WikittyStorageSubscriberJMS storageSubscriberJMS =
                    new WikittyStorageSubscriberJMS(wikittyStorage, properties);
                
                WikittyExtensionStorage extensionStorage = configuration.getCurrentMasterExtensionStorage();
                WikittyExtensionSubscriberJMS extensionSubscriberJMS =
                    new WikittyExtensionSubscriberJMS(extensionStorage, properties);
                
                // starts subscribers
                storageSubscriberJMS.start();
                extensionSubscriberJMS.start();
                
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                throw new WikittyException(e);
            }
        }
    }
    
    static void configurePublisherJMS(MultiStorageConfiguration configuration) {
        Properties brokerProperties = Configuration.getInstance().getJmsPublisherProperties();
        if (brokerProperties.isEmpty()) {
            return;
        }
        
        try {
            String connectorURL = brokerProperties.getProperty("connectorURL");
            if (connectorURL == null) {
                return;
            }
            
            // starts broker
            URL jmsConfigURL = Configuration.getInstance().getJmsPublisherURL();
            BrokerFactory.createBroker("wikitty:" + jmsConfigURL.toExternalForm(), true);
            
            // starts publishers
            Properties properties = new Properties();
            properties.put(WikittyJMSUtil.JNDI_PROVIDER_URL, connectorURL);

            WikittyExtensionStorage extensionStorage = new WikittyExtensionStorageJMS(properties);
            WikittyStorage wikittyStorage = new WikittyStorageJMS(properties);
            
            // add publishers to configuration
            configuration.addExtensionStorage(extensionStorage, false);
            configuration.addWikittyStorage(wikittyStorage, false);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new WikittyException(e);
        }
    }
    
    static void configureJDBC(MultiStorageConfiguration configuration) {
        Properties properties = Configuration.getInstance().getJdbcConfigProperties();
        
        WikittyExtensionStorage extensionStorage = new WikittyExtensionStorageJDBC(properties);
        WikittyStorage wikittyStorage = new WikittyStorageJDBC(extensionStorage, properties);
        
        configuration.addExtensionStorage(extensionStorage, true);
        configuration.addWikittyStorage(wikittyStorage, true);
    }

    public static Configuration getServiceConfiguration(){
        return Configuration.getInstance();
    }
    
    public static void main(String[] args) {
        ServiceFactory.getWikittyProxy();
    }
}
